TaskerApp.controller('TaskerController',
    function ($scope, $location, $sce, $compile,
        WorkspaceFileService, TaskService, DeviceService, SignalService,
        TriggerService, ScheduleService, LoggerService) {


        var _this = this;

        _this.DaysOfWeek = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];

        _this.WebVersion = APP.getVersion();
        _this.Version = 'not running';
        _this.IsRunning = false;

        _this.Workspaces = null;
        _this.Workspace = null;
        _this.WorkspaceLastModified = 0;

        _this.knownDevices = [];
        _this.Signals = [];
        _this.Devices = [];

        _this.SaveDialog = null;
        _this.CancelDialog = null;
        _this.ConfigurationExternallyChangedDialog = null;

        _this.log = [];

        _this.Latitude;
        _this.Longitude;


        _this.saveRegKey = function (key, value) {
            JnrWebsocket.registryWrite(key, value);
        };



        _this.showHelp = function (section) {
            var helpUrl = 'help/';
            if (section && '' != section) helpUrl += section + '/';
            var template = '<div ng-include="\'' + helpUrl + '\'"></div>';
            var linkFn = $compile(template);
            var html = linkFn($scope);

            var title = "Help"
            if (section && '' != section) title += ' - ' + section;

            _this.helpDialog = bootbox.dialog({
                className: 'fullscreen bb-primary',
                title: "Help",
                message: html,
                size: 'large',
                onEscape: function () {
                    _this.helpDialog.modal('hide');
                }
            });
            _this.helpDialog.init(function () {
                $compile(_this.helpDialog.contents())($scope);
            });
        };



        _this.getLog = function () {
            var log = _this.log.join('<br>');
            if ("" === log) log = "EMPTY";
            return $sce.trustAsHtml(log);
        };



        _this.showLog = function () {
            var template = '<div ng-bind-html="tasker.getLog()"></div>';
            var linkFn = $compile(template);
            var html = linkFn($scope);

            _this.logDialog = bootbox.dialog({
                className: 'fullscreen bb-primary text-log',
                title: "Log",
                message: html,
                size: 'large',
                onEscape: function () {
                    _this.logDialog.modal('hide');
                }
            });
            _this.logDialog.init(function () {
                $compile(_this.logDialog.contents())($scope);
            });
        };



        /**
         * creates a new workspace
         * 
         * @param {*} $event 
         */
        _this.new = function ($event) {
            if (undefined != $event) {
                // check if there are currently changes being made
                var changesInProgress = _this.Workspace && _this.hasChanged(_this.Workspace.Config, _this.Workspace.SavedConfig);
                if (changesInProgress) {
                    bootbox.dialog({
                        className: 'bb-warning',
                        title: "There are changes in progress",
                        message: "There are currently changes in progress.  You can continue editing and save your changes or discard them now.",
                        buttons: {
                            cancel: {
                                label: 'Continue Editing',
                                className: 'btn-primary'
                            },
                            confirm: {
                                label: 'Discard Changes',
                                className: 'btn-warning',
                                callback: function (result) {
                                    if (result) {
                                        $location.search('workspace', '');
                                        $location.search('view', null);
                                        $location.search('taskname', null);

                                        // '' indicates that a new workspace is desired
                                        _this.Workspace = new WorkspaceFile();
                                        _this.WorkspaceLastModified = 0;
                                        _this.parseConfig(_this.Workspace.Config);
                                    }
                                }
                            }
                        }
                    });
                } else {
                    $location.search('workspace', '');
                    $location.search('view', null);
                    $location.search('taskname', null);

                    // '' indicates that a new workspace is desired
                    _this.Workspace = new WorkspaceFile();
                    _this.WorkspaceLastModified = 0;
                    _this.parseConfig(_this.Workspace.Config);
                }
            }
        };



        /**
         * closes the current workspace
         */
        _this.closeRequest = function () {
            // check if there are currently changes being made
            var changesInProgress = _this.Workspace && _this.hasChanged(_this.Workspace.Config, _this.Workspace.SavedConfig);
            if (changesInProgress) {
                bootbox.dialog({
                    className: 'bb-warning',
                    title: "There are changes in progress",
                    message: "There are currently changes in progress.  You can continue editing and save your changes or discard them now.",
                    buttons: {
                        cancel: {
                            label: 'Continue Editing',
                            className: 'btn-primary'
                        },
                        confirm: {
                            label: 'Discard Changes',
                            className: 'btn-warning',
                            callback: function (result) {
                                if (result) {

                                    $location.search('workspace', undefined);
                                    $location.search('view', undefined);
                                    $location.search('taskname', undefined);
                                    _this.Workspace = undefined;

                                    // only call scope.apply if needed
                                    if (!$scope.$$phase) $scope.$apply();
                                }
                            }
                        }
                    }
                }).init(function () {
                    $(".modal-header").css("background-color", "#f0ad4c");
                    $(".modal-header").css("color", "#fff");
                });

            } else {
                $location.search('workspace', undefined);
                $location.search('view', undefined);
                $location.search('taskname', undefined);
                _this.Workspace = undefined;

                // only call scope.apply if needed
                if (!$scope.$$phase) $scope.$apply();
            }
        };



        /**
         * opens a worksapce by name.  this function sets the workspace name in the
         * URL.  the $watch method sees the update and performs opening and reading 
         * of the configuration
         * 
         * @param {*} workspace 
         */
        _this.openRequest = function (workspace) {
            // check if there are currently changes being made
            var changesInProgress = _this.Workspace && _this.hasChanged(_this.Workspace.Config, _this.Workspace.SavedConfig);
            if (changesInProgress) {
                bootbox.dialog({
                    className: "bb-warning",
                    title: "There are changes in progress",
                    message: "There are currently changes in progress.  You can continue editing and save your changes or discard them now.",
                    buttons: {
                        cancel: {
                            label: 'Continue Editing',
                            className: 'btn-primary'
                        },
                        confirm: {
                            label: 'Discard Changes',
                            className: 'btn-warning',
                            callback: function () {
                                $location.search('workspace', workspace.Name);
                            }
                        }
                    }
                }).init(function () {
                });

            } else {
                $location.search('workspace', workspace.Name);
            }
        };



        /**
         * prompts the user to clone the workspace
         * 
         * @param {*} workspace 
         */
        _this.cloneRequest = function (workspace) {
            _this.EditWorkspaceNameDialog = bootbox.prompt({
                className: 'bb-success',
                title: "Please enter the new name for the '" + workspace.Name + "' workspace",
                message: "<p>Any changes that are in progress will not be saved at this time.</p>",
                buttons: {
                    cancel: {
                        label: 'Cancel',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Save',
                        className: 'btn-success'
                    }
                },
                value: workspace.Name.replaceAll('.json', ''),
                callback: function (result) {
                    if (result) {
                        if (!result.endsWith('.json')) {
                            result += '.json';
                        }

                        // we must make sure that the name has changed.
                        var workspaceFile = _this.Workspaces.find((element) => {
                            return element.Name == result
                        });

                        if (workspaceFile) {
                            bootbox.alert({
                                className: 'bb-warning',
                                title: 'Workspace Clone Failed!',
                                message: 'Must specify a name that does not already exist'
                            });
                            return false;
                        } else {
                            // save the workspace as a new file on the jnior
                            workspace.saveas(result, function () {
                                $scope.$apply();

                                _this.getWorkspaceFiles();
                            });
                        }
                    }
                }
            });
            _this.EditWorkspaceNameDialog.init(function () {
                $("#bootbox-confirm-btn").prop('disabled', false);
                $("#bootbox-cancel-btn").prop('disabled', false);
                validate_alphanumeric($(".bootbox-input"));
                confirm_or_submit_on_enter($(".bootbox"));
            });
        };



        /**
         * prompts the user to rename the workspace
         * 
         * @param {*} workspace 
         */
        _this.renameRequest = function (workspace) {
            _this.EditWorkspaceNameDialog = bootbox.prompt({
                className: 'bb-success',
                title: "Please enter the new name for the '" + workspace.Name + "' workspace",
                message: "<p>Any changes that are in progress will not be saved at this time.</p>",
                buttons: {
                    cancel: {
                        label: 'Cancel',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Save',
                        className: 'btn-success'
                    }
                },
                value: workspace.Name.replaceAll('.json', ''),
                callback: function (result) {
                    if (result) {
                        if (!result.endsWith('.json')) {
                            result += '.json';
                        }

                        workspace.rename(result);

                        $scope.$apply();

                        _this.getWorkspaceFiles();
                    }
                }
            });
            _this.EditWorkspaceNameDialog.init(function () {
                $("#bootbox-confirm-btn").prop('disabled', false);
                $("#bootbox-cancel-btn").prop('disabled', false);
                validate_alphanumeric($(".bootbox-input"));
                confirm_or_submit_on_enter($(".bootbox"));
            });
        };



        /**
         * handles a user request to delete workspace
         * 
         * @param {*} workspaceName 
         */
        _this.deleteRequest = function (workspace) {
            _this.DeleteDialog = bootbox.confirm({
                className: 'bb-primary',
                title: "Are you sure you want to delete the " + workspace.Name + " workspace?",
                message: '<p>Are you sure you want to delete the <b>' + workspace.Name + '</b> workspace?</p>' +
                    '<p>Deleting the workspace will make a backup copy of the workspace and ' +
                    'delete the current version.  This will prevent the workspace logic from ' +
                    'running and allow you to restore if you change your mind later.<p>',
                buttons: {
                    confirm: {
                        label: 'Delete the Workspace',
                        className: 'btn-danger'
                    },
                    cancel: {
                        label: 'Nevermind',
                        className: 'btn-primary'
                    }
                },
                callback: function (result) {
                    if (result) {

                        workspace.delete(function () {
                            JnrWebsocket.postMessage(2010, { Message: 'config.updated' });

                            _this.getWorkspaceFiles();

                            bootbox.alert({
                                className: 'bb-primary',
                                title: 'Workspace Removed!',
                                message: '<p><b>' + workspace.Name + '</b> has been deleted!</p>' +
                                    '<p>But dont worry.  A <b>backup</b> has been created for you to ' +
                                    '<b>restore</b> from later!</p>'
                            });

                            // if this workspace is open then we will close it
                            if (_this.Workspace == workspace) _this.closeRequest($event);
                        });
                    }
                }
            });
        };



        /**
         * sets whether or not the given workspace is enabled
         * 
         * @param {*} workspace 
         */
        _this.setEnabled = function (workspace) {
            if (workspace.isEnabled()) _this.disableWorkspace(workspace);
            else _this.enableWorkspace(workspace);
        };



        _this.enableWorkspace = function (workspace, callback) {
            var newName = workspace.Name.replaceAll('#', '');
            workspace.rename(newName, function () {
                _this.getWorkspaceFiles(function () {
                    if (callback) callback();
                });
            });
        };



        _this.disableWorkspace = function (workspace, callback) {
            var newName = '#' + workspace.Name;
            workspace.rename(newName, function () {
                _this.getWorkspaceFiles(function () {
                    if (callback) callback();
                });
            });
        };



        /**
         * allows the user to edit the notes for the workspace.
         * 
         * @param {*} workspace 
         * @param {*} saveOnFinish
         */
        _this.editNotes = function (workspace) {
            bootbox.prompt({
                className: 'bb-primary',
                title: "Edit the Workspace Notes",
                inputType: 'textarea',
                value: workspace.Config.Notes,
                callback: function (result) {
                    // trim off any whitespace
                    result = result.replace(/^[ \n]*|[ \n]*$/g, "");

                    workspace.Config.Notes = result;

                    if (_this.Workspace != workspace) {
                        workspace.save();
                    }

                    $scope.$apply();
                }
            }).init(function () {
                confirm_or_submit_on_enter($(".bootbox"));
            });
        };



        _this.parseConfig = function (configJson) {
            try {
                // we are about to load.  invalidate our saved workspace modification time
                console.log('resetting the saved last modified time for ' + _this.Workspace);
                _this.WorkspaceLastModified = 0;

                if (_this.Workspace.Config) {
                    _this.Workspace.Config.TaskList = TaskService.load(configJson.TaskList);
                    _this.Workspace.Config.DeviceList = DeviceService.load(configJson.DeviceList);
                    _this.Workspace.Config.LoggerList = LoggerService.load(configJson.LoggerList);
                    _this.Workspace.Config.SignalList = SignalService.load(configJson.SignalList);
                    _this.Workspace.Config.TriggerList = TriggerService.load(configJson.TriggerList);
                    _this.Workspace.Config.ScheduleList = ScheduleService.load(configJson.ScheduleList);
                }

                // broadcast to any listeners that the workspace was loaded locally
                $scope.$broadcast('workspace.loaded', {});

            } catch (err) {
                console.error(err);
            }

            setTimeout(function () {
                // validate all tabs
                TaskService.validate();
                DeviceService.validate();
                LoggerService.validate();
                SignalService.validate();
                TriggerService.validate();
                ScheduleService.validate();

                // // only call scope.apply if needed
                // if (!$scope.$$phase) {
                $scope.$apply();
                // }
            }, 100);


            setTimeout(function () {
                onresize();
                $('[data-toggle="tooltip"]').tooltip();
            }, 100);
        };


        _this.View = null;

        // watch for changes to our $location.url() object.
        $scope.$watch(function () {
            return $location.url();
        }, function (url) {
            if (url) {
                var view = $location.search().view;
                if (view) {
                    _this.setView(view);
                } else {
                    _this.setView(null);
                }

                var workspaceName = $location.search().workspace;

                if (workspaceName && (!_this.Workspace || _this.Workspace.Name != workspaceName)) {

                    // before requesting the new config lets unload the current config
                    _this.new();

                    function loadConfigWhenLoggedIn() {
                        console.log(new Date() + ' loadConfigWhenLoggedIn()');
                        console.log('JnrWebsocket.isLoggedIn(): ' + JnrWebsocket.isLoggedIn());
                        if (JnrWebsocket.isLoggedIn() && _this.Workspaces) {
                            // now find our workspacefile object
                            _this.Workspace = _this.Workspaces.find((element) => {
                                return element.Name == workspaceName
                            });

                            _this.load(); //_this.configReceived);
                        } else {
                            setTimeout(function () {
                                loadConfigWhenLoggedIn();
                            }, 250);
                        }
                    }
                    loadConfigWhenLoggedIn();
                }
            } else _this.closeRequest();
        });



        $scope.$on('workspace.uploaded', function (event, data) {
            // get the workspace files.  when they are retrieved the inner function 
            // will be called.  we then use the filename from the data to set the 
            // workspace to the one we just uploaded
            _this.getWorkspaceFiles(function () {
                $location.search('workspace', data.filename);
            });
        });



        _this.CurrentService = TaskService;

        // push the view selection to our $location.url() object.
        _this.setView = function (view) {
            if (_this.View == view) return;

            // check to see if the current service is valid before switching tabs
            var valid = true;
            if (_this.CurrentService && null != _this.CurrentService) {
                valid = _this.CurrentService.validate();
            }

            // we no longer want to prevent users from switching tabs.
            // if (!valid && _this.CurrentService == TaskService) return;

            _this.View = view;
            if (null != view) {
                if (null === _this.View || "tasks" === _this.View) {
                    _this.CurrentService = TaskService;
                } else if ("device-profiles" === _this.View) {
                    _this.CurrentService = DeviceService;
                } else if ("signal-profiles" === _this.View) {
                    _this.CurrentService = SignalService;
                } else if ("trigger-profiles" === _this.View) {
                    _this.CurrentService = TriggerService;
                } else if ("schedule" === _this.View) {
                    _this.CurrentService = ScheduleService;
                } else if ("logging-profiles" === _this.View) {
                    _this.CurrentService = LoggerService;
                }

                $location.search('view', view);
            }
        };



        _this.activateView = function (ele) {
            $compile(ele.contents())($scope);
            $scope.$apply();
        };



        _this.getWorkspaceFiles = function (callback) {
            JnrWebsocket.getFileListing('/flash/tasker/workspaces')
                .then(function (response) {

                    // we have a list of files from the workspaces directory.  we 
                    // want to gather only the .json files.  to do that we will 
                    // filter them out of the array that was returned.
                    response.Content = response.Content.filter(function (element) {
                        return element.Name.endsWith('.json');
                    });

                    // now lets sort the remaining array.  the array, if just sorted
                    // alphabetically would put all the # (disabled) workspaces first. 
                    // we dont want that.  when performing the comparison we will
                    // remove the # at the start of the filename.
                    response.Content.sort(function compareFn(firstEl, secondEl) {
                        var nameA = firstEl.Name.replaceAll('#', '').toUpperCase();
                        var nameB = secondEl.Name.replaceAll('#', '').toUpperCase();
                        if (nameA < nameB) return -1;
                        if (nameA > nameB) return 1;
                        return 0;
                    });

                    // now create a Workspaces array that will hold workspace files and
                    // go through each file and create the workspacefile object and add
                    // it to the array.
                    var workspaces = [];
                    response.Content.forEach(function (fileEntry) {
                        var workspaceFile = null;

                        if (_this.Workspaces) {
                            // see if the filename is already a workspacefile object
                            var workspaceFile = _this.Workspaces.find(function (element) {
                                return element.Name == fileEntry.Name;
                            });
                            console.log(fileEntry.Name + ' is found ' + workspaceFile);
                        }

                        // if the filename wasnt found in the workspacefile object array 
                        // then we will create a new workspacefile and add it to the array
                        if (!workspaceFile) {
                            workspaceFile = new WorkspaceFile(JnrWebsocket, fileEntry);
                            // perform an initial open to get the version and the notes
                            // from the configuration
                            workspaceFile.open();
                        }
                        +        workspaces.push(workspaceFile);
                    });
                    // assign our Workspaces array with our new array
                    _this.Workspaces = workspaces;

                    $scope.$apply();

                    _this.Workspaces.forEach(function (workspace) {
                        if (workspace.Name === _this.Workspace) {
                            _this.checkForExternalWorkspaceModifycation(workspace);
                        }
                    });

                    if (callback) callback();
                });
        };



        _this.JniorTimeOffset = -1;
        setInterval(function () {
            if (-1 != _this.JniorTimeOffset) {
                var timestamp = new Date().getTime() + _this.JniorTimeOffset;
                _this.JniorTime = moment(timestamp).format('MMMM Do YYYY, h:mm:ss a');
                $scope.$apply();
            } else {
                _this.JniorTime = '';
            }
        }, 750);



        JnrWebsocket.addOnLoggedInListener(function (json) {
            var jniorCurrentTimestamp = json.Timestamp;
            _this.JniorTimeOffset = jniorCurrentTimestamp - new Date().getTime();

            // get the workspace files on load and periodically
            _this.getWorkspaceFiles();
            setInterval(_this.getWorkspaceFiles, 30000);

            JnrWebsocket.readRegistryKeys(['ipconfig/hostname', '$LastNtpSuccess', 'device/latitude', 'device/longitude'], function (key, value) {
                if ('$LastNtpSuccess' == key) _this.LastNtpSuccess = value;
                else if ('ipconfig/hostname' == key) {
                    _this.Hostname = value;
                    document.title = "Tasker - " + _this.Hostname;
                }
                else if ('device/latitude' == key) _this.Latitude = value;
                else if ('device/longitude' == key) _this.Longitude = value;
                $scope.$apply();
            });
        });




        _this.workspaceFileUploadRequest = function () {
            // check if there are currently changes being made
            var changesInProgress = _this.Workspace && _this.hasChanged(_this.Workspace.Config, _this.Workspace.SavedConfig);
            if (changesInProgress) {
                _this.WorkspaceFileUploadDialog = bootbox.dialog({
                    className: 'bb-primary',
                    title: "Changes have been made to Unsaved Workspace",
                    message: "Changes have been made to the current worksapce.  Proceeding to " +
                        "upload a new workspace will close the current workspace and open the " +
                        "new workspace.  Changes would be lost.  How do you wish to proceed?",
                    buttons: {
                        cancel: {
                            label: 'Cancel',
                            className: 'btn-default'
                        },
                        saveAndContinue: {
                            label: 'Continue Anyway',
                            className: 'btn-warning',
                            callback: function () {
                                _this.workspaceFileUpload();
                            }
                        },
                        continue: {
                            label: 'Save and Continue',
                            className: 'btn-success',
                            callback: function () {
                                _this.save(function () {
                                    _this.workspaceFileUpload();
                                });
                            }
                        }
                    },
                });

            } else {
                _this.workspaceFileUpload();
            }
        };



        _this.workspaceFileUpload = function () {
            var $uploadFile = $('#workspace-file-upload');
            var fileCtrl = $uploadFile[0];
            fileCtrl.onchange = function (evt) {
                var file = evt.target.files[0];
                WorkspaceFileService.uploadFile(file);
                $('#workspace-file-upload').val('');
            };
            fileCtrl.click();
        };



        var lastAnnouncementTime = new Date().getTime() - 10000;
        JnrWebsocket.addOnMessageListener(function (json) {
            console.log(JSON.stringify(json, null, 2));

            if ('status.update' == json.Message) {
                var title = "";
                if (undefined != status.Title) title += ' : ' + status.Title;
                toastr.success(json.StatusMessage, new Date() + ' ' + title);

                // now push the message to our log object so we can recall it later
                _this.log.push(moment().format('MM/DD/YY h:mm:ss a') + ', ' + title + ' - ' + json.StatusMessage);
                $scope.$apply();
            }

            else if ('workspace.loaded' == json.Message) {
                var workspaceName = json.WorkspaceName;
                var loadTimeInSeconds = json.LoadTime;
                toastr.success('<b>' + workspaceName + '</b> was Loaded in <b>' + loadTimeInSeconds.toFixed(2) + '</b> seconds');

                // only enable the TasksLoaded if the workspace that Tasker loaded in the one that 
                // the user currently has loaded
                if (workspaceName == _this.Workspace.Name) TaskService.TasksLoaded = true;

                // now push the message to our log object so we can recall it later
                _this.log.push(moment().format('MM/DD/YY h:mm:ss a') + ', ' + workspaceName + ' was Loaded in ' + loadTimeInSeconds.toFixed(2) + ' seconds');
                $scope.$apply();

            }

            else if ('error' == json.Message) {
                // var message = json.StackTrace ? json.StackTrace.replaceAll('\r\n', '<br>') : '';
                var message = "Please check the exception log for more details.";
                toastr.error(message, json.ErrorMessage, { closeButton: true, timeOut: 10000, extendedTimeOut: 0 });

                // now push the json object to our log object so we can recall it later
                var message = json.StackTrace ? json.StackTrace.replaceAll('\r\n', '<br>') : '';
                _this.log.push(moment().format('MM/DD/YY h:mm:ss a') + ' : ' + json.ErrorMessage + ' - ' + message);
                $scope.$apply();

            }

            else if ('announce' == json.Message) {
                var now = new Date().getTime();
                // var timeSinceLastAnnouncementRequest = now - lastAnnouncementRequestTime;
                // console.log('timeSinceLastAnnouncementRequest: ' + timeSinceLastAnnouncementRequest);
                var timeSinceLastAnnouncement = now - lastAnnouncementTime;
                console.log('timeSinceLastAnnouncement: ' + timeSinceLastAnnouncement);
                lastAnnouncementTime = now;

                if (null != taskerNotRunningToast) {
                    toastr.clear(taskerNotRunningToast);
                    taskerNotRunningToast = null;
                }

                _this.IsRunning = true;
                _this.Version = json.Version;
                $scope.$apply();

            }

            else if (json.Message === "File List Response") {
                if (json.Folder.endsWith(_this.Workspace)) {
                    _this.checkForExternalWorkspaceModifycation(json.Content[0]);
                }
            }

            else if ('death' == json.Message) {
                // this message is received when tasker exits gracefully.  for 
                // instance, when rebooting
                taskerNotRunningToast = toastr.error("Unable to communicate with the Tasker application", "Tasker is not running",
                    { closeButton: false, timeOut: 0, extendedTimeOut: 0 });
            }

            else if ('user.alert' == json.Message) {
                toastr.success('<b>User Alert: </b>' + json.UserAlertMessage);
            }

        });
        ``


        _this.checkForExternalWorkspaceModifycation = function (workspace) {
            if (0 == _this.WorkspaceLastModified) {
                console.log('Set the last modified time for ' + _this.Workspace + ' to ' + workspace.Mod);
                _this.WorkspaceLastModified = workspace.Mod;
            } else {
                console.log('saved last modified time for ' + _this.Workspace + ' is ' + _this.WorkspaceLastModified + ' and is now ' + workspace.Mod);
                if (_this.WorkspaceLastModified !== workspace.Mod) {
                    if (!_this.ConfigurationExternallyChangedDialog) {
                        _this.ConfigurationExternallyChangedDialog = bootbox.alert({
                            className: 'bb-danger',
                            title: "Configuration externally changed!",
                            message: "Some other process has changed the <b>'" + _this.Workspace + "'</b> configuration since the last time it was " +
                                "loaded.  You could relead the page or your changes will overwrite the configuration.",
                            buttons: {
                                ok: {
                                    label: 'OK',
                                    className: 'btn-danger'
                                }
                            }
                        });

                        _this.WorkspaceLastModified = workspace.Mod;
                    }
                }
            }
        };



        JnrWebsocket.connect();
        JnrWebsocket.postMessage(2010, { Message: 'announce.request' });



        var taskerNotRunningToast = null;
        var lastAnnouncementRequestTime = 0;
        setInterval(function () {
            var now = new Date().getTime();
            // console.log(new Date());

            // we want to make sure the Task.jar application is ALIVE.  to do this we will 
            // periodically ping the application and expect a response in a reasonable 
            // amount of time.  We will first try pinging every 10 seconds and expect a 
            // response within 5 seconds.  These values may change.
            var timeSinceLastAnnouncementRequest = now - lastAnnouncementRequestTime;
            if (10000 < timeSinceLastAnnouncementRequest) {
                lastAnnouncementRequestTime = new Date().getTime();
                JnrWebsocket.postMessage(2010, { Message: 'announce.request' });
            }

            // see how long it has been since we give the last announcemenet message from 
            // the Task.jar java application.  We know that it is sent every 10 seconds.  
            // So expect that it has been received within the past 15 seconds.
            var timeSinceLastAnnouncementResponse = now - lastAnnouncementTime;
            // console.log('timeSinceLastAnnouncementResponse: ' + timeSinceLastAnnouncementResponse);
            if (15000 < timeSinceLastAnnouncementResponse) {
                if (null == taskerNotRunningToast) {
                    TaskService.TasksLoaded = false;
                    _this.IsRunning = false;
                    $scope.$apply();

                    taskerNotRunningToast = toastr.error("Unable to communicate with the Tasker application", "Tasker is not running",
                        { closeButton: false, timeOut: 0, extendedTimeOut: 0 });
                }
            }
        }, 1000);



        _this.load = function (callback) {
            if (!_this.Workspace) return;

            if (undefined != _this.ConfigurationLoadingDialog) {
                var workspaceNameElem = _this.ConfigurationLoadingDialog.find('#workspace-name');
                if (0 < workspaceNameElem.length) {
                    workspaceNameElem[0].forEach(function (element) {
                        element.innerText = _this.Workspace;
                    });
                }
            }
            else {
                _this.ConfigurationLoadingDialog = bootbox.dialog({
                    className: 'bb-success',
                    title: 'Loading <span id="workspace-name">' + _this.Workspace.Name + '</span>',
                    message: '<p><b>Loading \'<span id="workspace-name">' + _this.Workspace.Name + '</span>\'</b></p>' +
                        '<p>This may take a few seconds.  This dialog will disappear when the configuration has been loaded.</p>',
                    closeButton: false
                });
                _this.ConfigurationLoadingDialog.css({
                    'top': '50%',
                    'margin-top': '-95px'
                });
            }

            setTimeout(function () {
                _this.ConfigurationLoadingDialogLoadTime = new Date();

                _this.Workspace.workspaceOpened = function (workspaceFile) {
                    _this.parseConfig(workspaceFile.Config);
                    if (callback) callback();

                    function closeLoadingDialog() {
                        var timeRemaining = new Date() - _this.ConfigurationLoadingDialogLoadTime;
                        timeRemaining = 1500 - timeRemaining;
                        setTimeout(function () {
                            if (0 >= timeRemaining) {
                                if (_this.ConfigurationLoadingDialog) {
                                    _this.ConfigurationLoadingDialog.modal('hide');
                                    _this.ConfigurationLoadingDialog = undefined;
                                }
                            } else closeLoadingDialog();
                        }, timeRemaining);
                    }
                    closeLoadingDialog();
                }
                _this.Workspace.open();

                // JnrWebsocket.readFile('/flash/tasker/workspaces/' + _this.Workspace, callback);
            }, 100);
        };



        _this.getHtmlSafe = function (content) {
            if (content) {
                var safeHtml = content.replaceAll('\n', '<br>');
                return $sce.trustAsHtml(safeHtml);
            }
        };



        _this.getHtmlSafeActions = function (signal) {
            if (signal.Actions) {
                var safeHtml = signal.Actions.replaceAll('\n', '<br>');
                return $sce.trustAsHtml(safeHtml);
            }
        };



        _this.hasChanged = function (config, savedConfig) {
            var hasChanged = !angular.equals(config, savedConfig);
            if (hasChanged) {
                addEventListener("beforeunload", beforeUnloadListener, { capture: true });
            } else {
                removeEventListener("beforeunload", beforeUnloadListener, { capture: true });
            }
            return hasChanged;
        };



        _this.schemaUpdated = function (file) {
            var fileSchemaArray = file.Schema.split(',');
            file.Schema = fileSchemaArray;
        };



        _this.saveChangesRequest = function () {
            // lets validate the form before saving
            TaskService.validate();
            DeviceService.validate();
            LoggerService.validate();
            SignalService.validate();
            TriggerService.validate();
            ScheduleService.validate();
            var valid = TaskService.isValid && DeviceService.isValid && LoggerService.isValid
                && SignalService.isValid && TriggerService.isValid && ScheduleService.isValid;
            // if (_this.CurrentService && null != _this.CurrentService) {
            //     valid = _this.CurrentService.validate();
            // }

            if (valid) {
                _this.SaveDialog = bootbox.confirm({
                    className: 'bb-success',
                    title: "Are you sure you want to save your changes?",
                    message: "Your changes will take affect without needing to reboot.",
                    buttons: {
                        cancel: {
                            label: 'Cancel',
                            className: 'btn-default'
                        },
                        confirm: {
                            label: 'Save',
                            className: 'btn-success'
                        }
                    },
                    callback: function (result) {
                        if (result) {
                            TaskService.TasksLoaded = false;
                            $scope.$apply();
                            _this.saveChanges();
                            return false;
                        }
                    }
                });
                _this.SaveDialog.init(function () {
                    $("#bootbox-confirm-btn").prop('disabled', false);
                    $("#bootbox-cancel-btn").prop('disabled', false);
                    confirm_or_submit_on_enter($(".bootbox"));
                });

            } else {
                bootbox.alert({
                    className: 'bb-danger',
                    title: 'Validation Error',
                    message: 'There was an error during validation.  Please check the tabs with an <i class="fa fa-exclamation-triangle text-danger" aria-hidden="true"></i> symbol and fix all fields in red and try again'
                });
            }

        };



        _this.saveAsChangesRequest = function () {
            // lets validate the form before saving
            TaskService.validate();
            DeviceService.validate();
            LoggerService.validate();
            SignalService.validate();
            TriggerService.validate();
            ScheduleService.validate();
            var valid = TaskService.isValid && DeviceService.isValid && LoggerService.isValid
                && SignalService.isValid && TriggerService.isValid && ScheduleService.isValid;
            // var valid = true;
            // if (_this.CurrentService && null != _this.CurrentService) {
            //     valid = _this.CurrentService.validate();
            // }

            if (valid) {
                _this.SaveDialog = bootbox.prompt({
                    className: 'bb-success',
                    title: "Please enter the new name for the workspace",
                    // message: "Your changes will take affect without needing to reboot.",
                    buttons: {
                        cancel: {
                            label: 'Cancel',
                            className: 'btn-default'
                        },
                        confirm: {
                            label: 'Save',
                            className: 'btn-success'
                        }
                    },
                    value: _this.Workspace.Name.replaceAll('.json', ''),
                    callback: function (result) {
                        if (result) {
                            if (!result.endsWith('.json')) {
                                result += '.json';
                            }

                            _this.Workspace.setName(result);
                            _this.saveChanges();

                            // update the url to contain the new workspace name
                            $location.search('workspace', result);

                            return false;
                        }
                    }
                });
                _this.SaveDialog.init(function () {
                    $("#bootbox-confirm-btn").prop('disabled', false);
                    $("#bootbox-cancel-btn").prop('disabled', false);
                    validate_alphanumeric($(".bootbox-input"));
                    confirm_or_submit_on_enter($(".bootbox"));
                });

            } else {
                bootbox.alert({
                    className: 'bb-danger',
                    title: 'Validation Error',
                    message: 'There was an error during validation.  Please check the tabs with an <i class="fa fa-exclamation-triangle text-danger" aria-hidden="true"></i> symbol and fix all fields in red and try again'
                });
            }

        };



        _this.saveComplete = function () {
            if (null !== _this.SaveDialog) {
                _this.SaveDialog.modal('hide');
                _this.SaveDialog = null;
            }

            // make sure to get the updated workspace files with this new filename
            _this.getWorkspaceFiles();

            TaskService.TasksLoaded = true;
            $scope.$apply();

            var fullWorkspaceFileName = "/flash/tasker/workspaces/" + _this.Workspace.Name;
            JnrWebsocket.postMessage(2010, { Message: 'config.updated', Workspace: fullWorkspaceFileName });
        };



        _this.saveChanges = function () {
            if (null != _this.Workspace && '' != _this.Workspace) {
                $("#bootbox-confirm-btn").html("Saving...");
                $("#bootbox-confirm-btn").prop('disabled', true);
                $("#bootbox-cancel-btn").prop('disabled', true);

                // we are about to save.  invalidate our saved workspace modification time
                console.log('resetting the saved last modified time for ' + _this.Workspace);
                _this.WorkspaceLastModified = 0;

                // perform the save
                _this.save(_this.saveComplete);

            } else {
                _this.saveAsChangesRequest();
            }
        };



        _this.save = function (saveCompleteCallback) {
            // write the saved config to a backup file
            var workspaceFileName = "/flash/tasker/workspaces/" + _this.Workspace.Name;

            var bakFileContent = _this.Workspace.getJson(_this.Workspace.SavedConfig);
            // call the writeFile method for the bak copy and only when the file has been 
            //  successfully written will we issue the write file for the save of the 
            //  current changes
            JnrWebsocket.writeFile(workspaceFileName + '.bak', bakFileContent, function () {
                var fileContent = _this.Workspace.getJson();
                _this.Workspace.save(saveCompleteCallback);
                // JnrWebsocket.writeFile(workspaceFileName, fileContent, saveCompleteCallback);
            });
        };



        _this.cancelChangesRequest = function () {
            _this.CancelDialog = bootbox.dialog({
                className: 'bb-warning',
                title: "There are changes in progress",
                message: "There are currently changes in progress.  You can continue editing and save your changes or discard them now.",
                buttons: {
                    cancel: {
                        label: 'Continue Editing',
                        className: 'btn-primary'
                    },
                    confirm: {
                        label: 'Discard Changes',
                        className: 'btn-warning',
                        callback: function (result) {
                            if (result) {
                                _this.cancelChanges();
                                return false;
                            }
                        }
                    }
                }
            });
            _this.CancelDialog.init(function () {
                $("#bootbox-confirm-btn").prop('disabled', false);
                $("#bootbox-cancel-btn").prop('disabled', false);
                confirm_or_submit_on_enter($(".bootbox"));
            });
        };



        _this.cancelComplete = function () {
            function closeCancelDialog() {
                var timeRemaining = new Date() - _this.ConfigurationLoadingDialogLoadTime;
                timeRemaining = 1500 - timeRemaining;
                setTimeout(function () {
                    if (0 >= timeRemaining) {
                        if (_this.CancelDialog) {
                            _this.CancelDialog.modal('hide');
                            _this.CancelDialog = undefined;
                        }
                    } else closeCancelDialog();
                }, timeRemaining);
            }
            closeCancelDialog();

            // if (null !== _this.CancelDialog) {
            //     _this.CancelDialog.modal('hide');
            //     _this.CancelDialog = null;
            // }

            $scope.$apply();
        };



        _this.cancelChanges = function () {
            $("#bootbox-confirm-btn").html("Reverting...");
            $("#bootbox-confirm-btn").prop('disabled', true);
            $("#bootbox-cancel-btn").prop('disabled', true);

            _this.load(_this.cancelComplete);
        };



        _this.editWorkspaceName = function () {
            _this.EditWorkspaceNameDialog = bootbox.prompt({
                className: 'bb-success',
                title: "Please enter the new name for the workspace",
                message: "<p>Any changes that are in progress will not be saved at this time.</p>",
                buttons: {
                    cancel: {
                        label: 'Cancel',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Save',
                        className: 'btn-success'
                    }
                },
                value: _this.Workspace.Name.replaceAll('.json', ''),
                callback: function (result) {
                    if (result) {
                        if (!result.endsWith('.json')) {
                            result += '.json';
                        }

                        // rename the workspace on the jnior
                        var oldWorkspaceFileName = "/flash/tasker/workspaces/" + _this.Workspace;
                        var newWorkspaceFileName = "/flash/tasker/workspaces/" + result;
                        JnrWebsocket.renameFile(oldWorkspaceFileName, newWorkspaceFileName);

                        // update the url to contain the new workspace name
                        $location.search('workspace', result);

                        _this.getWorkspaceFiles();
                    }
                }
            });
            _this.EditWorkspaceNameDialog.init(function () {
                $("#bootbox-confirm-btn").prop('disabled', false);
                $("#bootbox-cancel-btn").prop('disabled', false);
                validate_alphanumeric($(".bootbox-input"));
                confirm_or_submit_on_enter($(".bootbox"));
            });
        };



        _this.isWorkspaceEnabled = function () {
            return _this.Workspace && _this.Workspace.isEnabled();
        };



        _this.disableCurrentWorkspace = function () {
            _this.disableWorkspace(_this.Workspace, function () {
                // update the url to contain the new workspace name
                $location.search('workspace', _this.Workspace.Name);

                onresize();
            });
        };



        _this.enableCurrentWorkspace = function () {
            _this.enableWorkspace(_this.Workspace, function () {
                // update the url to contain the new workspace name
                $location.search('workspace', _this.Workspace.Name);

                onresize();
            });
        };



        _this.restoreRequest = function () {
            _this.RestoreDialog = bootbox.confirm({
                className: 'bb-primary',
                title: "Are you sure you want to restore the backup version of the " + _this.Workspace + " workspace?",
                // message: '<p>This dialog will present all of the bakup files.  It will show you the ' +
                //     'last modified time and if there is a newer version of the file in progress.  ' +
                //     'You can select a file to have it restored to a non backed up version</p>' +
                //     '<table class="table"><tr><th></th><th>File Name</th><th>Last Modified</th></tr>' +
                //     '<tr><td><input type="checkbox"></td><td>File 1</td><td>Apr 16 2019 16:08</td></tr></table>' +
                //     '<tr><td><input type="checkbox"></td><td>File 2</td><td>Apr 16 2019 16:08</td></tr></table>' +
                //     '<tr><td><input type="checkbox"></td><td>File 3</td><td>Apr 16 2019 16:08</td></tr></table>' +
                //     '<p class="alert alert-danger">This feature is not yet implemented</p>',
                message: '<p>Are you sure you want to restore the backup version of the <b>' + _this.Workspace + '</b> workspace?</p>' +
                    '<p>The previous version of this file was last modified on <b>Apr 16 2019 16:08</b></p>' +
                    '<p class="alert alert-danger">This feature is not yet implemented</p>',
                buttons: {
                    cancel: {
                        label: 'Nevermind',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Restore the Workspace',
                        className: 'btn-primary'
                    }
                },
                callback: function (result) {
                    if (result) {
                        // return false;
                    }
                }
            });
            _this.RestoreDialog.init(function () {
                $("#bootbox-confirm-btn").prop('disabled', true);
                $("#bootbox-cancel-btn").prop('disabled', false);
            });
        };



        _this.showPreferences = function () {
            var template = '<div ng-include="\'preferences.html\'"></div>';
            var linkFn = $compile(template);
            var html = linkFn($scope);

            bootbox.dialog({
                className: 'bb-primary',
                title: "Preferences",
                message: html,
                buttons: {
                    cancel: {
                        label: 'Cancel',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Set',
                        className: 'btn-success',
                        callback: function (result) {
                            if (result) {
                                var form = $("#preferences-form");
                                form.validate({ errorPlacement: function (error, element) { } });

                                if (form.valid()) {
                                    // if (undefined == schedule.Params) schedule.Params = {};
                                    // if (undefined == schedule.Params.Rules) schedule.Params.Rules = [];
                                    // schedule.Params.Rules.push(_this.SelectedRule);

                                    // $scope.$apply();
                                } // else return false;
                            }
                        }
                    }
                }
            });
        };



        ///////////////////////////////////////////////////////



        _this.fileNameChange = function (filename) {
            if (filename) {
                alert(filename);
            }
        };


        var lastFocused;
        _this.setLastFocused = function ($event) {
            if (!$event) {
                lastFocused = null;
            } else if (lastFocused !== $event.currentTarget) {
                lastFocused = $event.currentTarget;
            }
        };

        _this.shouldInsertType = function () {
            return lastFocused && (null !== lastFocused);
        };

        _this.isModulePresent = function (signal) {
            var moduleId = null;
            for (var deviceIndex in _this.Devices) {
                if (_this.Devices[deviceIndex].Type === signal.Type) {
                    moduleId = _this.Devices[deviceIndex].Id;
                }
            }

            var modulePresent = false;
            if (null !== moduleId) {
                modulePresent = _this.knownDevices[moduleId];
            }

            return modulePresent;
        };

        _this.insertText = function (text) {
            insertText(lastFocused, text);
        };



        _this.getDownloadUri = function (workspaceName) {
            if (null != workspaceName) {
                var htmlSafeWorkspaceName = workspaceName.replaceAll('#', '%23');
                var uri = 'download.php?filename=/flash/tasker/workspaces/'
                    + htmlSafeWorkspaceName + '&download=' + htmlSafeWorkspaceName;
                return uri;
            }
            return '';
        };



        const beforeUnloadListener = (event) => {
            event.preventDefault();
            return event.returnValue = "Are you sure you want to exit?";
        };



        _this.isTasksTabValid = function () {
            return TaskService.isValid;
        };



        _this.isDevicesTabValid = function () {
            return DeviceService.isValid;
        };



        _this.isLoggersTabValid = function () {
            return LoggerService.isValid;
        };



        _this.isSignalsTabValid = function () {
            return SignalService.isValid;
        };



        _this.isTriggersTabValid = function () {
            return TriggerService.isValid;
        };



        _this.isSchedulesTabValid = function () {
            return ScheduleService.isValid;
        };
    });


var fileNameTarget = null;
var fileNameScope = null;

TaskerApp.directive('filenameChange', function () {
    return function (scope, element, attrs) {
        element.bind("keydown keyup", function (event) {
            var target = event.target;
            console.log("keyup: " + event.which);
            console.log("scope " + scope);
            // did the user press '{'
            if (event.which === 219) {
                var selectionStart = target.selectionStart;
                console.log("selectionStart: " + selectionStart);
                if (1 < selectionStart) {
                    // check to see if the typed character and the previous character make "{{"
                    var value = target.value;
                    console.log("value: " + value);
                    var substr = value.substring(selectionStart - 2, selectionStart);
                    console.log("substr: " + substr);
                    if ("{{" === substr) {
                        fileNameTarget = target;
                        fileNameScope = scope;
                        showFileNameOptions();
                    }
                }

            } else if ("keydown" === event.type && event.which === 112) {

                var selectionStart = target.selectionStart;
                console.log("selectionStart: " + selectionStart);
                var prevLeftCurlyBrace = target.value.lastIndexOf("{{", selectionStart);
                console.log("prevLeftCurlyBrace: " + prevLeftCurlyBrace);
                var prevRightCurlyBrace = target.value.lastIndexOf("}}", selectionStart);
                console.log("prevRightCurlyBrace: " + prevRightCurlyBrace);
                var nextLeftCurlyBrace = target.value.indexOf("{{", selectionStart);
                console.log("nextLeftCurlyBrace: " + nextLeftCurlyBrace);
                var nextRightCurlyBrace = target.value.indexOf("}}", selectionStart);
                console.log("nextRightCurlyBrace: " + nextRightCurlyBrace);

                if ((prevLeftCurlyBrace > prevRightCurlyBrace) && -1 !== nextRightCurlyBrace && (-1 === nextLeftCurlyBrace || nextRightCurlyBrace < nextLeftCurlyBrace)) {
                    fileNameTarget = target;
                    fileNameScope = scope;
                    showFileNameOptions();
                }

                event.preventDefault();
            }
        });
    };
});



TaskerApp.service('WorkspaceFileService', function ($rootScope, $location) {
    var _service = this;

    _service.uploadFile = function (file) {
        if (file) {
            // make sure it is a .json file
            if (!file.name.endsWith(".json")) {
                bootbox.alert({
                    className: 'bb-danger',
                    title: "Invalid File",
                    message: "'" + file.name + "' is an invalid file.  Please select a valid workspace file that ends in .json"
                });
                return;
            }

            // work space files shouldnt be above 100K
            if (file.size > 102400) {
                bootbox.alert({
                    className: 'bb-danger',
                    title: "Invalid File",
                    message: "Selected file is too large"
                });
                return;
            }

            bootbox.confirm({
                className: 'bb-primary',
                title: "Are you sure you want to load the selected file?",
                message: "Are you sure you want to upload the '" + file.name + "' workspace?",
                buttons: {
                    cancel: {
                        label: 'No',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Yes',
                        className: 'btn-primary'
                    }
                },
                callback: function (result) {
                    if (result) {
                        var r = new FileReader();
                        r.onload = function (e) {
                            var contents = e.target.result;

                            try {
                                // make sure the filename does not contain spaces
                                var filename = file.name.replaceAll(' ', '_');
                                var fullname = "/flash/tasker/workspaces/" + filename;

                                JnrWebsocket.writeFile(fullname, contents, function () {
                                    bootbox.alert({
                                        className: 'bb-success',
                                        title: 'Upload Success!',
                                        message: '<b>' + file.name + "</b> uploaded as <b>" + fullname + '</b>',
                                        callback: function () {
                                            // also alert out java application that the new file should be ready to consume.
                                            JnrWebsocket.postMessage(2010, { Message: 'config.updated', fullname });

                                            // broadcast to any listeners that the workspace was loaded locally
                                            $rootScope.$broadcast('workspace.uploaded', { filename: filename });

                                            // $location.search('workspace', file.name);
                                        }
                                    });
                                });

                            } catch (e) {
                                alert("Failed uploading the file" + e.toString());
                                return false;
                            }
                            return true;
                        };
                        r.readAsText(file);
                    }
                }
            });
        }
    };

}
);