function parseAction(json, parent) {
    var type = json.Type;
    var action = null;

    try {
        switch (type) {
            case 'Control':
                action = parseControlAction(json);
                break;

            case 'Timing':
                action = parseTimingAction(json);
                break;

            case 'AnalogControl':
                action = parseAnalogControlAction(json);
                break;

            case 'RelayControl':
                action = parseRelayControlAction(json);
                break;

            case 'DinControl':
                action = parseDinControlAction(json);
                break;

            case 'ControlPanel':
                action = parseControlPanelAction(json);
                break;

            case 'DimmerControl':
                action = parseDimmerControlAction(json);
                break;

            case 'Network':
                action = parseNetworkAction(json);
                break;

            case 'Logger':
                action = parseLoggerAction(json);
                break;

            case 'Email':
                action = parseEmailAction(json);
                break;

            case 'Other':
                action = parseOtherAction(json);
                break;

            default:
                throw new Error('Unknown Action Type: ' + type);
        }
    } catch (err) {
        action = new UnknownAction(err);
    }

    if (null != action) {
        action.parse(json);
        if (parent) {
            // action.Parent = parent;
            console.log(action + ' added to ' + parent.Type + ':' + parent.Method);
        }
    } else {
        action = new UnknownAction("Unknown Action: " + JSON.stringify(json, null, 2));
    }

    return action;
}



function parseOtherAction(json) {
    var method = json.Method;

    switch (method) {
        case 'Comment':
            return new Comment();

        case 'CommandLine':
            return new CommandLine();

        case 'Reboot':
            return new Reboot();

        case 'ExecTask':
            return new ExecTask();

        case 'CancelTask':
            return new CancelTask();

        case 'ExecScript':
            return new ExecScript();

        case 'UserAlert':
            return new UserAlert();

        case 'FileDelete':
            return new FileDelete();

        case 'FilePrepend':
            return new FilePrepend();

        case 'FileMove':
            return new FileMove();

        case 'FileCopy':
            return new FileCopy();

        case 'RegistryWrite':
            return new RegistryWrite();

        case 'SetVariable':
            return new SetVariable();

        default:
            throw new Error('Unknown Other Action Method: ' + method);
    }
}



class Action extends TaskerObject {
    constructor() {
        super();
        this.Type = 'Action';
        this.Parent = undefined;
        this.Params = {};
    }


    parse(json) {
        if (json) {
            for (var prop in json) {
                if (json.hasOwnProperty(prop)) {
                    this[prop] = json[prop];
                }
            }
        }
    };


    getJson() {
        return JSON.stringify(this, function replacer(key, value) {
            // we want to remove a few items.  remove $$hashKey object as well as references to a 
            // parent.  Also remove any empty objects.
            if ('$$hashKey' === key || 'Parent' === key || {} == value) {
                return;
            }
            return value;
        }, 2);
    };


    // getIndex() {
    //     return this.Parent.Params.Actions.indexOf(this);
    // };



    getAdditionalHtml() {
        return '';
    };
}



class Actions extends Action {
    constructor() {
        super();
        this.Type = 'Actions';
        this.Params.Actions = [];
    }



    parse(json) {
        super.parse(json);

        for (var i = 0; i < this.Params.Actions.length; i++) {
            var actionJson = this.Params.Actions[i];
            this.Params.Actions[i] = parseAction(actionJson, this);
        }
    };



    getAdditionalHtml() {
        return '\r\n<div style="margin: 4px 24px; border-left: 1px solid #333; padding: 8px;">\r\n' +
            '<action-block action="action" actions="action.Params.Actions" edit="edit" parent="action" controller="controller"></action-block>\r\n' +
            '</div>\r\n';
    };
}



class Task {
    constructor(name, enabled) {
        this.Name = name || '';
        this.Enabled = enabled;
        this.Action = new ControlFunction();
        this.GUID = guid();
    }


    isEmpty() {
        if (this.Action && null != this.Action) {
            if (0 < this.Action.Params.Actions.length) {
                return false;
            }
        }

        return true;
    };
}



class ControlFunction extends Actions {
    constructor() {
        super();
        this.Type = 'Control';
        this.Method = 'Function';
    }
}



class OtherActionType extends Action {
    constructor() {
        super();
        this.Type = 'Other';
    }
}



class Comment extends OtherActionType {
    constructor(comment) {
        super();
        this.Method = 'Comment';
        this.Params.Comment = comment;
    }

    editTemplate() {
        return '// <textarea name="' + guid() + '" rows="1" required class="form-control" ng-model="action.Params.Comment" data-toggle="tooltip" title="Some text that is helpful to identify what actions are to follow" style="width: 800px;" />';
    }
}



class CommandLine extends OtherActionType {
    constructor(command) {
        super();
        this.Method = 'CommandLine';
        this.Params.Command = command || '';
    }

    editTemplate() {
        return 'Execute Command Line <input name="' + guid() + '" required class="form-control" ng-model="action.Params.Command" data-toggle="tooltip" title="The command that would be typed at the command line" style="width: 300px;" />';
    }
}



class Reboot extends OtherActionType {
    constructor() {
        super();
        this.Method = 'Reboot';
    }

    editTemplate() {
        return '<strong>REBOOT</strong> <i>note: any actions added after this will not be executed.</i>';
    }
}



class ExecTask extends OtherActionType {
    constructor(taskName) {
        super();
        this.Method = 'ExecTask';
        this.Params.TaskName = taskName || '';
    }

    editTemplate() {
        return 'Execute Task <input name="' + guid() + '" required class="form-control" ng-model="action.Params.TaskName" data-toggle="tooltip" title="The Task to be executed.  Variables declared in this Task will stay in Scope be passed to the new Task." style="width: 200px;" />';
    }
}



class CancelTask extends OtherActionType {
    constructor(taskName) {
        super();
        this.Method = 'CancelTask';
        this.Params.TaskName = taskName || '';
    }

    editTemplate() {
        return 'Cancel Task <input name="' + guid() + '" required class="form-control" ng-model="action.Params.TaskName" data-toggle="tooltip" title="The Task to be cancelled" style="width: 200px;" />';
    }
}



class ExecScript extends OtherActionType {
    constructor(script) {
        super();
        this.Method = 'ExecScript';
        this.Params.Script = script || '';
    }

    editTemplate() {
        return '<div style="display:inline-grid">'
            + 'Execute Script <textarea name="' + guid() + '" required rows=8 class="form-control" ng-model="action.Params.Script" data-toggle="tooltip" title="The Task to be executed" style="width: 600px;" />'
            + '</div>';
    }
}



class UserAlert extends OtherActionType {
    constructor(message) {
        super();
        this.Method = 'UserAlert';
        this.Params.Message = message || '';
    }

    editTemplate() {
        return '<div style="display:inline-grid">'
            + 'User Alert <textarea name="' + guid() + '" required rows=1 class="form-control" ng-model="action.Params.Message" data-toggle="tooltip" title="The Message to send to the web page to show to the user" style="width: 600px;" />'
            + '</div>';
    }
}



class SetVariable extends OtherActionType {
    constructor() {
        super();
        this.Method = 'SetVariable';
        this.Params = {};
    }

    editTemplate() {
        $.validator.addMethod("reservedname", function (value, element) {
            value = value.toLowerCase();
            var invalid = (value == 'din'
                || value == 'rout'
                || value == 'iin'
                || value == 'iout'
                || value == 'vin'
                || value == 'vout'
                || value == 'temp'
                || value == 'env'
                || value == 'reg'
                || value == 'registry'
                || value == 'string'
                || value == 'date'
                || value == 'double'
                || value == 'int'
                || value == 'fpled'
                || value == 'arrayutils'
                || value == 'file'
                || value == 'time'
                || value == 'math');
            return !invalid;
        }, "The Name you are trying to use is RESERVED");

        return '<b>Set</b> <input class="form-control alphanumeric" ng-model="action.Params.Variable" style="width:150px" data-toggle="tooltip" title="A variable name.  Prefixing \'$$\' makes the variable Global and can be referenced by other Tasks" data-rule-reservedname="true" />' +
            ' = <input class="form-control" ng-model="action.Params.Conditional" style="width:450px" data-toggle="tooltip" title="The value to assign to the variable" />';
    }
}



class FileDelete extends OtherActionType {
    constructor(filename) {
        super();
        this.Method = 'FileDelete';
        this.Params.Filename = filename || '';
    }

    editTemplate() {
        return 'Delete File <input name="' + guid() + '" required class="form-control" ng-model="action.Params.Filename" data-toggle="tooltip" title="The Filename to be deleted" style="width: 400px;" />';
    }
}



class FilePrepend extends OtherActionType {
    constructor(filename, text) {
        super();
        this.Method = 'FilePrepend';
        this.Params.Filename = filename || '';
        this.Text = text || '';
    }

    editTemplate() {
        return '<div style="display:inline-grid">' +
            'Prepend to File <input name="' + guid() + '" required class="form-control" ng-model="action.Params.Filename" data-toggle="tooltip" title="The Filename to prepend the following data to" style="width: 400px;" />' +
            'Text <input type="text" name="' + guid() + '" class="form-control" id="prepend-text" placeholder="File Prepend Text" ng-model="action.Params.Text" ng-focus="loggerCtrl.setLastFocused($event)" style="width: 800px" name="headers-{{$index}}" required>' +
            '</div>';
    }
}



class FileMove extends OtherActionType {
    constructor() {
        super();
        this.Method = 'FileMove';
    }

    editTemplate() {
        return 'Move Source File <input name="' + guid() + '" required class="form-control" ng-model="action.Params.SourceFilename" data-toggle="tooltip" title="The Source Filename" style="width: 300px;" required />' +
            ' to Destination <input name="' + guid() + '" required class="form-control" ng-model="action.Params.DestinationFilename" data-toggle="tooltip" title="The Destination Filename" style="width: 300px;" required/ >' +
            // ' Should overwrite <input type="checkbox" name="' + guid() + '" ng-model="action.Params.Overwirte" class="form-control"' +
            // ' ng-init="action.Params.Overwirte = (undefined == action.Params.Overwirte) ? true: action.Params.Overwirte">';
            '';
    }
}



class FileCopy extends OtherActionType {
    constructor() {
        super();
        this.Method = 'FileCopy';
    }

    editTemplate() {
        return 'Copy Source File <input name="' + guid() + '" required class="form-control" ng-model="action.Params.SourceFilename" data-toggle="tooltip" title="The Source Filename" style="width: 300px;" required />' +
            ' to Destination <input name="' + guid() + '" required class="form-control" ng-model="action.Params.DestinationFilename" data-toggle="tooltip" title="The Destination Filename" style="width: 300px;" required/ >' +
            // ' Should overwrite <input type="checkbox" name="' + guid() + '" ng-model="action.Params.Overwirte" class="form-control"' +
            // ' ng-init="action.Params.Overwirte = (undefined == action.Params.Overwirte) ? true: action.Params.Overwirte">';
            '';
    }
}



class RegistryWrite extends OtherActionType {
    constructor() {
        super();
        this.Method = 'RegistryWrite';
    }

    editTemplate() {
        return '<b>Registry Write</b> Key: <input name="' + guid() + '" required class="form-control" ng-model="action.Params.KeyName" data-toggle="tooltip" title="Registry Key" style="width: 300px;" required />' +
            ' Value <input name="' + guid() + '" required class="form-control" ng-model="action.Params.Value" data-toggle="tooltip" title="Registry Key Value" style="width: 300px;" required/ >';
    }
}



class UnknownAction extends Action {
    constructor(err) {
        super();
        this.Err = err;
    }

    editTemplate() {
        return 'Unknown action: ' + this.Err;
    }
}