/* global bootbox, Tasker */

TaskerApp.controller('DevicesController', function ($scope, $compile, DeviceService) {
    var _this = this;



    _this.areDevicesLoaded = function () {
        return DeviceService.DevicesLoaded;
    };



    _this.getDevices = function () {
        return DeviceService.DeviceList;
    };



    _this.selectDevice = function (device) {
        DeviceService.SelectedDevice = device;
    };



    _this.getSelectedDevice = function () {
        return DeviceService.SelectedDevice;
    };



    _this.addDevice = function () {
        if (undefined != _this.addDeviceDialog) return;

        var template = '<form id="device-form">' +
            ' <div class="form-group" style="margin-bottom:12px;">' +
            '  <label>What is the name of the new Device?</label>' +
            '  <input type="text" id="device-name" class="form-control bootbox-input" required>' +
            ' </div>' +
            ' <div class="form-group" style="margin-bottom:12px;">' +
            '  <label>What is the new Device Type</label>' +
            '  <p><i>Disabled devices below are coming soon.</i></p>' +
            '  <select id="device-type" class="form-control" ng-mousedown="deviceCtrl.ShowDebugDevices = $event.ctrlKey" required>' +
            '   <option>Ethernet Device</option>' +
            '   <option>SNMP Device</option>' +
            '   <option ng-disabled ="!deviceCtrl.ShowDebugDevices">MODBUS Device</option>' +
            '   <option ng-disabled ="!deviceCtrl.ShowDebugDevices">Projector Device</option>' +
            '   <option ng-disabled ="!deviceCtrl.ShowDebugDevices">Sound Processor Device</option>' +
            '  </select>' +
            ' </div>' +
            '</form>';

        var linkFn = $compile(template);
        var html = linkFn($scope);

        _this.addDeviceDialog = bootbox.dialog({
            className: 'bb-success',
            title: "Create a new Device",
            message: html,
            buttons: {
                cancel: {
                    label: 'Cancel',
                    className: 'btn-default'
                },
                confirm: {
                    label: 'OK',
                    className: 'btn-success',
                    callback: function (result) {
                        var form = $("#device-form");
                        form.validate({ errorPlacement: function (error, element) { } });

                        if (form.valid()) {
                            var nameElement = form.find('#device-name');
                            var name = nameElement.val();

                            if (name) {
                                var typeElement = form.find('#device-type');
                                var type = typeElement.children("option:selected").val();
                                var device = null;
                                switch (type) {
                                    case 'Ethernet Device':
                                        device = new EthernetDevice(name);
                                        break;

                                    case 'SNMP Device':
                                        device = new SNMPDevice(name);
                                        break;

                                    case 'MODBUS Device':
                                        device = new ModbusDevice(name);
                                        break;

                                    case 'Projector Device':
                                        device = new ProjectorDevice(name);
                                        break;

                                    case 'Sound Processor Device':
                                        device = new SoundProcessorDevice(name);
                                        break;
                                }

                                if (null != device) {
                                    if (undefined == DeviceService.DeviceList) DeviceService.DeviceList = [];
                                    DeviceService.DeviceList.push(device);
                                }

                                $scope.$apply();
                            } else return false;
                        } else return false;
                    }
                }
            }
        });

        _this.addDeviceDialog.init(function () {
            $compile(_this.addDeviceDialog.contents())($scope);

            validate_alphanumeric($(".bootbox-input"));
            confirm_or_submit_on_enter($(".bootbox"));

            var newName = getNextAvailableName(DeviceService.DeviceList, "New_Device");
            $("#device-name").val(newName);
        }).on('shown.bs.modal', function () {
            $('#device-name').focus();
        }).on('hidden.bs.modal', function () {
            _this.addDeviceDialog = undefined;
        });
    };



    _this.editDeviceName = function (device) {
        if (undefined != _this.editDeviceNameDialog) return;

        _this.editDeviceNameDialog = bootbox.prompt({
            className: 'bb-success',
            title: "Edit the name of the Device?",
            message: '<p class="alert alert-warning"><b>Careful!</b>  Changing the name of the' +
                ' Device can have negative consequences if it it referenced elsewhere in Tasker. ' +
                ' Please make sure to change the name used in the references.</p>',
            buttons: {
                cancel: {
                    label: 'Cancel', className: 'btn-default'
                },
                confirm: {
                    label: 'Update', className: 'btn-success'
                }
            },
            value: device.Name,
            callback: function (result) {
                if (result) {
                    device.Name = result;
                    $scope.$apply();
                }
            }
        }).init(function () {
            validate_alphanumeric($(".bootbox-input"));
            confirm_or_submit_on_enter($(".bootbox-input"));
        }).on('shown.bs.modal', function () {
            $('#device-name').focus();
        }).on('hidden.bs.modal', function () {
            _this.editDeviceNameDialog = undefined;
        });
    };



    _this.removeDevice = function (device) {
        if (undefined != _this.removeDeviceDialog) return;

        _this.removeDeviceDialog = bootbox.confirm({
            className: 'bb-warning',
            title: "Are you sure?",
            message: "Are you sure you wish to remove the '" + device.Name + "' device?",
            buttons: {
                cancel: {
                    label: 'Cancel',
                    className: 'btn-default'
                },
                confirm: {
                    label: 'Remove',
                    className: 'btn-warning'
                }
            },
            callback: function (result) {
                if (result) {

                    var index = DeviceService.DeviceList.indexOf(device);
                    if (-1 != index) {
                        DeviceService.DeviceList.splice(index, 1);
                    }

                    $scope.$apply();
                }

            }
        }).init(function () {
            confirm_or_submit_on_enter($(".bootbox"));
        }).on('hidden.bs.modal', function () {
            _this.removeDeviceDialog = undefined;
        });
    };



    _this.getProjectorDevices = function () {
        return DeviceService.Projectors;
    };



    _this.getProjectorForType = function (type) {
        var projectors = DeviceService.Projectors.filter(function (element) {
            return element.Type == type;
        });
        if (1 == projectors.length) {
            return projectors[0];
        } return null;
    };



    _this.getSoundProcessorDevices = function () {
        return DeviceService.SoundProcessors;
    };



    _this.getSoundProcessorForType = function (type) {
        var soundProcessors = DeviceService.SoundProcessors.filter(function (element) {
            return element.Type == type;
        });
        if (1 == soundProcessors.length) {
            return soundProcessors[0];
        } return null;
    };



    _this.moveUp = function (device, currentIndex) {
        // remove from current position
        DeviceService.DeviceList.splice(currentIndex, 1);

        // assign the device to the previous position 
        DeviceService.DeviceList.splice(currentIndex - 1, 0, device);
    };



    _this.moveDown = function (device, currentIndex) {
        // remove from current position
        DeviceService.DeviceList.splice(currentIndex, 1);

        // assign the device to the previous position 
        DeviceService.DeviceList.splice(currentIndex + 1, 0, device);
    };
});



/*
   SERVICE
*/
TaskerApp.service('DeviceService', function ($rootScope) {
    var _this = this;
    _this.DeviceList = [];
    _this.SelectedDevice = null;
    _this.DevicesLoaded = false;



    _this.load = function (deviceListJson) {
        if (undefined == deviceListJson) deviceListJson = [];

        try {
            parseDeviceList(deviceListJson);
        } catch (err) {
            console.error(err);
        }

        _this.SelectedDevice = null;
        _this.DevicesLoaded = true;

        return _this.DeviceList;
    };



    function parseDeviceList(deviceArray) {
        // empty our array of devices
        _this.DeviceList = [];

        deviceArray.forEach(function (deviceJson) {
            var device = null;
            switch (deviceJson.Type) {
                case 'EthernetDevice':
                    device = new EthernetDevice(deviceJson.Name);
                    break;

                case 'SNMPDevice':
                    device = new SNMPDevice(deviceJson.Name);
                    break;

                case 'ModbusDevice':
                    device = new ModbusDevice(deviceJson.Name);
                    break;

                case 'ProjectorDevice':
                    device = new ProjectorDevice(deviceJson.Name);
                    break;

                case 'SoundProcessorDevice':
                    device = new SoundProcessorDevice(deviceJson.Name);
                    break;

            }
            device.Params = deviceJson.Params;
            device.GUID = deviceJson.GUID;
            _this.DeviceList.push(device);
        });
    }



    _this.isValid = true;
    _this.validate = function () {
        var form = $("#devices-form");
        if (0 < form.length) {
            form.validate({ ignore: ".ignore", errorPlacement: function (error, element) { } });
            _this.isValid = form.valid();
        } else _this.isValid = true;
        return _this.isValid;
    };



    $.get('/tasker/devices/projectors/projectors.json', function (data) {
        _this.Projectors = JSON.parse(data);
        $rootScope.$apply();
    });



    $.get('/tasker/devices/soundprocessors/soundprocessors.json', function (data) {
        _this.SoundProcessors = JSON.parse(data);
        $rootScope.$apply();
    });

});



TaskerApp.directive('deviceEdit', function ($compile) {
    return {
        restrict: 'A', // attribute
        scope: {
            device: '=',
            controller: '='
        },
        link: function ($scope, element, attrs) {
            var template = $scope.device.editTemplate();
            element.html(template).show();
            $compile(element.contents())($scope);

            $('[data-toggle="tooltip"]').tooltip({ delay: 500 });


            $scope.$watch('device.Params', function (newValue, oldValue) {
                if ($scope.device instanceof SoundProcessorDevice) {
                    $scope.deviceType =
                        $scope.controller.getSoundProcessorForType($scope.device.Params.SoundProcessorType);
                } else if ($scope.device instanceof ProjectorDevice) {
                    $scope.deviceType =
                        $scope.controller.getProjectorForType($scope.device.Params.ProjectorType);
                }
            }, true);
        }
    };
});