/* global bootbox, Tasker */

TaskerApp.controller('TriggerController',
    function ($scope, $compile, TaskService, TriggerService, DevicesService, SignalService) {
        var _this = this;



        _this.getDevicesInfo = function () {
            return DevicesService.DevicesInfo;
        };



        _this.getDeviceInfoByType = function (type) {
            return DevicesService.getDeviceInfoByType(type);
        };



        _this.getTasks = function () {
            return TaskService.TaskList;
        };



        _this.getSignals = function () {
            return SignalService.SignalsList;
        };



        _this.getDiscreteSignals = function () {
            var discreteSignals = [];
            SignalService.SignalsList.forEach(function (signal) {
                if ("DIN" == signal.SelectedDevice || "ROUT" == signal.SelectedDevice)
                    discreteSignals.push(signal);
            });
            return discreteSignals;
        };



        _this.getContinuousSignals = function () {
            var continuousSignals = [];
            SignalService.SignalsList.forEach(function (signal) {
                if ("DIN" != signal.SelectedDevice && "ROUT" != signal.SelectedDevice)
                    continuousSignals.push(signal);
            });
            return continuousSignals;
        };



        _this.areTriggersLoaded = function () {
            return TriggerService.TriggersLoaded;
        };



        _this.getTriggers = function () {
            return TriggerService.TriggersList;
        };



        _this.addTriggerProfile = function () {
            _this.NewTrigger = {};

            var template = '<div ng-include="\'triggers/new-trigger.html\'"></div>';
            var linkFn = $compile(template);
            var html = linkFn($scope);

            bootbox.dialog({
                className: 'bb-success',
                title: "Add Trigger",
                message: html,
                buttons: {
                    cancel: {
                        label: 'Cancel',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Add Trigger',
                        className: 'btn-success',
                        callback: function () {
                            var form = $("#new-trigger-form");
                            form.validate();

                            if (form.valid()) {
                                if (undefined == TriggerService.TriggersList) TriggerService.TriggersList = [];

                                // make sure there isnt already one with the same name
                                if (TriggerService.TriggersList) {
                                    for (var i = 0; i < TriggerService.TriggersList.length; i++) {
                                        if (TriggerService.TriggersList[i].Name === _this.NewTrigger.Name) {
                                            bootbox.alert({
                                                className: 'bb-danger',
                                                title: "Duplicate Trigger Name",
                                                message: "Trigger already exists with the name: <b>" + _this.NewTrigger.Name + '</b>'
                                            });
                                            return false;
                                        }
                                    }
                                }

                                var newTrigger = { Name: _this.NewTrigger.Name, Params: { Enabled: true }, GUID: guid() };
                                if ('DiscreteSignalTrigger' == _this.NewTrigger.Type) {
                                    newTrigger.Params.Trigger = new DiscreteSignalTrigger();
                                } else if ('ContinuousSignalTrigger' == _this.NewTrigger.Type) {
                                    newTrigger.Params.Trigger = new ContinuousSignalTrigger();
                                } else if ('ControlPanelSingleSwitchTrigger' == _this.NewTrigger.Type) {
                                    newTrigger.Params.Trigger = new ControlPanelSingleSwitchTrigger();
                                } else if ('ControlPanelMultiSwitchTrigger' == _this.NewTrigger.Type) {
                                    newTrigger.Params.Trigger = new ControlPanelMultiSwitchTrigger();
                                }

                                TriggerService.TriggersList.push(newTrigger);
                                $scope.$apply();
                            } else {
                                return false;
                            }
                        }
                    }
                }
            }).init(function () {
                $("#trigger-name").focus();
                submit_on_enter($(".bootbox-input"));
            });

        };



        _this.editTriggerName = function (trigger) {
            bootbox.prompt({
                className: 'bb-success',
                title: "Edit the name of the Trigger?",
                // message: '<p class="alert alert-warning"><b>Careful!</b>  Changing the name of the' +
                //     ' Trigger can have negative consequences if it it referenced elsewhere in Tasker. ' +
                //     ' Please make sure to change the name used in the references.</p>',
                buttons: {
                    cancel: {
                        label: 'Cancel',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Update',
                        className: 'btn-success'
                    }
                },
                value: trigger.Name,
                callback: function (result) {
                    if (result) {
                        trigger.Name = result;
                        $scope.$apply();
                    }
                }
            }).init(function () {
                validate_alphanumeric($(".bootbox-input"));
                submit_on_enter($(".bootbox-input"));
            });
        };



        _this.removeTriggerProfile = function (triggerProfile) {
            _this.SaveDialog = bootbox.confirm({
                className: 'bb-warning',
                title: "Are you sure you want to remove the " + triggerProfile.Name + " trigger profile?",
                message: "Are you sure you want to remove the <b>" + triggerProfile.Name + "</b> trigger profile?",
                buttons: {
                    cancel: {
                        label: 'Cancel',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Remove',
                        className: 'btn-warning'
                    }
                },
                callback: function (result) {
                    if (result) {
                        for (var i = TriggerService.TriggersList.length - 1; i >= 0; i--) {
                            if (TriggerService.TriggersList[i] === triggerProfile) {
                                TriggerService.TriggersList.splice(i, 1);
                                $scope.$apply();
                                return;
                            }
                        }
                    }
                }
            });
            _this.SaveDialog.init(function () {
                $("#bootbox-confirm-btn").prop('disabled', false);
                $("#bootbox-cancel-btn").prop('disabled', false);
            });
        };



        _this.addTriggerReset = function (triggerProfile) {
            _this.NewReset = {};

            var template = '<div ng-include="\'triggers/new-reset.html\'"></div>';
            var linkFn = $compile(template);
            var html = linkFn($scope);

            bootbox.dialog({
                className: 'bb-success',
                title: "Add Reset",
                message: html,
                buttons: {
                    cancel: {
                        label: 'Cancel',
                        className: 'btn-default'
                    },
                    confirm: {
                        label: 'Add Reset',
                        className: 'btn-success',
                        callback: function () {
                            var form = $("#new-reset-form");
                            form.validate();

                            if (form.valid()) {
                                if ('DelayedReset' == _this.NewReset.Type) {
                                    triggerProfile.Params.Reset = new DelayedReset();
                                } else if ('DiscreteSignalReset' == _this.NewReset.Type) {
                                    triggerProfile.Params.Reset = new DiscreteSignalReset();
                                } else if ('ContinuousSignalReset' == _this.NewReset.Type) {
                                    triggerProfile.Params.Reset = new ContinuousSignalReset();
                                } else if ('ControlPanelSingleSwitchReset' == _this.NewReset.Type) {
                                    triggerProfile.Params.Reset = new ControlPanelSingleSwitchReset();
                                } else if ('ControlPanelMultiSwitchReset' == _this.NewReset.Type) {
                                    triggerProfile.Params.Reset = new ControlPanelMultiSwitchReset();
                                }

                                $scope.$apply();
                                $scope.$broadcast('reset.added');
                            } else {
                                return false;
                            }
                        }
                    }
                }
            }).init(function () {
                $("#trigger-name").focus();
                validate_alphanumeric($("#trigger-name"));
                submit_on_enter($(".bootbox-input"));
            });

        };

    });



/*
   SERVICE
*/
TaskerApp.service('TriggerService', function ($rootScope) {
    var _this = this;
    _this.TriggersList = [];
    _this.SelectedTrigger = null;
    _this.TriggersLoaded = false;



    _this.load = function (triggerListJson) {
        if (undefined == triggerListJson) triggerListJson = [];

        try {
            _this.TriggersList = triggerListJson;

            _this.TriggersList.forEach(function (trigger) {
                var triggerTrigger = trigger.Params.Trigger;
                var triggerObject = null;
                if ('DiscreteSignalTrigger' == triggerTrigger.Type) {
                    triggerObject = new DiscreteSignalTrigger();
                } else if ('ContinuousSignalTrigger' == triggerTrigger.Type) {
                    triggerObject = new ContinuousSignalTrigger();
                } else if ('ControlPanelSingleSwitchTrigger' == triggerTrigger.Type) {
                    triggerObject = new ControlPanelSingleSwitchTrigger();
                } else if ('ControlPanelMultiSwitchTrigger' == triggerTrigger.Type) {
                    triggerObject = new ControlPanelMultiSwitchTrigger();
                }
                if (null != triggerObject) {
                    triggerObject.parse(triggerTrigger);
                    trigger.Params.Trigger = triggerObject;
                }

                var triggerReset = trigger.Params.Reset;
                var resetObject = null;
                if (null != triggerReset) {
                    if ('DelayedReset' == triggerReset.Type) {
                        resetObject = new DelayedReset();
                    } else if ('DiscreteSignalReset' == triggerReset.Type) {
                        resetObject = new DiscreteSignalReset();
                    } else if ('ContinuousSignalReset' == triggerReset.Type) {
                        resetObject = new ContinuousSignalReset();
                    } else if ('ControlPanelSingleSwitchReset' == triggerReset.Type) {
                        resetObject = new ControlPanelSingleSwitchReset();
                    } else if ('ControlPanelMultiSwitchReset' == triggerReset.Type) {
                        resetObject = new ControlPanelMultiSwitchReset();
                    }
                    if (null != resetObject) {
                        resetObject.parse(triggerReset);
                        trigger.Params.Reset = resetObject;
                    }
                }
            });
        } catch (err) {
            console.error(err);
        }

        _this.SelectedTrigger = null;
        _this.TriggersLoaded = true;

        return _this.TriggersList;
    };



    _this.validate = function () {
        var form = $("#triggers-form");
        form.validate({ errorPlacement: function (error, element) { } });
        return form.valid();
    };
});



TaskerApp.directive('triggerEdit', function ($compile) {
    return {
        restrict: 'A', // attribute
        scope: {
            trigger: '=',
            controller: '='
        },
        link: function ($scope, element, attrs) {
            // var trigger = $scope.trigger.Params.Trigger;
            // if (trigger) {
            var template = $scope.trigger.editTemplate();
            element.html(template).show();
            $compile(element.contents())($scope);

            $('[data-toggle="tooltip"]').tooltip();
            // }
        }
    };
});



class TriggerObject extends TaskerObject {
    constructor() {
        super();
        this.Type = 'Trigger';
        this.Parent = undefined;
        this.Params = {};
    }


    parse(json) {
        if (json) {
            for (var prop in json) {
                if (json.hasOwnProperty(prop)) {
                    this[prop] = json[prop];
                }
            }
        }
    };


    getJson() {
        return JSON.stringify(this, function replacer(key, value) {
            // we want to remove a few items.  remove $$hashKey object as well as references to a 
            // parent.  Also remove any empty objects.
            if ('$$hashKey' === key || 'Parent' === key || {} == value) {
                return;
            }
            return value;
        }, 2);
    };
}



class DiscreteSignalTrigger extends TriggerObject {
    constructor() {
        super();
        this.Type = 'DiscreteSignalTrigger';
    }

    editTemplate() {
        return 'On '
            + ' <select class="form-control" ng-model="trigger.Params.Signal"'
            + '   ng-options="availableSignal.Name group by availableSignal.Type for availableSignal in controller.getDiscreteSignals() track by availableSignal.Name"'
            + '   style="display: inherit; width: 150px" name="trigger-signal-2" required=""><option value="" disabled="" class="">Select a signal</option></select>'
            + ' <select class="form-control" style="display: inline;" ng-model="trigger.Params.State"'
            + '   ng-options="option as option for option in controller.getDeviceInfoByType(trigger.Params.Signal.SelectedDevice).Variables[0].Options"'
            + '   name="trigger-variable-1" required=""><option value="" disabled="" selected="" class="">State</option></select>';
    }
}



class ContinuousSignalTrigger extends TriggerObject {
    constructor() {
        super();
        this.Type = 'ContinuousSignalTrigger';
    }

    editTemplate() {
        return 'When '
            + ' <select class="form-control" ng-model="trigger.Params.Signal"'
            + '   ng-options="availableSignal.Name group by availableSignal.Type for availableSignal in controller.getContinuousSignals() track by availableSignal.Name"'
            + '   style="display: inherit; width: 150px" name="trigger-signal-' + guid() + '" required>'
            + '     <option value="" disabled>Select a signal</option>'
            + ' </select>'
            + ' <select class="form-control" style="display: inline;"'
            + '   ng-model="trigger.Params.Comparator"'
            + '   ng-options="option as option for option in controller.getDeviceInfoByType(trigger.Params.Signal.SelectedDevice).Variables[0].Comparators"'
            + '   name="trigger-comparator-' + guid() + '" required>'
            + ' </select>'
            + ' <input class="form-control" type="text" style="display: inline; width: 75px;"'
            + '   ng-model="trigger.Params.Value"'
            + '   onkeypress="return (event.charCode >= 48 && event.charCode <= 57) || event.charCode === 46"'
            + '   name="trigger-value-' + guid() + '" required>'
            + ' <i class="fa fa-exclamation-circle text-primary" data-toggle="tooltip" title="" data-original-title="A Continuous Signal Trigger needs to have a Reset defined otherwise it could trigger repeatedly.  A default 10 second DelayedReset will be used if a Reset is not defined."></i>';
    }
}



class ControlPanelSingleSwitchTrigger extends TriggerObject {
    constructor() {
        super();
        this.Type = 'ControlPanelSingleSwitchTrigger';
    }

    editTemplate() {
        return 'On Control Panel Switch Press'
            + ' <select name="control-panel-switch" class="form-control" ng-model="trigger.Params.Switch"'
            + '   ng-options="channel as channel for channel in [1,2,3,4,5,6,7,8,9,10,11,12]"></select>';
    }
}



class ControlPanelMultiSwitchTrigger extends TriggerObject {
    constructor() {
        super();
        this.Type = 'ControlPanelMultiSwitchTrigger';
    }

    editTemplate() {
        return 'On Control Panel Switches'
            + ' <select class="form-control" ng-model="trigger.Params.Switch1"'
            + '   ng-options="channel as channel for channel in [1,2,3,4,5,6,7,8,9,10,11,12]" required></select> and'
            + ' <select class="form-control" ng-model="trigger.Params.Switch2"'
            + '   ng-options="channel as channel for channel in [1,2,3,4,5,6,7,8,9,10,11,12]" required></select>'
            + ' <i class="fa fa-exclamation-circle text-primary" data-toggle="tooltip" title="" data-original-title="You must select a switch combination that combines two different physical switches"></i>';
    }
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////
TaskerApp.directive('resetEdit', function ($compile) {
    return {
        restrict: 'A', // attribute
        scope: {
            reset: '=',
            controller: '='
        },
        link: function ($scope, element, attrs, transclude) {
            if ($scope.reset) {
                var template = $scope.reset.editTemplate();
                element.html(template).show();
                $compile(element.contents())($scope);

                $('[data-toggle="tooltip"]').tooltip();
            }

            // here we didnt yet have a reset object decalred to load the template.  so 
            //  lets watch for one to get created.
            else {
                $scope.$on('reset.added', function () {
                    if ($scope.reset) {
                        var template = $scope.reset.editTemplate();
                        element.html(template).show();
                        $compile(element.contents())($scope);

                        $('[data-toggle="tooltip"]').tooltip();
                    }
                });
            }
        }
    };
});



class ResetObject extends TaskerObject {
    constructor() {
        super();
        this.Type = 'Reset';
        this.Parent = undefined;
        this.Params = {};
    }


    parse(json) {
        if (json) {
            for (var prop in json) {
                if (json.hasOwnProperty(prop)) {
                    this[prop] = json[prop];
                }
            }
        }
    };


    getJson() {
        return JSON.stringify(this, function replacer(key, value) {
            // we want to remove a few items.  remove $$hashKey object as well as references to a 
            // parent.  Also remove any empty objects.
            if ('$$hashKey' === key || 'Parent' === key || {} == value) {
                return;
            }
            return value;
        }, 2);
    };
}



class DelayedReset extends ResetObject {
    constructor() {
        super();
        this.Type = 'DelayedReset';
    }

    editTemplate() {
        return 'After <input type="text" class="form-control" ng_model="reset.Params.DelayTime" style="width:50px;"> seconds';
    }
}



class DiscreteSignalReset extends ResetObject {
    constructor() {
        super();
        this.Type = 'DiscreteSignalReset';
    }

    editTemplate() {
        return 'On '
            + ' <select class="form-control" ng-model="reset.Params.Signal"'
            + '   ng-options="availableSignal.Name group by availableSignal.Type for availableSignal in controller.getDiscreteSignals() track by availableSignal.Name"'
            + '   style="display: inherit; width: 150px" name="reset-signal-2" required=""><option value="" disabled="" class="">Select a signal</option></select>'
            + ' <select class="form-control" style="display: inline;" ng-model="reset.Params.State"'
            + '   ng-options="option as option for option in controller.getDeviceInfoByType(reset.Params.Signal.SelectedDevice).Variables[0].Options"'
            + '   name="reset-variable-1" required=""><option value="" disabled="" selected="" class="">State</option></select>';
    }
}



class ContinuousSignalReset extends ResetObject {
    constructor() {
        super();
        this.Type = 'ContinuousSignalReset';
    }

    editTemplate() {
        return 'When '
            + ' <select class="form-control" ng-model="reset.Params.Signal"'
            + '   ng-options="availableSignal.Name group by availableSignal.Type for availableSignal in controller.getContinuousSignals() track by availableSignal.Name"'
            + '   style="display: inherit; width: 150px" name="reset-signal-' + guid() + '" required>'
            + '     <option value="" disabled>Select a signal</option>'
            + ' </select>'
            + ' <select class="form-control" style="display: inline;"'
            + '   ng-model="reset.Params.Comparator"'
            + '   ng-options="option as option for option in controller.getDeviceInfoByType(reset.Params.Signal.SelectedDevice).Variables[0].Comparators"'
            + '   name="reset-comparator-' + guid() + '" required>'
            + ' </select>'
            + ' <input class="form-control" type="text" style="display: inline; width: 75px;"'
            + '   ng-model="reset.Params.Value"'
            + '   onkeypress="return (event.charCode >= 48 && event.charCode <= 57) || event.charCode === 46"'
            + '   name="reset-value-' + guid() + '" required>';
    }
}



class ControlPanelSingleSwitchReset extends ResetObject {
    constructor() {
        super();
        this.Type = 'ControlPanelSingleSwitchReset';
    }

    editTemplate() {
        return 'On Control Panel Switch Press'
            + ' <select class="form-control" name="switch-number-' + guid() + '"'
            + ' ng-model="reset.Params.Switch" ng-options="channel as channel for channel in [1,2,3,4,5,6,7,8,9,10,11,12]"></select>';
    }
}



class ControlPanelMultiSwitchReset extends ResetObject {
    constructor() {
        super();
        this.Type = 'ControlPanelMultiSwitchReset';
    }

    editTemplate() {
        return 'On Control Panel Switches'
            + ' <select class="form-control" name="switch-number1-' + guid() + '" ng-model="reset.Params.Switch1"'
            + '   ng-options="channel as channel for channel in [1,2,3,4,5,6,7,8,9,10,11,12]" required></select> and'
            + ' <select class="form-control" name="switch-number2-' + guid() + '" ng-model="reset.Params.Switch2"'
            + '   ng-options="channel as channel for channel in [1,2,3,4,5,6,7,8,9,10,11,12]" required></select>'
            + ' <i class="fa fa-exclamation-circle text-primary" data-toggle="tooltip" title="" data-original-title="You must select a switch combination that combines two different physical switches"></i>';
    }
}