function parseAction(json, parent) {
    var type = json.Type;
    var action = null;

    try {
        switch (type) {
            case 'Control':
                action = parseControlAction(json);
                break;

            case 'Timing':
                action = parseTimingAction(json);
                break;

            case 'AnalogControl':
                action = parseAnalogControlAction(json);
                break;

            case 'RelayControl':
                action = parseRelayControlAction(json);
                break;

            case 'ControlPanel':
                action = parseControlPanelAction(json);
                break;

            case 'DimmerControl':
                action = parseDimmerControlAction(json);
                break;

            case 'Network':
                action = parseNetworkAction(json);
                break;

            case 'Logger':
                action = parseLoggerAction(json);
                break;

            case 'Email':
                action = parseEmailAction(json);
                break;

            case 'Other':
                action = parseOtherAction(json);
                break;

            default:
                throw new Error('Unknown Action Type: ' + JSON.stringify(json, null, 2));
        }
    } catch (err) {
        action = new UnknownAction(err);
    }

    if (null != action) {
        action.parse(json);
        if (parent) {
            // action.Parent = parent;
            console.log(action + ' added to ' + parent.Type + ':' + parent.Method);
        }
    }

    return action;
}



function parseOtherAction(json) {
    var method = json.Method;

    switch (method) {
        case 'Comment':
            return new Comment();

        case 'CommandLine':
            return new CommandLine();

        case 'Reboot':
            return new Reboot();

        case 'ExecTask':
            return new ExecTask();

        default:
            throw new Error('Unknown Action Method: ' + JSON.stringify(json, null, 2));
    }
}



class Action extends TaskerObject {
    constructor() {
        super();
        this.Type = 'Action';
        this.Parent = undefined;
        this.Params = {};
    }


    parse(json) {
        if (json) {
            for (var prop in json) {
                if (json.hasOwnProperty(prop)) {
                    this[prop] = json[prop];
                }
            }
        }
    };


    getJson() {
        return JSON.stringify(this, function replacer(key, value) {
            // we want to remove a few items.  remove $$hashKey object as well as references to a 
            // parent.  Also remove any empty objects.
            if ('$$hashKey' === key || 'Parent' === key || {} == value) {
                return;
            }
            return value;
        }, 2);
    };


    // getIndex() {
    //     return this.Parent.Params.Actions.indexOf(this);
    // };



    getAdditionalHtml() {
        return '';
    };
}



class Actions extends Action {
    constructor() {
        super();
        this.Type = 'Actions';
        this.Params.Actions = [];
    }



    parse(json) {
        super.parse(json);

        for (var i = 0; i < this.Params.Actions.length; i++) {
            var actionJson = this.Params.Actions[i];
            this.Params.Actions[i] = parseAction(actionJson, this);
        }
    };



    getAdditionalHtml() {
        return '\r\n<div style="margin: 4px 24px; border-left: 1px solid #333; padding: 8px;">\r\n' +
            '<action-block action="action" actions="action.Params.Actions" edit="edit" parent="action" controller="controller"></action-block>\r\n' +
            '</div>\r\n';
    };
}



class Task {
    constructor(name, enabled) {
        this.Name = name || '';
        this.Enabled = enabled || true;
        this.Action = new ControlFunction();
        this.GUID = guid();
    }


    isEmpty = function () {
        if (this.Action && null != this.Action) {
            if (0 < this.Action.Params.Actions.length) {
                return false;
            }
        }

        return true;
    };
}



class ControlFunction extends Actions {
    constructor() {
        super();
        this.Type = 'Control';
        this.Method = 'Function';
    }
}



class OtherActionType extends Action {
    constructor() {
        super();
        this.Type = 'Other';
    }
}



class Comment extends OtherActionType {
    constructor(comment) {
        super();
        this.Method = 'Comment';
        this.Params.Comment = comment;
    }

    editTemplate() {
        return '// <input name="' + guid() + '" required class="form-control" ng-model="action.Params.Comment" data-toggle="tooltip" title="Some text that is helpful to identify what actions are to follow" style="width: 800px;" />';
    }
}



class CommandLine extends OtherActionType {
    constructor(command) {
        super();
        this.Method = 'CommandLine';
        this.Params.Command = command || '';
    }

    editTemplate() {
        return 'Execute Command Line <input name="' + guid() + '" required class="form-control" ng-model="action.Params.Command" data-toggle="tooltip" title="The command that would be typed at the command line" style="width: 300px;" />';
    }
}



class Reboot extends OtherActionType {
    constructor() {
        super();
        this.Method = 'Reboot';
    }

    editTemplate() {
        return '<strong>REBOOT</strong> <i>note: any actions added after this will not be executed.</i>';
    }
}



class ExecTask extends OtherActionType {
    constructor(taskName) {
        super();
        this.Method = 'ExecTask';
        this.Params.TaskName = taskName || '';
    }

    editTemplate() {
        return 'Execute Task <input name="' + guid() + '" required class="form-control" ng-model="action.Params.TaskName" data-toggle="tooltip" title="The Task to be executed" style="width: 200px;" />';
    }
}



class UnknownAction extends Action {
    constructor(err) {
        super();
        this.Err = err;
    }

    editTemplate() {
        return 'Unknown action: ' + this.Err;
    }
}