/**
 * @license
 * Visual Blocks Language
 *
 * Copyright 2012 Google Inc.
 * https://developers.google.com/blockly/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview Generating Json for procedure blocks.
 * @author fraser@google.com (Neil Fraser)
 */
'use strict';

goog.provide('Blockly.Json.procedures');

goog.require('Blockly.Json');


Blockly.Json['procedures_defreturn'] = function (block) {
    // Call a procedure with no return value.
    var funcName = Blockly.Json.variableDB_.getName(block.getFieldValue('NAME'), Blockly.Procedures.NAME_TYPE);
    var branch = Blockly.Json.statementToCode(block, 'STACK');
    var branchJson = [];
    if ('' !== branch) branchJson = JSON.parse('[' + branch + ']');

    var codeJson = { Type: 'Control', Method: 'Function', Params: { Name: funcName, Actions: branchJson } };
    var code = JSON.stringify(codeJson, null, 2);
    return code;
};


// Defining a procedure without a return value uses the same generator as
// a procedure with a return value.
Blockly.Json['procedures_defnoreturn'] =
    Blockly.Json['procedures_defreturn'];

Blockly.Json['procedures_callreturn'] = function (block) {
    // Call a procedure with a return value.
    var funcName = Blockly.Json.variableDB_.getName(
        block.getFieldValue('NAME'), Blockly.Procedures.NAME_TYPE);
    var args = [];
    for (var x = 0; x < block.arguments_.length; x++) {
        args[x] = Blockly.Json.valueToCode(block, 'ARG' + x,
            Blockly.Json.ORDER_COMMA) || 'null';
    }
    var code = funcName + '(' + args.join(', ') + ')';
    return [code, Blockly.Json.ORDER_FUNCTION_CALL];
};

Blockly.Json['procedures_callnoreturn'] = function (block) {
    // Call a procedure with no return value.
    var funcName = Blockly.Json.variableDB_.getName(block.getFieldValue('NAME'), Blockly.Procedures.NAME_TYPE);

    var codeJson = { Type: 'Control', Method: 'FunctionCallNoReturn', Params: { Name: funcName } };
    var code = JSON.stringify(codeJson, null, 2);
    // if (block.nextConnection.targetConnection != null) code += ',\n';
    return code;
};

Blockly.Json['procedures_ifreturn'] = function (block) {
    // Conditionally return value from a procedure.
    var condition = Blockly.Json.valueToCode(block, 'CONDITION',
        Blockly.Json.ORDER_NONE) || 'false';
    var code = 'if (' + condition + ') {\n';
    if (block.hasReturnValue_) {
        var value = Blockly.Json.valueToCode(block, 'VALUE',
            Blockly.Json.ORDER_NONE) || 'null';
        code += '  return ' + value + ';\n';
    } else {
        code += '  return;\n';
    }
    code += '}\n';
    return code;
};
