/**
 * @license
 * Visual Blocks Language
 *
 * Copyright 2012 Google Inc.
 * https://developers.google.com/blockly/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview Helper functions for generating Json for blocks.
 * @author fraser@google.com (Neil Fraser)
 */
'use strict';

goog.provide('Blockly.Json');

goog.require('Blockly.Generator');


/**
 * Json code generator.
 * @type {!Blockly.Generator}
 */
Blockly.Json = new Blockly.Generator('Json');

/**
 * List of illegal variable names.
 * This is not intended to be a security feature.  Blockly is 100% client-side,
 * so bypassing this list is trivial.  This is intended to prevent users from
 * accidentally clobbering a built-in object or function.
 * @private
 */
Blockly.Json.addReservedWords(
  'Blockly'
);


/**
 * Initialise the database of variable names.
 * @param {!Blockly.Workspace} workspace Workspace to generate code from.
 */
Blockly.Json.init = function (workspace) {
  // Create a dictionary of definitions to be printed before the code.
  Blockly.Json.definitions_ = Object.create(null);
  // Create a dictionary mapping desired function names in definitions_
  // to actual function names (to avoid collisions with user functions).
  Blockly.Json.functionNames_ = Object.create(null);

  if (!Blockly.Json.variableDB_) {
    Blockly.Json.variableDB_ =
      new Blockly.Names(Blockly.Json.RESERVED_WORDS_);
  } else {
    Blockly.Json.variableDB_.reset();
  }

  var defvars = [];
  var variables = Blockly.Variables.allVariables(workspace);
  for (var i = 0; i < variables.length; i++) {
    defvars[i] = 'var ' +
      Blockly.Json.variableDB_.getName(variables[i],
        Blockly.Variables.NAME_TYPE) + ';';
  }
  Blockly.Json.definitions_['variables'] = defvars.join('\n');
};

/**
 * Prepend the generated code with the variable definitions.
 * @param {string} code Generated code.
 * @return {string} Completed code.
 */
Blockly.Json.finish = function (code) {
  // code = '{"Commands":[\n' + code + '\n]}';
  var json = JSON.parse(code);
  //  code = JSON.stringify(json); //, null, 4);
  return code;
};

/**
 * Naked values are top-level blocks with outputs that aren't plugged into
 * anything.  A trailing semicolon is needed to make this legal.
 * @param {string} line Line of generated code.
 * @return {string} Legal line of code.
 */
Blockly.Json.scrubNakedValue = function (line) {
  return '{' + line + '}';
};

/**
 * Encode a string as a properly escaped Json string, complete with
 * quotes.
 * @param {string} string Text to encode.
 * @return {string} Json string.
 * @private
 */
Blockly.Json.quote_ = function (string) {
  // TODO: This is a quick hack.  Replace with goog.string.quote
  string = string.replace(/\\/g, '\\\\')
    .replace(/\n/g, '\\\n')
    .replace(/'/g, '\\\'');
  return '\"' + string + '\"';
};

/**
 * Common tasks for generating Json from blocks.
 * Handles comments for the specified block and any connected value blocks.
 * Calls any statements following this block.
 * @param {!Blockly.Block} block The current block.
 * @param {string} code The Json code created for this block.
 * @return {string} Json code with comments and subsequent blocks added.
 * @private
 */
Blockly.Json.scrub_ = function (block, code) {
  var commentCode = '';

  try {
    //
    // make sure all blocks contain guid and disabled attributes
    var json = JSON.parse(code);
    if (typeof json === 'object' && null !== json) {
      if (block.disabled) json.Disabled = true;
      json.Guid = block.guid;
      code = JSON.stringify(json);
    }
  } catch (err) { }

  var nb = block;
  while (nb.nextConnection != null) {

    nb = nb.nextConnection;
  }

  var nextBlock = block.nextConnection && block.nextConnection.targetBlock();
  if (null != nextBlock && '' !== code) code += ',\n';
  var nextCode = Blockly.Json.blockToCode(nextBlock);
  return commentCode + code + nextCode;
};
