/**
 * Blockly Demos: Code
 *
 * Copyright 2012 Google Inc.
 * https://developers.google.com/blockly/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview JavaScript for Blockly's Code demo.
 * @author fraser@google.com (Neil Fraser)
 */
'use strict';

/**
 * Create a namespace for the application.
 */
var Code = {};

/**
 * Lookup for names of supported languages.  Keys should be in ISO 639 format.
 */
Code.LANGUAGE_NAME = {
    'en': 'English'
};

/**
 * List of RTL languages.
 */
Code.LANGUAGE_RTL = ['ar', 'fa', 'he'];

/**
 * Blockly's main workspace.
 * @type {Blockly.WorkspaceSvg}
 */
Code.workspace = null;

/**
 * Extracts a parameter from the URL.
 * If the parameter is absent default_value is returned.
 * @param {string} name The name of the parameter.
 * @param {string} defaultValue Value to return if paramater not found.
 * @return {string} The parameter value or the default value if not found.
 */
Code.getStringParamFromUrl = function (name, defaultValue) {
    var val = location.search.match(new RegExp('[?&]' + name + '=([^&]+)'));
    return val ? decodeURIComponent(val[1].replace(/\+/g, '%20')) : defaultValue;
};

/**
 * Get the language of this user from the URL.
 * @return {string} User's language.
 */
Code.getLang = function () {
    var lang = Code.getStringParamFromUrl('lang', '');
    if (Code.LANGUAGE_NAME[lang] === undefined) {
        // Default to English.
        lang = 'en';
    }
    return lang;
};

/**
 * Is the current language (Code.LANG) an RTL language?
 * @return {boolean} True if RTL, false if LTR.
 */
Code.isRtl = function () {
    return Code.LANGUAGE_RTL.indexOf(Code.LANG) != -1;
};

/**
 * Load blocks saved on App Engine Storage or in session/local storage.
 * @param {string} defaultXml Text representation of default blocks.
 */
Code.loadBlocks = function (defaultXml) {
    try {
        var loadOnce = window.sessionStorage.loadOnceBlocks;
    } catch (e) {
        // Firefox sometimes throws a SecurityError when accessing sessionStorage.
        // Restarting Firefox fixes this, so it looks like a bug.
        var loadOnce = null;
    }
    if ('BlocklyStorage' in window && window.location.hash.length > 1) {
        // An href with #key trigers an AJAX call to retrieve saved blocks.
        BlocklyStorage.retrieveXml(window.location.hash.substring(1));
    } else if (loadOnce) {
        // Language switching stores the blocks during the reload.
        delete window.sessionStorage.loadOnceBlocks;
        var xml = Blockly.Xml.textToDom(loadOnce);
        Blockly.Xml.domToWorkspace(Code.workspace, xml);
    } else if (defaultXml) {
        // Load the editor with default starting blocks.
        var xml = Blockly.Xml.textToDom(defaultXml);
        Blockly.Xml.domToWorkspace(Code.workspace, xml);
    } else if ('BlocklyStorage' in window) {
        // Restore saved blocks in a separate thread so that subsequent
        // initialization is not affected from a failed load.
        window.setTimeout(BlocklyStorage.restoreBlocks, 0);
    }
};

/**
 * Save the blocks and reload with a different language.
 */
Code.changeLanguage = function () {
    // Store the blocks for the duration of the reload.
    // This should be skipped for the index page, which has no blocks and does
    // not load Blockly.
    // MSIE 11 does not support sessionStorage on file:// URLs.
    if (typeof Blockly != 'undefined' && window.sessionStorage) {
        var xml = Blockly.Xml.workspaceToDom(Code.workspace);
        var text = Blockly.Xml.domToText(xml);
        window.sessionStorage.loadOnceBlocks = text;
    }

    var languageMenu = document.getElementById('languageMenu');
    var newLang = encodeURIComponent(
        languageMenu.options[languageMenu.selectedIndex].value);
    var search = window.location.search;
    if (search.length <= 1) {
        search = '?lang=' + newLang;
    } else if (search.match(/[?&]lang=[^&]*/)) {
        search = search.replace(/([?&]lang=)[^&]*/, '$1' + newLang);
    } else {
        search = search.replace(/\?/, '?lang=' + newLang + '&');
    }

    window.location = window.location.protocol + '//' +
        window.location.host + window.location.pathname + search;
};

/**
 * Bind a function to a button's click event.
 * On touch enabled browsers, ontouchend is treated as equivalent to onclick.
 * @param {!Element|string} el Button element or ID thereof.
 * @param {!Function} func Event handler to bind.
 */
Code.bindClick = function (el, func) {
    if (typeof el == 'string') {
        el = document.getElementById(el);
    }
    // el.addEventListener('click', func, true);
    // el.addEventListener('touchend', func, true);
};



/**
 * Compute the absolute coordinates and dimensions of an HTML element.
 * @param {!Element} element Element to match.
 * @return {!Object} Contains height, width, x, and y properties.
 * @private
 */
Code.getBBox_ = function (element) {
    var height = element.offsetHeight;
    var width = element.offsetWidth;
    var x = 0;
    var y = 0;
    do {
        x += element.offsetLeft;
        y += element.offsetTop;
        element = element.offsetParent;
    } while (element);
    return {
        height: height,
        width: width,
        x: x,
        y: y
    };
};

/**
 * User's language (e.g. "en").
 * @type {string}
 */
Code.LANG = Code.getLang();

/**
 * List of tab names.
 * @private
 */
Code.TABS_ = ['blocks', 'json'];

Code.selected = 'blocks';

/**
 * Switch the visible pane when a tab is clicked.
 * @param {string} clickedName Name of tab clicked.
 */
Code.tabClick = function (clickedName) {
    // If the XML tab was open, save and render the content.
    // if (document.getElementById('tab_blocks').className == 'tabon') {
    //   Code.workspace.setVisible(false);
    // }
    // // Deselect all tabs and hide all panes.
    // for (var i = 0; i < Code.TABS_.length; i++) {
    //   var name = Code.TABS_[i];
    //   document.getElementById('tab_' + name).className = 'taboff';
    //   document.getElementById('content_' + name).style.visibility = 'hidden';
    // }

    // Select the active tab.
    Code.selected = clickedName;
    // document.getElementById('tab_' + clickedName).className = 'tabon';
    // Show the selected pane.
    document.getElementById('content_' + clickedName).style.visibility =
        'visible';
    Code.renderContent();
    if (clickedName == 'blocks') {
        Code.workspace.setVisible(true);
    }
    Blockly.fireUiEvent(window, 'resize');
};

/**
 * Populate the currently selected pane with content generated from the blocks.
 */
Code.renderContent = function () {
    var content = document.getElementById('content_' + Code.selected);
    // Initialize the pane.
    if (content.id == 'content_json') {
        var code = Blockly.Json.workspaceToCode(Code.workspace);
        content.textContent = code;
    }
};

/**
 * Initialize Blockly.  Called on page load.
 */
Code.init = function () {
    Code.initLanguage();

    var rtl = Code.isRtl();
    var container = document.getElementById('content_area');
    var onresize = function (e) {
        var bBox = Code.getBBox_(container);
        for (var i = 0; i < Code.TABS_.length; i++) {
            var el = document.getElementById('content_' + Code.TABS_[i]);
            el.style.top = bBox.y + 'px';
            el.style.left = bBox.x + 'px';
            // Height and width need to be set, read back, then set again to
            // compensate for scrollbars.
            el.style.height = bBox.height + 'px';
            el.style.height = (2 * bBox.height - el.offsetHeight) + 'px';
            el.style.width = bBox.width + 'px';
            el.style.width = (2 * bBox.width - el.offsetWidth) + 'px';
        }
        // Make the 'Blocks' tab line up with the toolbox.
        // if (Code.workspace && Code.workspace.toolbox_.width) {
        //   document.getElementById('tab_blocks').style.minWidth =
        //   (Code.workspace.toolbox_.width - 38) + 'px';
        //       // Account for the 19 pixel margin and on each side.
        //     }
    };
    onresize();
    window.addEventListener('resize', onresize, false);

    var toolbox = document.getElementById('toolbox');
    Code.workspace = Blockly.inject('content_blocks',
        {
            grid:
            {
                spacing: 25,
                length: 3,
                colour: '#ccc',
                snap: true
            },
            media: 'media/',
            rtl: rtl,
            toolbox: toolbox,
            zoom: {
                enabled: true,
                controls: true,
                wheel: false
            }
        });

    Code.loadBlocks('');

    if ('BlocklyStorage' in window) {
        // Hook a save function onto unload.
        BlocklyStorage.backupOnUnload(Code.workspace);
    }

    Code.tabClick(Code.selected);

    Code.bindClick('trashButton',
        function () {
            Code.discard();
            Code.renderContent();
        });
    Code.bindClick('runButton', Code.runJS);
    // Disable the link button if page isn't backed by App Engine storage.
    var linkButton = document.getElementById('linkButton');
    if ('BlocklyStorage' in window) {
        BlocklyStorage['HTTPREQUEST_ERROR'] = MSG['httpRequestError'];
        BlocklyStorage['LINK_ALERT'] = MSG['linkAlert'];
        BlocklyStorage['HASH_ERROR'] = MSG['hashError'];
        BlocklyStorage['XML_ERROR'] = MSG['xmlError'];
        Code.bindClick(linkButton,
            function () {
                BlocklyStorage.link(Code.workspace);
            });
    } else if (linkButton) {
        linkButton.className = 'disabled';
    }

    for (var i = 0; i < Code.TABS_.length; i++) {
        var name = Code.TABS_[i];
        Code.bindClick('tab_' + name,
            function (name_) {
                return function () {
                    Code.tabClick(name_);
                };
            }(name));
    }

    // Lazy-load the syntax-highlighting.
    window.setTimeout(Code.importPrettify, 1);


    // Code.load();
};







/**
 * Initialize the page language.
 */
Code.initLanguage = function () {
    // Set the HTML's language and direction.
    var rtl = Code.isRtl();
    document.dir = rtl ? 'rtl' : 'ltr';
    document.head.parentElement.setAttribute('lang', Code.LANG);

    // Sort languages alphabetically.
    var languages = [];
    for (var lang in Code.LANGUAGE_NAME) {
        languages.push([Code.LANGUAGE_NAME[lang], lang]);
    }
    var comp = function (a, b) {
        // Sort based on first argument ('English', 'Русский', '简体字', etc).
        if (a[0] > b[0])
            return 1;
        if (a[0] < b[0])
            return -1;
        return 0;
    };
    languages.sort(comp);
    // // Populate the language selection menu.
    // var languageMenu = document.getElementById('languageMenu');
    // languageMenu.options.length = 0;
    // for (var i = 0; i < languages.length; i++) {
    //   var tuple = languages[i];
    //   var lang = tuple[tuple.length - 1];
    //   var option = new Option(tuple[0], lang);
    //   if (lang == Code.LANG) {
    //     option.selected = true;
    //   }
    //   languageMenu.options.add(option);
    // }
    // languageMenu.addEventListener('change', Code.changeLanguage, true);

    // // Inject language strings.
    // document.title += ' ' + MSG['title'];
    // document.getElementById('title').textContent = MSG['title'];
    // document.getElementById('tab_blocks').textContent = MSG['blocks'];

    // document.getElementById('linkButton').title = MSG['linkTooltip'];
    // document.getElementById('runButton').title = MSG['runTooltip'];
    // document.getElementById('trashButton').title = MSG['trashTooltip'];




    var categories = ['catLogic', 'catLoops', /* 'catDevices', */ 'catText', 'catVariables', 'catFunctions'];
    for (var i = 0, cat; cat = categories[i]; i++) {
        document.getElementById(cat).setAttribute('name', MSG[cat]);
    }
    var textVars = document.getElementsByClassName('textVar');
    for (var i = 0, textVar; textVar = textVars[i]; i++) {
        textVar.textContent = MSG['textVariable'];
    }
    var listVars = document.getElementsByClassName('listVar');
    for (var i = 0, listVar; listVar = listVars[i]; i++) {
        listVar.textContent = MSG['listVariable'];
    }
};






/**
 * Discard all blocks from the workspace.
 */
Code.discard = function () {
    var count = Code.workspace.getAllBlocks().length;
    if (count < 2 ||
        window.confirm(MSG['discard'].replace('%1', count))) {
        Code.workspace.clear();
        window.location.hash = '';
    }
};

// Load the Code demo's language strings.
document.write('<script src="msg/' + Code.LANG + '.js"></script>\n');
// Load Blockly's language strings.
document.write('<script src="../../msg/js/' + Code.LANG + '.js"></script>\n');

window.addEventListener('load', Code.init);





Code.getWorkspace = function () {
    var xmlDom = Blockly.Xml.workspaceToDom(Code.workspace);
    return Blockly.Xml.domToText(xmlDom);
};


Code.getJsonCode = function () {
    // var stacks = [];

    try {
        Blockly.Json.init(Code.workspace);
        var blocks = Code.workspace.getTopBlocks(true);
        var code = '';
        for (var x = 0, block; block = blocks[x]; x++) {
            var blockCode = Blockly.Json.blockToCode(block);
            code += blockCode;

            // code = '[ ' + code + ' ]';
            // var jsonStack = JSON.parse(code);
            // var statements = { Statements: jsonStack };
            // stacks.push(JSON.stringify(statements));
        }

        if (0 < blocks.length && typeof blocks[0] !== 'function') {
            alert('not a function');
        }
    } catch (err) {
        console.log(err.stack);
        throw err;
    }

    return code;
};



Code.guid = function () {
    function s4() {
        return Math.floor((1 + Math.random()) * 0x10000)
            .toString(16)
            .substring(1);
    }
    return s4() + s4() + '-' + s4() + '-' + s4() + '-' +
        s4() + '-' + s4() + s4() + s4();
};



String.prototype.endsWith = function (suffix) {
    return this.indexOf(suffix, this.length - suffix.length) !== -1;
};







toastr.options.positionClass = 'toast-bottom-full-width';
toastr.options.tapToDismiss = false;


var _host = window.location.hostname;

// create our websocket connection
var JnrWebsocket = new JnrWebsocket();

var app = angular.module('taskPlusPlusApp', []);
app.controller('taskPlusPlusCtrl', function ($scope) {
    var _this = this;
    _this.currentFilename = "";
    $scope.snapFiles = [];
    _this.Version = '1.1';


    JnrWebsocket.connect();


    JnrWebsocket.addOnLoggedInListener(function () {
        JnrWebsocket.postMessage(1510, { Message: 'stacks.get' });
    });


    JnrWebsocket.addOnMessageListener(function (json) {
        var message = json.Message;

        if ('stacks.update' == message) {
            Blockly.Procedures.StackNames = json.StackNames;
        }

        else if ('error' == json.Message) {
            var message = json.StackTrace ? json.StackTrace.replaceAll('\r\n', '<br>') : '';
            toastr.error(message, json.ErrorMessage, { closeButton: true, timeOut: 0, extendedTimeOut: 0 });
        }
    });



    $scope.getStacks = function () {
        JnrWebsocket.getFileListing('/flash/snap/snaps')
            .then(function (response) {
                response.Content.sort(function (a, b) { return a.Name.toLowerCase().localeCompare(b.Name.toLowerCase()) });
                $scope.snapFiles = response.Content;
                $scope.$apply();
            });
    };



    $scope.new = function () {
        Code.workspace.clear();
        _this.currentFilename = "";
    };



    $scope.load = function (stackFile) {
        Code.workspace.clear();

        var fullPath = '/flash/snap/snaps/' + stackFile;
        JnrWebsocket.readFile(fullPath, function (response) {
            if ("Succeed" === response.Status) {
                var fileContents = Base64.decode(response.Data);
                var stackFileJson = JSON.parse(fileContents);

                var workspace = stackFileJson.Workspace;
                var xml = Blockly.Xml.textToDom(workspace);
                Blockly.Xml.domToWorkspace(Code.workspace, xml);

                _this.currentFilename = stackFile;
                $scope.$apply();
            }
        });
    };



    $scope.revert = function () {
        $scope.load(_this.currentFilename);
    };



    $scope.saveAs = function () {
        _this.currentFilename = prompt("Enter a name for the stack");
        _this.currentFilename.replaceAll(' ', '_');
        $scope.save();
    };



    $scope.save = function () {
        try {
            if (_this.currentFilename == "")
                $scope.saveAs();

            var workspace = Code.getWorkspace();

            // var stacks = Code.getJsonCode();
            // stacks = '[ ' + stacks.join(', ') + ' ]';  // Blank line between each section.
            // stackFileJson.Stacks = JSON.parse(stacks); //Blockly.Json.workspaceToCode();
            // console.log("Stacks: " + stackFileJson.Stacks);

            var codeJson = {};
            try {
                var code = Blockly.Json.workspaceToCode();
                codeJson = JSON.parse(code);
            } catch (err) {
                alert(err + '<br>' + err.stack);
                // bootbox.alert({
                //     className: 'bb-warning',
                //     title: 'Error',
                //     message: 'Unable to covert workspace to code: <b>' + err + '</b>'
                // });
            }

            var snapFileJson = { Snaps: codeJson, Workspace: workspace };
            var snapFile = JSON.stringify(snapFileJson, null, 2);
            // console.log("snapFile: " + snapFile);


            if (!_this.currentFilename.endsWith(".json"))
            _this.currentFilename += ".json";

            var filename = _this.currentFilename;
            if (filename.indexOf("stacks") != 0)
                filename = "/flash/snap/snaps/" + filename;

            JnrWebsocket.writeFile(filename, snapFile)
                .then(function (response) {
                    if ('Succeed' === response.Status) {
                        JnrWebsocket.postMessage(1510, { Message: 'stack.updated', Filename: filename });

                        bootbox.alert({
                            className: 'bb-success',
                            title: 'Success',
                            message: '<b>' + filename + '</b> has been successfully saved'
                        });
                    } else {
                        bootbox.alert({
                            className: 'bb-danger',
                            title: 'Error',
                            message: 'Unable to save <b>' + filename + '</b>'
                        });
                    }
                });
        } catch (err) {
            alert(err.stack);
        }
    };



    $scope.execute = function () {
        var topBlocks = Code.workspace.getTopBlocks();
        for (var i = 0; i < topBlocks.length; i++) {
            var block = topBlocks[i];

            if (block.previousConnection != null) {
                block.select();
                alert("invalid top block");
                return;
            }

            while (block != null) {
                console.log("top block: " + block.getRootBlock().id + ", block: " + block.id);
                block = block.getNextBlock();
            }
        }


        try {
            var block = null;
            for (var i = 0; i < topBlocks.length; i++) {
                var block = topBlocks[i];
                if (block === Blockly.selected)
                    break;
            }
            var functionName = block.getFieldValue('NAME');
            JnrWebsocket.postMessage(1510, { Message: 'stack.execute', FunctionName: functionName.replaceAll(' ', '_') });
        } catch (err) {
            alert(err.stack);
        }
    };



    $scope.delete = function () {
        bootbox.confirm({
            className: 'bb-primary',
            title: 'Are you sure?',
            message: 'Are you sure you want to permenantly remove <b>' + _this.currentFilename + '</b>?',
            callback: function (result) {
                if (result) {
                    JnrWebsocket.removeFile('/flash/snap/snaps/' + _this.currentFilename)
                        .then(function (response) {
                            if (response.Succeed) {
                                bootbox.alert({
                                    className: 'bb-success',
                                    title: 'Success',
                                    message: '<b>' + _this.currentFilename + '</b> has been successfully removed'
                                });
                            }
                        });
                }
            }
        });
    };



    $scope.getBlockCount = function () {
        var count = Code.workspace.getAllBlocks().length;
        return count;
    };
});