/**
 * @license
 * Visual Blocks Editor
 *
 * Copyright 2012 Google Inc.
 * https://developers.google.com/blockly/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview Utility functions for handling devices.
 * @author fraser@google.com (Neil Fraser)
 */
'use strict';

goog.provide('Blockly.Devices');

// TODO(scr): Fix circular dependencies
// goog.require('Blockly.Block');
goog.require('Blockly.Field');
goog.require('Blockly.Names');
goog.require('Blockly.Workspace');


/**
 * Category to separate device names from variables and generated functions.
 */
Blockly.Devices.NAME_TYPE = 'DEVICE';

/**
 * Find all user-created device definitions in a workspace.
 * @param {!Blockly.Workspace} root Root workspace.
 * @return {!Array.<!Array.<!Array>>} Pair of arrays, the
 *     first contains devices without return variables, the second with.
 *     Each device is defined by a three-element list of name, parameter
 *     list, and return value boolean.
 */
Blockly.Devices.allDevices = function (root) {
  var blocks = root.getAllBlocks();
  var devices = [];
  for (var i = 0; i < blocks.length; i++) {
    if (blocks[i].getDeviceName) {
      var tuple = blocks[i].getDeviceName();
      if (tuple) {
        devices.push(tuple);
      }
    }
  }
  devices.sort();
  return [devices];
};



/**
 * Ensure two identically-named devices don't exist.
 * @param {string} name Proposed device name.
 * @param {!Blockly.Block} block Block to disambiguate.
 * @return {string} Non-colliding name.
 */
Blockly.Devices.findLegalName = function (name, block) {
  if (block.isInFlyout) {
    // Flyouts can have multiple devices called 'do something'.
    return name;
  }
  while (!Blockly.Devices.isLegalName(name, block.workspace, block)) {
    // Collision with another device.
    var r = name.match(/^(.*?)(\d+)$/);
    if (!r) {
      name += '2';
    } else {
      name = r[1] + (parseInt(r[2], 10) + 1);
    }
  }
  return name;
};

/**
 * Does this device have a legal name?  Illegal names include names of
 * devices already defined.
 * @param {string} name The questionable name.
 * @param {!Blockly.Workspace} workspace The workspace to scan for collisions.
 * @param {Blockly.Block=} opt_exclude Optional block to exclude from
 *     comparisons (one doesn't want to collide with oneself).
 * @return {boolean} True if the name is legal.
 */
Blockly.Devices.isLegalName = function (name, workspace, opt_exclude) {
  var blocks = workspace.getAllBlocks();
  // Iterate through every block and check the name.
  for (var i = 0; i < blocks.length; i++) {
    if (blocks[i] == opt_exclude) {
      continue;
    }
    if (blocks[i].getDeviceDef) {
      var procName = blocks[i].getDeviceDef();
      if (Blockly.Names.equals(procName[0], name)) {
        return false;
      }
    }
  }
  return true;
};

/**
 * Rename a device.  Called by the editable field.
 * @param {string} text The proposed new name.
 * @return {string} The accepted name.
 * @this {!Blockly.Field}
 */
Blockly.Devices.rename = function (text) {
  // Strip leading and trailing whitespace.  Beyond this, all names are legal.
  text = text.replace(/^[\s\xa0]+|[\s\xa0]+$/g, '');

  // Ensure two identically-named devices don't exist.
  text = Blockly.Devices.findLegalName(text, this.sourceBlock_);
  // Rename any callers.
  var blocks = this.sourceBlock_.workspace.getAllBlocks();
  for (var i = 0; i < blocks.length; i++) {
    if (blocks[i].renameDevice) {
      blocks[i].renameDevice(this.text_, text);
    }
  }
  return text;
};

/**
 * Construct the blocks required by the flyout for the device category.
 * @param {!Array.<!Blockly.Block>} blocks List of blocks to show.
 * @param {!Array.<number>} gaps List of widths between blocks.
 * @param {number} margin Standard margin width for calculating gaps.
 * @param {!Blockly.Workspace} workspace The flyout's workspace.
 */
Blockly.Devices.flyoutCategory = function (blocks, gaps, margin, workspace) {
  var block = Blockly.Block.obtain(workspace, 'devices_set');
  block.initSvg();
  blocks.push(block);
  gaps.push(margin * 2);


  // if (Blockly.Blocks['devices_get']) {
  //   var block = Blockly.Block.obtain(workspace, 'devices_get');
  //   block.initSvg();
  //   blocks.push(block);
  //   gaps.push(margin * 2);
  // }
  // if (Blockly.Blocks['devices_send']) {
  //   var block = Blockly.Block.obtain(workspace, 'devices_send');
  //   block.initSvg();
  //   blocks.push(block);
  //   gaps.push(margin * 2);
  // }
  // if (gaps.length) {
  //   // Add slightly larger gap between system blocks and user calls.
  //   gaps[gaps.length - 1] = margin * 3;
  // }

  function populateDevices(deviceList, templateName) {
    // var deviceList = Blockly.Devices.allDevices(workspace.targetWorkspace);
    for (var x = 0; x < deviceList.length; x++) {
      var block = Blockly.Block.obtain(workspace, templateName);
      block.setFieldValue(deviceList[x][0], 'NAME');
      block.initSvg();
      blocks.push(block);
      gaps.push(margin * 2);
    }
  }

  var tuple = Blockly.Devices.allDevices(workspace.targetWorkspace);
  // populateDevices(tuple[0], 'devices_callnoreturn');
  populateDevices(tuple[0], 'devices_send');
};

/**
 * Find all the callers of a named device.
 * @param {string} name Name of device.
 * @param {!Blockly.Workspace} workspace The workspace to find callers in.
 * @return {!Array.<!Blockly.Block>} Array of caller blocks.
 */
Blockly.Devices.getCallers = function (name, workspace) {
  var callers = [];
  var blocks = workspace.getAllBlocks();
  // Iterate through every block and check the name.
  for (var i = 0; i < blocks.length; i++) {
    if (blocks[i].getDeviceCall) {
      var procName = blocks[i].getDeviceCall();
      // Device name may be null if the block is only half-built.
      if (procName && Blockly.Names.equals(procName, name)) {
        callers.push(blocks[i]);
      }
    }
  }
  return callers;
};

/**
 * When a device definition is disposed of, find and dispose of all its
 *     callers.
 * @param {string} name Name of deleted device definition.
 * @param {!Blockly.Workspace} workspace The workspace to delete callers from.
 */
Blockly.Devices.disposeCallers = function (name, workspace) {
  var callers = Blockly.Devices.getCallers(name, workspace);
  for (var i = 0; i < callers.length; i++) {
    callers[i].dispose(true, false);
  }
};

/**
 * When a device definition changes its parameters, find and edit all its
 * callers.
 * @param {string} name Name of edited device definition.
 * @param {!Blockly.Workspace} workspace The workspace to delete callers from.
 * @param {!Array.<string>} paramNames Array of new parameter names.
 * @param {!Array.<string>} paramIds Array of unique parameter IDs.
 */
Blockly.Devices.mutateCallers = function (name, workspace,
  paramNames, paramIds) {
  var callers = Blockly.Devices.getCallers(name, workspace);
  for (var i = 0; i < callers.length; i++) {
    callers[i].setDeviceParameters(paramNames, paramIds);
  }
};

/**
 * Find the definition block for the named device.
 * @param {string} name Name of device.
 * @param {!Blockly.Workspace} workspace The workspace to search.
 * @return {Blockly.Block} The device definition block, or null not found.
 */
Blockly.Devices.getDefinition = function (name, workspace) {
  var blocks = workspace.getAllBlocks();
  for (var i = 0; i < blocks.length; i++) {
    if (blocks[i].getDeviceDef) {
      var tuple = blocks[i].getDeviceDef();
      if (tuple && Blockly.Names.equals(tuple[0], name)) {
        return blocks[i];
      }
    }
  }
  return null;
};
