/**
 * @license
 * Visual Blocks Language
 *
 * Copyright 2012 Google Inc.
 * https://developers.google.com/blockly/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview Generating Json for logic blocks.
 * @author q.neutron@gmail.com (Quynh Neutron)
 */
'use strict';

goog.provide('Blockly.Json.logic');

goog.require('Blockly.Json');


Blockly.Json['controls_if'] = function (block) {
  // If/elseif/else condition.

  var params = {};
  var n = 0;

  // var if_condition = Blockly.Json.valueToCode(block, 'IF' + n, Blockly.Json.ORDER_ATOMIC);
  // if ('' !== if_condition) if_condition = JSON.parse(if_condition);
  // else if_condition = [];
  // params.IfConditional = if_condition;

  // var if_statements = Blockly.Json.statementToCode(block, 'DO' + n);
  // if ('' !== if_statements) if_statements = JSON.parse('[' + if_statements + ']');
  // else if_statements = [];
  // params.IfActions = if_statements;

  // ++n;
  while (block.getInput('IF' + n)) {
    var if_condition = Blockly.Json.valueToCode(block, 'IF' + n, Blockly.Json.ORDER_ATOMIC);
    if ('' !== if_condition) if_condition = JSON.parse(if_condition);
    else if_condition = [];
    params['IfConditional' + n] = if_condition;

    var if_statements = Blockly.Json.statementToCode(block, 'DO' + n);
    if ('' !== if_statements) if_statements = JSON.parse('[' + if_statements + ']');
    else if_statements = [];
    params['IfActions' + n] = if_statements;

    ++n;
  }

  var else_statements = Blockly.Json.statementToCode(block, 'ELSE');
  if ('' !== else_statements) else_statements = JSON.parse('[' + else_statements + ']');
  else else_statements = [];
  params.ElseActions = else_statements;

  var codeJson = { Type: 'Control', Method: 'IfBlock', Params: params };
  var code = JSON.stringify(codeJson);
  // if (block.nextConnection.targetConnection != null) code += ',\n';
  return code;
};


Blockly.Json['logic_expression'] = function (block) {
  var expression = block.getFieldValue('EXPRESSION');

  var codeJson = { Type: 'Logical', Method: 'LogicalExpression', Params: { Expression: expression } };
  var code = JSON.stringify(codeJson);
  return [code, Blockly.Json.ORDER_ATOMIC];
};


// Blockly.Json['logic_compare'] = function (block) {
//   // Comparison operator.
//   var OPERATORS = {
//     'EQ': '==',
//     'NEQ': '!=',
//     'LT': '<',
//     'LTE': '<=',
//     'GT': '>',
//     'GTE': '>='
//   };
//   var operator = OPERATORS[block.getFieldValue('OP')];
//   var order = (operator == '==' || operator == '!=') ?
//     Blockly.Json.ORDER_EQUALITY : Blockly.Json.ORDER_RELATIONAL;
//   var argument0 = Blockly.Json.valueToCode(block, 'A', order) || '0';
//   var argument1 = Blockly.Json.valueToCode(block, 'B', order) || '0';
//   var code = argument0 + ' ' + operator + ' ' + argument1;
//   return [code, order];
// };





Blockly.Json['logic_operation'] = function (block) {
  // Operations 'and', 'or'.
  var operator = (block.getFieldValue('OP') == 'AND') ? '&&' : '||';
  var order = (operator == '&&') ? Blockly.Json.ORDER_LOGICAL_AND :
    Blockly.Json.ORDER_LOGICAL_OR;
  var argument0 = Blockly.Json.valueToCode(block, 'A', order);
  var argument1 = Blockly.Json.valueToCode(block, 'B', order);
  if (!argument0 && !argument1) {
    // If there are no arguments, then the return value is false.
    argument0 = 'false';
    argument1 = 'false';
  } else {
    // Single missing arguments have no effect on the return value.
    var defaultArgument = (operator == '&&') ? 'true' : 'false';
    if (!argument0) {
      argument0 = defaultArgument;
    } else {
      argument0 = JSON.parse(argument0);
    }

    if (!argument1) {
      argument1 = defaultArgument;
    } else {
      argument1 = JSON.parse(argument1);
    }
  }

  var codeJson = { Type: 'Logical', Method: 'LogicalOperation', Params: { A: argument0, Operator: operator, B: argument1 } };
  var code = JSON.stringify(codeJson);
  return [code, Blockly.ORDER_ATOMIC];
};



Blockly.Json['logic_negate'] = function (block) {
  // Negation.
  var order = Blockly.Json.ORDER_LOGICAL_NOT;
  var argument0 = Blockly.Json.valueToCode(block, 'BOOL', order);
  if ('' !== argument0) argument0 = JSON.parse(argument0);
  else argument0 = null;

  var codeJson = { Type: 'Logical', Method: 'Negate', Params: { A: argument0 } };
  var code = JSON.stringify(codeJson);
  return [code, order];
};



// Blockly.Json['logic_boolean'] = function (block) {
//   // Boolean values true and false.
//   var code = (block.getFieldValue('BOOL') == 'TRUE') ? 'true' : 'false';
//   return [code, Blockly.Json.ORDER_ATOMIC];
// };

// Blockly.Json['logic_null'] = function (block) {
//   // Null data type.
//   return ['null', Blockly.Json.ORDER_ATOMIC];
// };

// Blockly.Json['logic_ternary'] = function (block) {
//   // Ternary operator.
//   var value_if = Blockly.Json.valueToCode(block, 'IF',
//     Blockly.Json.ORDER_CONDITIONAL) || 'false';
//   var value_then = Blockly.Json.valueToCode(block, 'THEN',
//     Blockly.Json.ORDER_CONDITIONAL) || 'null';
//   var value_else = Blockly.Json.valueToCode(block, 'ELSE',
//     Blockly.Json.ORDER_CONDITIONAL) || 'null';
//   var code = value_if + ' ? ' + value_then + ' : ' + value_else;
//   return [code, Blockly.Json.ORDER_CONDITIONAL];
// };
