 /*
  JNIOR Monitor/Configuration Page
  INTEG Process Group, Inc., 2919 E Hardies Rd, Gibsonia PA 
  724.933.9350
  
  File: console.js
  
  Javascript to simulate a console connection with access to the JANOS Command Line
  Interpreter. The emulates a Telnet connection or the RS-232 COM port serial connection.
  
  The user is hereby granted license to use, modify, and redistribute the contents of this
  file for any purpose, commercial or otherwise. No prior authorization by INTEG Process 
  Group, Inc. is required.
*/

// function display(string) emulates a static display
var caret = 0;	  
var col = 0;
var ins = 1;
var connected = false;
var transfer = false
var displaycache = "";

// **************************************************
// process received messages
var console_onmessage = chan.onmessage;
chan.onmessage = function(evt) {
	var jobj = JSON.parse(evt.data);

  if (jobj['Message'] === 'Console Stdout') {
  		displaycache = displaycache.concat(jobj['Data']);
  }	  

  else if (console_onmessage)
		console_onmessage(evt);	// chain message processing
}

function rundisplay() {
	if (displaycache.length > 0) {
		display(displaycache);
		displaycache = "";
	}
}

setInterval(rundisplay, 100);

function display(cdata) {
	var screen = document.getElementById("screen");
	var status = document.getElementById("status");
	
	// temp UTF-8 conversion
	try {
		cdata = decodeURIComponent(escape(cdata));
	} catch(err) {
	}
  	
	content = screen.value;
	for (i = 0; i < cdata.length; i++) {
		
		chcode = cdata.charCodeAt(i);
		
		// carriage return moves caret to beginning of line
		if (chcode == 0x0D) {
			// move to beginning or position after last newline
  		while (caret > 0 && content.charAt(caret-1) != '\n') {
  			caret--;
  		}
  		col = 0;
  		continue;
  	}
		
		// newline moves to the next line and resets the overstrike mode
		if (chcode == 0x0A) {
			// move ahead to end or next newline
  		while (caret < content.length && content.charAt(caret) != '\n') {
  			caret++;
  		}
  		// if at end append newline
  		if (caret == content.length) {
  			content += "\n";
  		}
  		
  		//position after newline
			caret++;
			col = 0;
  		ins = 1;
  		continue;
  	}
  	
  	// backspace moves caret back
		if (chcode == 0x08) {
			if (caret > 0) {
				caret--;
				if (col > 0)
					col--;
			}
			continue;
		}

		// overstrike if within content
		if (caret < content.length) {
			content = content.substr(0, caret) + cdata.charAt(i) + content.substr(caret+1);
			caret++;
			col++;
			continue;
		}

		content += cdata.charAt(i);
		caret++;
		col++;
	}
	
	// truncate textarea to maintain response
	if (content.length > 250000) {
		var n = 50000;
		while (n < content.length && content.charAt(n++) != '\n');
		content = content.substr(n);		
	}
	
	screen.value = content;
	screen.scrollTop = screen.scrollHeight;
	
	// position the caret
	screen.setSelectionRange(caret, caret);
	status.innerHTML = (ins === 1 ? "Ins " : "Ovr ") + col.toString();
}

// **************************************************
// function to process keystrokes in the text area
function onKeypress(e) {
	e.preventDefault();
	if (connected && !transfer) {
		ch = e.which;
		removeSel();		
		chan.sendJson({'Message': 'Console Stdin', 'Data': String.fromCharCode(ch)});
	}
}

function removeSel() {
	var screen = document.getElementById("screen");
	var start = screen.selectionStart;
	var end = screen.selectionEnd;
	var pent, pstart, pend, n, str;

	content = screen.value;
	if (start != end && ins == 1) {
		// determine beginning of entry line
		pent = caret;
		while (pent > 0 && content.charAt(pent-1) != '\n') {
			pent--;
		}
		
		// determine beginning of start highlighted line
		pstart = start;
		while (pstart > 0 && content.charAt(pstart-1) != '\n') {
			pstart--;
		}
		
		// determine beginning of end highlighted line
		pend = start;
		while (pend > 0 && content.charAt(pend-1) != '\n') {
			pend--;
		}
		
		// remove selection if on entry line
		if (pstart == pent && pend == pent) {
			// move to beginning of selection
			if (start < caret) {
				n = caret - start;
				str = "";
				while (n--)
					str += "\x1b[D";
				chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			}
			else if (caret < start) {
				n = start - caret;
				str = "";
				while (n--)
					str += "\x1b[C";
				chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			}
			
			// perfrom appropriate number of deletions
			screen.setSelectionRange(caret, caret);
			n = end - start;
			str = "";
			while (n--)
				str += "\x7f";
			chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			return true;
		}
	}
	return false;
}

// **************************************************
// function to process keystrokes in the text area
function onKeydown(e) {
	if (connected && !transfer) {
		var status = document.getElementById("status");
		
		ch = e.which;
		str = null;
		
		// pass recognized special keystrokes. Note that UP ARROW and DOWN ARROW
		//  reset the overstrike mode.
		if (ch === 8 || ch === 9)
			str = String.fromCharCode(ch);
		else if (e.ctrlKey && ch === 0x43)
			str = String.fromCharCode(3);
		else if (ch === 0x25)	// LEFT ARROW
			str = "\x1b[D";
		else if (ch === 0x26)	{ // UP ARROW
			str = "\x1b[A";
			ins = 1;
		}
		else if (ch === 0x27)	// RIGHT ARROW
			str = "\x1b[C";
		else if (ch === 0x28)	{ // DOWN ARROW
			str = "\x1b[B";
			ins = 1;
		}
		else if (ch === 0x23)	// END
			str = "\x1b[4~";
		else if (ch === 0x24)	// HOME
			str = "\x1b[1~";
		else if (ch === 0x2e) { // DEL
			if (!removeSel() || ins == 0)
				str = "\x7f";
		}
		else if (ch === 0x2d) {	// INSERT
			str = "\x1b[2~";
			ins ^= 1;
		}
		
		if (str != null)
		{
			e.preventDefault();
			chan.sendJson({'Message': 'Console Stdin', 'Data': str});
		}

		status.innerHTML = (ins === 1 ? "Ins " : "Ovr ") + col.toString();
	}
}

// **************************************************
function onCut(screen) {
	var content = screen.value;
	setTimeout( function() {
		screen.value = content;
	}, 4)
}

// **************************************************
function onPaste(e) {
  var clipboardData, pastedData, str, n;

  e.stopPropagation();
  e.preventDefault();

  clipboardData = e.clipboardData || window.clipboardData;
  pastedData = clipboardData.getData('Text');
  
  // Cannot paste multi-line data
  for (n=0; n<pastedData.length; n++) 
  	if (pastedData.charCodeAt(n) < 0x20 || pastedData.charCodeAt(n) >= 0x7f)
  		break;
  		
  str = pastedData.substring(0, n);
	chan.sendJson({'Message': 'Console Stdin', 'Data': str});
}

// **************************************************
function onPosition(screen) {
	var start = screen.selectionStart;
	var end = screen.selectionEnd;
	var pent, pclick, n, str;
	
	content = screen.value;
	if (start == end) {
		// determine beginning of entry line
		pent = caret;
		while (pent > 0 && content.charAt(pent-1) != '\n') {
			pent--;
		}
		
		// determine beginning of line clicked
		pclick = start;
		while (pclick > 0 && content.charAt(pclick-1) != '\n') {
			pclick--;
		}
		
		// prevent movement if not the same line
		if (pclick == pent) {
			screen.setSelectionRange(caret, caret);
			if (start < caret) {
				n = caret - start;
				str = "";
				while (n--)
					str += "\x1b[D";
				chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			}
			else if (caret < start) {
				n = start - caret;
				str = "";
				while (n--)
					str += "\x1b[C";
				chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			}
		}
	}
}
	
// **************************************************
// session
function onOpen() {
	chan.sendJson({'Message': 'Console Open'});
	document.getElementById('close').className = "button";
	document.getElementById('open').className = "button hide";
	document.getElementById('screen').focus();
	connected = true;
	transfer = false;
	document.getElementById('screen').className = "screen";
}

function onClose() {
	chan.sendJson({'Message': 'Console Close'});
	document.getElementById('close').className = "button hide";
	document.getElementById('open').className = "button";
	document.getElementById('open').focus();
	document.getElementById('status').innerHTML = "No Active Session";
	connected = false;
	transfer = false;
	document.getElementById('screen').className = "screen inactive";
	}
	
// **************************************************
// checks if console should be reopened
chan.onopen = function() {
	if (connected)
		setTimeout(onOpen, 1000);
}

function onClear() {
	var screen = document.getElementById("screen");
	var status = document.getElementById("status");
	
	screen.focus();			
	screen.value = "";
	caret = 0;
	col = 0;
	
	if (connected && !transfer)
		status.innerHTML = (ins === 1 ? "Ins " : "Ovr ") + col.toString();
	else
		status.innerHTML = "No Active Session";		
};
	
function scnpaste(e) {
	if (e.dataTransfer) {
		var data = e.dataTransfer.getData("Text");
		if (data) {
			removeSel();
			chan.sendJson({'Message': 'Console Stdin', 'Data': data});
			e.stopPropagation();
		  e.preventDefault();
		}
	}
}
