 /*
  JNIOR Monitor/Configuration Page
  INTEG Process Group, Inc., 2919 E Hardies Rd, Gibsonia PA 
  724.933.9350
  
  File: console.js
  
  Javascript to simulate a console connection with access to the JANOS Command Line
  Interpreter. The emulates a Telnet connection or the RS-232 COM port serial connection.
  
  The user is hereby granted license to use, modify, and redistribute the contents of this
  file for any purpose, commercial or otherwise. No prior authorization by INTEG Process 
  Group, Inc. is required.
*/

// function display(string) emulates a static display
var caret = 0;	  
var col = 0;
var ins = 1;
var connected = false;
var transfer = false
var displaycache = "";
var esc = false;
var seq = false;
var parm;
var selAnchor = -1;

// **************************************************
// process received messages
var console_onmessage = chan.onmessage;
chan.onmessage = function(evt) {
	var jobj = JSON.parse(evt.data);

  if (jobj['Message'] === 'Console Stdout') {
  		displaycache = displaycache.concat(jobj['Data']);
  }	  

  else if (console_onmessage)
		console_onmessage(evt);	// chain message processing
}

function rundisplay() {
	if (displaycache.length > 0) {
		display(displaycache);
		displaycache = "";
	}
}

setInterval(rundisplay, 100);

function display(cdata) {
	var screen = document.getElementById("screen");
	var status = document.getElementById("status");
	
	// temp UTF-8 conversion
	try {
		cdata = decodeURIComponent(escape(cdata));
	} catch(err) {
	}
  	
	content = screen.value;
	for (i = 0; i < cdata.length; i++) {
		
		chcode = cdata.charCodeAt(i);
		ch = cdata.charAt(i);
		
		// We handle ANSI escape sequences. Specifically for the movenment of the caret
		//  in all 4 directions. In addtion we support the erase to end of line function.
		//  Here we trap the ESC (0x1b) code and process.
		if (chcode == 0x1B) {
			esc = true;
			seq = false;
			continue;
		}
		
		if (esc && ch == '[') {
			seq = true;
			parm = 0;
			continue;
		}
		
		if (seq) {
			esc = false;
			
			if (chcode >= 0x30 && chcode <= 0x39) {
				parm = 10 * parm + chcode - 0x30;
				continue;
			}
			
			// wrap control - ignored
			if (ch == '?') {
				continue;
			}
			if (ch == 'l') {
				seq = false;
				continue;
			}			
			
			// UP
			if (ch == 'A') {
				seq = false;
				
				if (parm == 0)
					parm = 1;
					
				while (parm-- > 0) {
					// locate position on this line
					for (pos = 0; caret > 0 && content.charAt(caret-1) != '\n'; pos++, caret--);
					// move to previous line
					caret--;
		  		while (caret > 1 && content.charAt(caret-1) != '\n') 
		  			caret--;
		  		col = 0;
		  		// now to same column if we can	
		  		while (pos > 0 && content.charAt(caret+1) != '\n') {
		  			caret++;
		  			col++;
		  			pos--;
		  		}
		  	}
				continue;
			}
			
			// DOWN
			if (ch == 'B') {
				seq = false;
				
				if (parm == 0)
					parm = 1;
					
				while (parm-- > 0) {
					// locate position on this line
					for (pos = 0; caret > 1 && content.charAt(caret-1) != '\n'; pos++, caret--);
					// move to next line
		  		while (caret < content.length && content.charAt(caret) != '\n') 
		  			caret++;
		  		if (caret == content.length)
		  				content += '\n';
		  		caret++;
		  		col = 0;
		  		// now to same column if we can	
		  		while (pos > 0 && caret+1 < content.length && content.charAt(caret+1) != '\n') {
		  			caret++;
		  			col++;
		  			pos--;
		  		}
		  	}
				continue;
			}
			
			// RIGHT
			if (ch == 'C') {
				seq = false;
				
				if (parm == 0)
					parm = 1;
					
				while (parm-- > 0) {
					if (content.charAt(caret) != '\n')
						caret++;
				}
				continue;
			}
			
			// LEFT
			if (ch == 'D') {
				seq = false;
				
				if (parm == 0)
					parm = 1;
					
				while (parm-- > 0) {
					if (content.charAt(caret-1) != '\n')
						caret--;
				}
				continue;
			}
			
			// clear to end of line
			if (ch == 'K') {
				seq = false;
				
				for (pos = caret; pos < content.length && content.charAt(pos) != '\n'; pos++);
				if (pos > caret) 
					content = content.substring(0, caret) + content.substring(pos);
				continue;
			}
			
			// formatting
			if (ch == 'm') {
				seq = false;
				continue;
			}
			
			// clear screen
			if (ch == 'J')
			{
				if (parm == 2)
				{
					content = "";
					caret = 0;
				}

				seq = false;
				continue;
			}
			
		}
		
		// reset escape sequencing
		esc = false;
		seq = false;
		
		// carriage return moves caret to beginning of line
		if (chcode == 0x0D) {
			// move to beginning or position after last newline
  		while (caret > 1 && content.charAt(caret-1) != '\n') {
  			caret--;
  		}
  		col = 0;
  		continue;
  	}
		
		// newline moves to the next line and resets the overstrike mode
		if (chcode == 0x0A) {
			// move ahead to end or next newline
  		while (caret < content.length && content.charAt(caret) != '\n') {
  			caret++;
  		}
  		// if at end append newline
  		if (caret == content.length) {
  			content += "\n";
  		}
  		
  		//position after newline
			caret++;
			col = 0;
  		ins = 1;
  		continue;
  	}
  	
  	// backspace moves caret back
		if (chcode == 0x08) {
			if (caret > 0) {
				caret--;
				if (col > 0)
					col--;
			}
			continue;
		}

		// overstrike if within content - insert at end of line
		if (caret < content.length) {
			if (content.charAt(caret) == '\n')
				content = content.substr(0, caret) + cdata.charAt(i) + content.substr(caret);
			else
				content = content.substr(0, caret) + cdata.charAt(i) + content.substr(caret+1);
			caret++;
			col++;
			continue;
		}

		content += cdata.charAt(i);
		caret++;
		col++;
	}
	
	// truncate textarea to maintain response
	if (content.length > 250000) {
		var n = 50000;
		while (n < content.length && content.charAt(n++) != '\n');
		content = content.substr(n);
		caret = content.length;
	}
	
	screen.value = content;
	screen.scrollTop = screen.scrollHeight;
	
	// position the caret
	if (selAnchor < 0)
		screen.setSelectionRange(caret, caret);
	else if (caret >= selAnchor)
		screen.setSelectionRange(selAnchor, caret);
	else
		screen.setSelectionRange(caret, selAnchor, "backwards");
		
	status.innerHTML = (ins === 1 ? "Ins " : "Ovr ") + col.toString();
}

// **************************************************
// function to process keystrokes in the text area
function onKeypress(e) {
	e.preventDefault();
	if (connected && !transfer) {
		ch = e.which;
		removeSel();
		selAnchor = -1;	// any character ends selection display
		chan.sendJson({'Message': 'Console Stdin', 'Data': String.fromCharCode(ch)});
	}
}

function removeSel() {
	if (selAnchor != -1)
		return false;
	
	var screen = document.getElementById("screen");
	var start = screen.selectionStart;
	var end = screen.selectionEnd;
	var pent, pstart, pend, n, str;

	content = screen.value;
	if (start != end && ins == 1) {
		// determine beginning of entry line
		pent = caret;
		while (pent > 0 && content.charAt(pent-1) != '\n') {
			pent--;
		}
		
		// determine beginning of start highlighted line
		pstart = start;
		while (pstart > 0 && content.charAt(pstart-1) != '\n') {
			pstart--;
		}
		
		// determine beginning of end highlighted line
		pend = start;
		while (pend > 0 && content.charAt(pend-1) != '\n') {
			pend--;
		}
		
		// remove selection if on entry line
		if (pstart == pent && pend == pent) {
			// move to beginning of selection
			if (start < caret) {
				n = caret - start;
				str = "";
				while (n--)
					str += "\x1b[D";
				chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			}
			else if (caret < start) {
				n = start - caret;
				str = "";
				while (n--)
					str += "\x1b[C";
				chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			}
			
			// perfrom appropriate number of deletions
			screen.setSelectionRange(caret, caret);
			n = end - start;
			str = "";
			while (n--)
				str += "\x7f";
			chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			return true;
		}
	}
	return false;
}

// **************************************************
// function to process keystrokes in the text area
function onKeydown(e) {
	if (connected && !transfer) {
		var status = document.getElementById("status");
		
		ch = e.which;
		str = null;
		
		// pass recognized special keystrokes. Note that UP ARROW and DOWN ARROW
		//  reset the overstrike mode.
		if (ch === 8 || ch === 9 | ch == 0x1b)
			str = String.fromCharCode(ch);
			
		// Ctrl-A (drops/raises anchor)
		else if (e.ctrlKey && ch === 0x41) {
			str = String.fromCharCode(1);
			
			var screen = document.getElementById("screen");
			if (selAnchor != -1) {
				selAnchor = -1;
				screen.setSelectionRange(caret, caret);
			}
			else {
				selAnchor = screen.selectionStart;
			}
		}
		
		// Ctrl-F (resets selection)
		else if (e.ctrlKey && ch === 0x46) {
			str = String.fromCharCode(6);
			selAnchor = -1;
		}

		// Ctrl-R (resets selection)
		else if (e.ctrlKey && ch === 0x52) {
			str = String.fromCharCode(0x12);
			selAnchor = -1;
		}
		// Ctrl-C (resets selection)
		else if (e.ctrlKey && ch === 0x43) {
			str = String.fromCharCode(3);
			selAnchor = -1;
		}

		// Ctrl-X (resets selection)
		else if (e.ctrlKey && ch === 0x58) {
			str = String.fromCharCode(0x18);
			selAnchor = -1;
		}

		// Ctrl-V (resets selection)
		else if (e.ctrlKey && ch === 0x56) {
			str = String.fromCharCode(0x16);
			selAnchor = -1;
		}

		// Ctrl-Q	(resets selection)
		else if (e.ctrlKey && ch === 0x51) {
			str = String.fromCharCode(0x11);
			selAnchor = -1;
		}
		
		else if (ch === 0x21)	// PAGE UP
			str = "\x1b[5~";
		else if (ch === 0x22)	// PAGE DOWN
			str = "\x1b[6~";
		
		// Up Arrow (resets overstrike)
		else if (ch === 0x26)	{
			str = "\x1b[A";
			ins = 1;
		}
		
		// Down Arrow (resets overstrike)
		else if (ch === 0x28)	{
			str = "\x1b[B";
			ins = 1;
		}
		
		// Right Arrow
		else if (ch === 0x27)
			str = "\x1b[C";
			
		// Left Arrow	
		else if (ch === 0x25)
			str = "\x1b[D";
			
		// End
		else if (ch === 0x23)
			str = "\x1b[4~";
			
		// Home
		else if (ch === 0x24)
			str = "\x1b[1~";
			
		// Del (resets selection)
		else if (ch === 0x2e) {
			if (!removeSel() || ins == 0)
				str = "\x7f";
			selAnchor = -1;
		}
		
		// Ins (resets selection)
		else if (ch === 0x2d) {
			str = "\x1b[2~";
			ins ^= 1;
			selAnchor = -1;
		}
		
		if (str != null)
		{
			e.preventDefault();
			chan.sendJson({'Message': 'Console Stdin', 'Data': str});
		}

		status.innerHTML = (ins === 1 ? "Ins " : "Ovr ") + col.toString();
	}
}

// **************************************************
function onCut(screen) {
	var content = screen.value;
	setTimeout( function() {
		screen.value = content;
	}, 4)
}

// **************************************************
function onPaste(e) {
  var clipboardData, pastedData, str, n;

  e.stopPropagation();
  e.preventDefault();

  clipboardData = e.clipboardData || window.clipboardData;
  pastedData = clipboardData.getData('Text');
  
  // Cannot paste multi-line data
  for (n=0; n<pastedData.length; n++) 
  	if (pastedData.charCodeAt(n) < 0x20 || pastedData.charCodeAt(n) >= 0x7f)
  		break;
  		
  str = pastedData.substring(0, n);
	chan.sendJson({'Message': 'Console Stdin', 'Data': str});
}

// **************************************************
function onPosition(screen) {
	var start = screen.selectionStart;
	var end = screen.selectionEnd;
	var pent, pclick, n, str;
	
	content = screen.value;
	if (start == end) {
		// determine beginning of entry line
		pent = caret;
		while (pent > 0 && content.charAt(pent-1) != '\n') {
			pent--;
		}
		
		// determine beginning of line clicked
		pclick = start;
		while (pclick > 0 && content.charAt(pclick-1) != '\n') {
			pclick--;
		}
		
		// prevent movement if not the same line
		if (pclick == pent) {
			screen.setSelectionRange(caret, caret);
			if (start < caret) {
				n = caret - start;
				str = "";
				while (n--)
					str += "\x1b[D";
				chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			}
			else if (caret < start) {
				n = start - caret;
				str = "";
				while (n--)
					str += "\x1b[C";
				chan.sendJson({'Message': 'Console Stdin', 'Data': str});
			}
		}
	}
}
	
// **************************************************
// session
function onOpen() {
	chan.sendJson({'Message': 'Console Open'});
	document.getElementById('close').className = "button";
	document.getElementById('open').className = "button hide";
	document.getElementById('screen').focus();
	connected = true;
	transfer = false;
	document.getElementById('screen').className = "screen";
	
	// move past scren data
	var screen = document.getElementById("screen");
	screen.value += "\n";
	caret = screen.value.length;
}

function onClose() {
	chan.sendJson({'Message': 'Console Close'});
	document.getElementById('close').className = "button hide";
	document.getElementById('open').className = "button";
	document.getElementById('open').focus();
	document.getElementById('status').innerHTML = "No Active Session";
	connected = false;
	transfer = false;
	document.getElementById('screen').className = "screen inactive";
	}
	
// **************************************************
// checks if console should be reopened
chan.onopen = function() {
	if (connected)
		setTimeout(onOpen, 1000);
}

function onClear() {
	var screen = document.getElementById("screen");
	var status = document.getElementById("status");
	
	screen.focus();			
	screen.value = "";
	caret = 0;
	col = 0;
	
	if (connected && !transfer)
		status.innerHTML = (ins === 1 ? "Ins " : "Ovr ") + col.toString();
	else
		status.innerHTML = "No Active Session";		
};
	
function scnpaste(e) {
	if (e.dataTransfer) {
		var data = e.dataTransfer.getData("Text");
		if (data) {
			removeSel();
			chan.sendJson({'Message': 'Console Stdin', 'Data': data});
			e.stopPropagation();
		  e.preventDefault();
		}
	}
}
