package com.integ.mqtt;

import com.integpg.janoslib.io.AppLog;
import com.integpg.janoslib.utils.FileUtils;
import com.integpg.janoslib.utils.JsonUtils;
import com.integpg.janoslib.utils.StringUtils;
import java.io.File;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Json;

public class Config {

    private static final File CONFIG_FILE = new File(MqttMain.FLASH_MQTT.getPath() + "config.json");
    private static Json _json = null;

    private static final ArrayList<ConfigurationListener> _configUpdatedEventListeners = new ArrayList<>();

    private static final ArrayList<String> _deviceGroups = new ArrayList<>();
    private static Hashtable<String, Integer> _outputMasks = new Hashtable<>();



    public static void addConfigurationUpdatedListener(ConfigurationListener listener) {
        _configUpdatedEventListeners.add(listener);
    }



    public static void load() {
        long start = System.currentTimeMillis();
        try {
            if (CONFIG_FILE.exists()) {
                synchronized (CONFIG_FILE) {
                    String fileContents = FileUtils.readAllText(CONFIG_FILE.getPath());
                    _json = new Json(fileContents);
                    loadDeviceGroups();
                    loadOutputMasks();

                    alertListeners();
                }
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        long elapsed = System.currentTimeMillis() - start;
        AppLog.info("Loaded configuration in " + (elapsed / 1000.0) + " seconds");
    }



    private static void alertListeners() {
        for (ConfigurationListener listener : _configUpdatedEventListeners) {
            listener.ConfigurationUpdated();
        }
    }



    public static String getBrokerHost() {
        if (null != _json && _json.containsKey("Broker")) {
            Json broker = (Json) _json.get("Broker");
            if (null != broker) {
                String host = broker.getString("Host");
                if (!"".equals(host)) {
                    return host;
                }
            }
        }
        return "";
    }



    public static int getBrokerPort() {
        if (null != _json && _json.containsKey("Broker")) {
            Json broker = (Json) _json.get("Broker");
            return JsonUtils.getInt(broker.get("Port"), 1883);
        }
        return 1883;
    }



    public static boolean getEncrypted() {
        if (null != _json && _json.containsKey("Broker")) {
            Json broker = (Json) _json.get("Broker");
            return JsonUtils.getBoolean(broker.get("Encrypted"), false);
        }
        return false;
    }



    public static String getLastWillTopic() {
        if (null != _json && _json.containsKey("Broker")) {
            Json broker = (Json) _json.get("Broker");
            if (null != broker && broker.containsKey("LastWill")) {
                Json lastWill = (Json) broker.get("LastWill");
                if (null != lastWill) {
                    String host = lastWill.getString("Topic");
                    if (!"".equals(host)) {
                        return host;
                    }
                }
            }
        }
        return null;
    }



    public static String getLastWillMessage() {
        if (null != _json && _json.containsKey("Broker")) {
            Json broker = (Json) _json.get("Broker");
            if (null != broker && broker.containsKey("LastWill")) {
                Json lastWill = (Json) broker.get("LastWill");
                if (null != lastWill) {
                    String host = lastWill.getString("Message");
                    if (!"".equals(host)) {
                        return host;
                    }
                }
            }
        }
        return null;
    }



    private static void loadDeviceGroups() {
        String[] groups = new String[0];
        if (null != _json) {
            String deviceGroupsString = _json.getString("DeviceGroups");
            if (null != deviceGroupsString) {
                groups = StringUtils.split(deviceGroupsString, ",");
                for (int i = 0; i < groups.length; i++) {
                    groups[i] = groups[i].trim();
                    _deviceGroups.add(groups[i]);
                }
            }
        }
    }



    public static String[] getDeviceGroups() {
        return (String[]) _deviceGroups.toArray();
    }



    private static void loadOutputMasks() {
        if (null != _json) {
            Json[] outputGroupsJsonArray = (Json[]) _json.get("OutputGroups");
            if (null != outputGroupsJsonArray) {
                _outputMasks.clear();

                for (int i = 0; i < outputGroupsJsonArray.length; i++) {
                    Json outputGroupJson = outputGroupsJsonArray[i];
                    System.out.println("outputGroupJson: " + outputGroupJson);
                    String groupName = outputGroupJson.getString("GroupName");
                    String maskString = outputGroupJson.getString("Mask");
                    int mask = 0;
                    if (maskString.startsWith("0x")) {
                        mask = Integer.valueOf(maskString.substring(2), 16);
                    } else {
                        mask = Integer.valueOf(maskString);
                    }
                    _outputMasks.put(groupName, mask);
                }
            }
        }
    }



    public static int getOutputMask(String name) {
        int mask = 0;
        if (_outputMasks.containsKey(name)) {
            mask = _outputMasks.get(name);
        }
        return mask;
    }

}

