package com.integ.mqtt;

import com.integ.mqtt.messagehandlers.DisconnectEventHandler;
import com.integ.mqtt.messagehandlers.PublishEventHandler;
import com.integ.mqtt.messagehandlers.ConnectEventHandler;
import com.integ.mqtt.messagehandlers.ConfigUpdatedEventHandler;
import com.integ.mqtt.messagehandlers.GetStatusEventHandler;
import com.integ.messagepumpengine.MessagePumpAppHandler;
import com.integ.registry.RegistryUpdateHandler;
import com.integ.registry.RegistryListener;
import com.integ.iolog.DigitalInputIoLogMonitor;
import com.integ.iolog.DigitalOutputIoLogMonitor;
import com.integ.messagepumpengine.MessagePumpEngine;
import com.integpg.janoslib.externalio.TypeFE;
import com.integpg.janoslib.io.AppLog;
import com.integpg.janoslib.logger.FileLog;
import com.integpg.janoslib.logger.Log;
import com.integpg.janoslib.logger.LogOptions;
import com.integpg.janoslib.messaging.MessageLoopTypes;
import com.integpg.janoslib.mqtt.*;
import com.integpg.janoslib.net.MessageReceivedEvent;
import com.integpg.janoslib.net.TcpConnectionListener;
import com.integpg.janoslib.system.Application;
import com.integpg.janoslib.system.UnitConfig;
import com.integpg.janoslib.text.QuickDateFormat;
import com.integpg.system.JANOS;
import com.integpg.system.Watchdog;
import java.io.File;
import java.util.ArrayList;
import java.util.EventObject;

/**
 * https://github.com/mqtt/mqtt.github.io/wiki/public_brokers
 *
 * @author kcloutier
 */
public class MqttMain implements MqttClientListener, SubscriptionListener, UnhandledSubscriptionListener, RegistryListener, ConfigurationListener {

    private static final Log UnhandledSubscriptionsLog = FileLog.getLog(new LogOptions(Application.getAppName() + "_unhandledsubscriptions.log"));
    public final static File FLASH_MQTT = new File("/flash/mqtt/");

    public static final String DEVICE_STATUS_TOPIC_PREFIX = "jnior/" + JANOS.getSerialNumber() + "/status/";
    public static final String DEVICE_SET_TOPIC_PREFIX = "jnior/" + JANOS.getSerialNumber() + "/set/";
    public static final QuickDateFormat CONNECTION_DATE_FORMAT = new QuickDateFormat("MM/dd/yyyy HH:mm:ss");

    private static MqttMain MAIN;

    public MqttClient MqttClient;

    private final ArrayList<FourTwentyInputs> _fourTwentyInputs = new ArrayList<>();
    private final ArrayList<FourTwentyOutputs> _fourTwentyOutputs = new ArrayList<>();

    private RelayOutputs _relayOutputs;
    private DigitalInputs _digitalInputs;



    public static void main(String[] args) throws InterruptedException {
        Application.init(new AssemblyInfo());

        MAIN = new MqttMain();
        MAIN.init();

        Application.enableApplicationWatchdog(Watchdog.WDT_REBOOT);

        MAIN.loop();
    }



    public static MqttMain getInstance() {
        return MAIN;
    }



    public void init() throws InterruptedException {
        AppLog.info(UnitConfig.MODEL + ": " + UnitConfig.getInputCount() + " ins, " + UnitConfig.getOutputCount() + " outs");

        if (!FLASH_MQTT.exists()) FLASH_MQTT.mkdir();

        Config.addConfigurationUpdatedListener(this);
        Config.load();

        MessagePumpEngine.start();

        MqttClient = new MqttClient();
        reportConnectionStatus();

        _digitalInputs = new DigitalInputs(MqttClient);
        DigitalInputIoLogMonitor inputMonitor = new DigitalInputIoLogMonitor();
        inputMonitor.addIoLogEventListener(_digitalInputs);
        inputMonitor.start();

        _relayOutputs = new RelayOutputs(MqttClient);
        DigitalOutputIoLogMonitor outputMonitor = new DigitalOutputIoLogMonitor();
        outputMonitor.addIoLogEventListener(_relayOutputs);
        outputMonitor.start();

        TypeFE[] typeFEDevices = TypeFE.getDeviceArray();
        System.out.println(String.format("%d four twenty devices", typeFEDevices.length));
        for (TypeFE typeFEDevice : typeFEDevices) {
            System.out.println(String.format("four twenty device: %s", typeFEDevice.AddressString));
            _fourTwentyInputs.add(new FourTwentyInputs(MqttClient, typeFEDevice));
            _fourTwentyOutputs.add(new FourTwentyOutputs(MqttClient, typeFEDevice));
        }

        RegistryUpdateHandler registryUpdateHandler = new RegistryUpdateHandler();
        registryUpdateHandler.addRegistryListener(this);
        registryUpdateHandler.addRegistryListener(_digitalInputs);
        registryUpdateHandler.addRegistryListener(_relayOutputs);

        // listen to the message pump
        MessagePumpEngine.addListener(registryUpdateHandler);

        MessagePumpAppHandler mqttAppListener = new MessagePumpAppHandler(MessageLoopTypes.SM_MQTT)
                .addCommandListener("get-status", new GetStatusEventHandler())
                .addCommandListener("config-updated", new ConfigUpdatedEventHandler())
                .addCommandListener("connect", new ConnectEventHandler())
                .addCommandListener("disconnect", new DisconnectEventHandler())
                .addCommandListener("publish", new PublishEventHandler());
        MessagePumpEngine.addListener(mqttAppListener);

        MqttClient.addConnectionListener(this);
        MqttClient.addSubscriptionListener(this);
        MqttClient.setUnhandledSubscriptionListener(this);

        // keep trying to connect until we are successful
        while (true) {
            try {
                MqttClient.connect();
                if (MqttClient.isConnected()) {
                    break;
                }
            } catch (Exception ex) {
                // we didnt connect
                AppLog.error("Unable to connect", ex);
            }

            Thread.sleep(10000);
        }


        try {
            MqttClient.publish(DEVICE_STATUS_TOPIC_PREFIX + "hostname", UnitConfig.getHostname().getBytes(), true);

            SubscribeOptions subscribeOptions = new SubscribeOptions()
                    .addTopic(new SubscribeTopic(DEVICE_SET_TOPIC_PREFIX + "#", QualityOfService.AT_LEAST_ONCE)); //set-hostname", 0))
//                    .addTopic(new SubscribeTopic(DEVICE_TOPIC_PREFIX + "digital/outputs/+/set-state", 0));
            MqttClient.subscribe(subscribeOptions);

            updateDeviceNameSubscription();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }



    public void loop() {
        while (true) {
            try {
                Thread.sleep(30000);
                MqttClient.sendPing();

                for (FourTwentyInputs fourTwentyInputs : _fourTwentyInputs) {
                    fourTwentyInputs.report();
                }

                for (FourTwentyOutputs fourTwentyOutputs : _fourTwentyOutputs) {
                    fourTwentyOutputs.report();
                }
            } catch (InterruptedException ex) {
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
    }



    @Override
    public boolean subscriptionUpdate(String topic, byte[] data) {
        if (topic.endsWith("set-hostname")) {
            String message = new String(data);
            System.out.println("setting hostname to " + message);
            JANOS.setRegistryString("IpConfig/Hostname", message);
            return true;
        }
        return false;
    }



    @Override
    public void connectionEstablished(EventObject evt) {
        AppLog.info("Connected to mqtt broker: " + MqttClient.getConnectionInfo());

        // update the connection established time registry key
        JANOS.setRegistryString("AppData/MQTT/ConnectionEstablishedTime",
                CONNECTION_DATE_FORMAT.format(System.currentTimeMillis()));

        reportConnectionStatus();
    }



    @Override
    public void connectionAknowledged(ConnectionAcknowledgedEventObject evt) {
        _digitalInputs.reportCurrentStates();
        _relayOutputs.reportCurrentStates();
    }



    @Override
    public void connectionClosed(EventObject evt) {
        System.out.println("connection closed");
        // update the connection lost time registry key
        JANOS.setRegistryString("AppData/MQTT/ConnectionLostTime",
                CONNECTION_DATE_FORMAT.format(System.currentTimeMillis()));

        reportConnectionStatus();
    }



    @Override
    public void messageReceived(MessageReceivedEvent evt) {
        // do nothing
    }



    @Override
    public void registryKeyUpdated(String registryKeyName) {
        // is it the hostname that changed?
        if ("ipconfig/hostname".equalsIgnoreCase(registryKeyName)) {
            try {
                MqttClient.publish(DEVICE_STATUS_TOPIC_PREFIX + "hostname", UnitConfig.getHostname().getBytes(), true);
            } catch (Exception ex) {
                AppLog.error("error handling registry key update for " + registryKeyName + " in MqttMain", ex);
            }
        }
    }



    @Override
    public void connectionAttempt(EventObject evt) {
        String host = Config.getBrokerHost();
        int port = Config.getBrokerPort();
        boolean secure = Config.getEncrypted();
        System.out.println(String.format("host: %s, port: %d, secure: %s", host, port, String.valueOf(secure)));

        if (!"".equalsIgnoreCase(host)) {
            MqttClient.setHost(host)
                    .setPort(port)
                    .setSecure(secure);

            String lastWillTopic = Config.getLastWillTopic();
            if (null == lastWillTopic || "".equals(lastWillTopic)) {
                lastWillTopic = "jnior/" + JANOS.getSerialNumber() + "/LastWill";
            }

            String lastWillMessage = Config.getLastWillMessage();
            if (null == lastWillMessage || "".equals(lastWillMessage)) {
                lastWillMessage = "Has disconnected";
            }

            ConnectOptions connectOptions = new ConnectOptions("jr" + JANOS.getSerialNumber());
            connectOptions.setCleanConnection(true)
                    .setKeepAliveSeconds(900)
                    .setWillTopic(lastWillTopic)
                    .setWillMessage(lastWillMessage)
                    .setWillQos(0)
                    .setWillRetain(Boolean.FALSE);

            MqttClient.setConnectOptions(connectOptions);
        }
    }



    @Override
    public void ConfigurationUpdated() {
        try {
            if (null != MqttClient
                    && !Config.getBrokerHost().equalsIgnoreCase(MqttClient.getHost())) {
                MqttClient.close();
            }

            updateDeviceNameSubscription();
        } catch (Exception ex) {
            AppLog.error("error in configuration listener", ex);
        }
    }



    @Override
    public void unhandledSubscriptionUpdate(String topic, byte[] data) {
        UnhandledSubscriptionsLog.info("topic " + topic + " has not been handled");

        // since no one handled this message let's push a message to the message pump so that other 
        // applications can react to the message.  
        String message = new String(data);
        MqttSystemMessage msg = new MqttSystemMessage("unhandled-topic")
                .put("Topic", topic)
                .put("Payload", message);

        MessagePumpEngine.postMessage(msg.getSystemMsg());
    }



    private void updateDeviceNameSubscription() throws Exception {
        if (null != MqttClient) {
            SubscribeOptions subscribeOptions = new SubscribeOptions();
            String[] mqttGroups = Config.getDeviceGroups();
            for (String mqttGroup : mqttGroups) {
                subscribeOptions.addTopic(new SubscribeTopic("jnior/" + mqttGroup.toLowerCase() + "/#", QualityOfService.AT_LEAST_ONCE));
                MqttClient.subscribe(subscribeOptions);
            }
        }
    }



    public void reportConnectionStatus() {
        MqttSystemMessage msg = new MqttSystemMessage("current-status")
                .put("ConnectionStatus", MqttClient.isConnected())
                .put("ConnectionInfo", MqttClient.getConnectionInfo());

        System.out.println(String.format("ConnectionStatus: %s", msg.getJson().toString()));
        MessagePumpEngine.postMessage(msg.getSystemMsg());
    }



    @Override
    protected void finalize() {
        AppLog.info(this.getClass().getName() + " finalized");
    }
}

