package com.integ.iolog;

import com.integpg.janoslib.logger.FileLog;
import com.integpg.janoslib.logger.Log;
import com.integpg.janoslib.logger.LogOptions;
import com.integpg.janoslib.system.Application;
import com.integpg.janoslib.text.QuickDateFormat;
import com.integpg.system.IoEvent;
import com.integpg.system.Iolog;
import java.util.ArrayList;

public abstract class IoLogMonitor implements Runnable {

    protected static final Log LOG = FileLog.getLog(new LogOptions("temp/" + Application.getAppName() + "_iologmonitor.log", 64 * 1024));
    protected static final QuickDateFormat QUICK_DATE_FORMAT = new QuickDateFormat("MM/dd/yyyy HH:mm:ss.fff");

    protected final Iolog _iolog = new Iolog();

    private Thread _thd;
    private ArrayList<IoLogListener> _ioLogEventListeners = new ArrayList<>();
    private long _refreshTimestamp;



    protected IoLogMonitor() {
    }



    public void start() {
        if (null == _thd) {
            _thd = new Thread((Runnable) this);
            _thd.setName(this.getClass().getName().substring(this.getClass().getName().lastIndexOf("/") + 1));
            _thd.start();
        }
    }



    public void addIoLogEventListener(IoLogListener ioLogEventListener) {
        _ioLogEventListeners.add(ioLogEventListener);
    }



    public void setRefreshTimestamp(long refreshTimestamp) {
        _refreshTimestamp = refreshTimestamp;
    }



    protected abstract IoEvent[] getIoEvents();



    @Override
    public void run() {
        while (true) {
            try {
                // refresh the iolog object so that we only get new events since the last time we queried it
                _iolog.refresh(_refreshTimestamp);
                // get the events for the inputs only
                IoEvent[] ioEvents = getIoEvents();

                // make sure we were returned events
                if (ioEvents.length > 0) {
                    if (1 < ioEvents.length) {
                        LOG.info(ioEvents.length + " events returned since " + QUICK_DATE_FORMAT.format(_refreshTimestamp));
                    }

                    int errorCount = 0;
                    // go through each event and alert listeners
                    for (int index = ioEvents.length - 1; index >= 0; index--) {
                        IoEvent ioEvent = ioEvents[index];

                        if (ioEvent.timestamp < _refreshTimestamp) {
                            errorCount++;
                            continue;
                        }

//                        System.out.println("ioEvent[" + index + "]:"
//                                + " time: " + QUICK_DATE_FORMAT.format(ioEvent.timestamp)
//                                + " changed input mask: " + Long.toHexString(ioEvent.mask)
//                                + " input states mask: " + Long.toHexString(ioEvent.states));

                        alertListenersOfIoEvent(ioEvent);
                    }

                    if (0 != errorCount) {
                        LOG.info("iolog returned " + errorCount + " events earlier than the requested time of "
                                + QUICK_DATE_FORMAT.format(_refreshTimestamp));
                    }

                    // update our lastConsumedTimestamp with the date of the most recent event
                    _refreshTimestamp = ioEvents[0].timestamp;
//                    System.out.println("_refreshTimestamp: " + QUICK_DATE_FORMAT.format(_refreshTimestamp));

                    alertListenersOfEventsProcessed();
                }

                Thread.sleep(100);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
    }



    private void alertListenersOfIoEvent(IoEvent ioEvent) {
        try {
            for (IoLogListener ioLogListener : _ioLogEventListeners) {
                if (null != ioLogListener) {
                    long start = System.currentTimeMillis();
                    alertIoListener(ioLogListener, ioEvent);
                    long elapsed = System.currentTimeMillis() - start;

                    if (100 < elapsed) {
                        System.out.println("process event took " + elapsed);
                    }
                }
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }



    public abstract void alertIoListener(IoLogListener ioLogListener, IoEvent ioEvent);



    private void alertListenersOfEventsProcessed() {
        try {
            for (IoLogListener ioLogListener : _ioLogEventListeners) {
                if (null != ioLogListener) {
                    long start = System.currentTimeMillis();
                    ioLogListener.onIoEventsProcessed();
                    long elapsed = System.currentTimeMillis() - start;

                    if (100 < elapsed) {
                        System.out.println("processing events took " + elapsed);
                    }
                }
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

}

