package com.integ.mqtt;

import com.integ.registry.RegistryListener;
import com.integ.iolog.DigitalIoEvent;
import com.integ.iolog.DigitalOutputEvent;
import com.integ.iolog.IoLogListener;
import com.integpg.janoslib.io.AppLog;
import com.integpg.janoslib.logger.FileLog;
import com.integpg.janoslib.logger.Log;
import com.integpg.janoslib.logger.LogOptions;
import com.integpg.janoslib.mqtt.MqttClient;
import com.integpg.janoslib.mqtt.SubscriptionListener;
import com.integpg.janoslib.system.Application;
import com.integpg.janoslib.system.UnitConfig;
import com.integpg.janoslib.utils.StringUtils;
import com.integpg.system.JANOS;
import java.io.IOException;
import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class RelayOutputs implements IoLogListener, SubscriptionListener, RegistryListener {

    protected static final Log REGEX_LOG = FileLog.getLog(new LogOptions("temp/" + Application.getAppName() + "_regexpattern.log", 128 * 1024));
    protected static final Log SUBSCRIPTION_LOG = FileLog.getLog(new LogOptions("temp/" + Application.getAppName() + "_subscriptionupdates.log", 128 * 1024));

    // this pattern is evaluated when an mqtt topic is received.  it is used to determine is the topic is 
    // associated with the relay outputs
    private static final Pattern OUTPUTS_PATTERN; // = Pattern.compile("jnior/([\\d\\w]+)/digital/outputs/(\\d+)/.*");

    // this pattern is evaluated when the mqtt topic is received that controls the relays
    private static final Pattern STATE_PATTERN = Pattern.compile(".*/state");

    // this pattern is evaluated when there is a change to a registry key.  it is used to determine if the key 
    // is associated with the relay outputs.  these keys are expected to change with there is a change to the 
    // $hourmeter key.
    private static final Pattern REGISTRY_OUTPUTS_PATTERN = Pattern.compile("io/outputs/rout(\\d)/.*", Pattern.CASE_INSENSITIVE);

    private final MqttClient _mqttClient;

    private final ArrayList<String> _mqttGroups = new ArrayList<>();



    static {
        String groupNameString = "";
        String groupMemberString = String.valueOf(JANOS.getSerialNumber());
        String[] mqttGroups = Config.getDeviceGroups();
        for (String mqttGroup : mqttGroups) {
            if (0 != groupNameString.length()) groupNameString += "-";
            groupNameString += mqttGroup.toLowerCase();
            groupMemberString += "|";
            groupMemberString += groupNameString;
        }

//        OUTPUTS_PATTERN = Pattern.compile("jnior/(" + groupMemberString + ")/digital/outputs/(\\d+)/.*");
        OUTPUTS_PATTERN = Pattern.compile("jnior/([\\d\\w-]+)/set/digital/outputs/([\\d\\w-]+)/.*");
        REGEX_LOG.info("groupMemberString: " + groupMemberString);
    }



    public RelayOutputs(MqttClient mqttClient) {
        _mqttClient = mqttClient;
        _mqttClient.addSubscriptionListener(this);
    }



    public void reportCurrentStates() {

        // report the current states
        int outputStates = JANOS.getOutputStates();
        for (int i = 1; i <= UnitConfig.getOutputCount(); i++) {

            try {
                boolean state = (1 == (outputStates & 1));
                _mqttClient.publish(MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/outputs/" + i + "/state",
                        String.valueOf(state).getBytes(), true);

                long usageMeter = JANOS.getUsageMeter((i - 1) + UnitConfig.getInputCount());
                _mqttClient.publish(MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/outputs/" + i + "/usagemeter",
                        String.valueOf(usageMeter / 3600000.0).getBytes(), true);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
    }



    @Override
    public void onIoEvent(DigitalIoEvent ioEvent) {
        try {
            System.out.println("output changed");
            DigitalOutputEvent digitalOutputEvent = (DigitalOutputEvent) ioEvent;

            _mqttClient.publish(MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/outputs/" + digitalOutputEvent.Channel + "/state",
                    String.valueOf(digitalOutputEvent.State).getBytes(), true);
            _mqttClient.publish(MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/outputs/" + digitalOutputEvent.Channel + "/usagemeter",
                    String.valueOf(digitalOutputEvent.UsageMeter / 3600000.0).getBytes(), true);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }



    @Override
    public void onIoEventsProcessed() {
        // do nothing
    }



    private static boolean isNumeric(String s) {
        boolean isNumeric = true;
        for (int j = 0; isNumeric && j < s.length(); j++) {
            isNumeric &= Character.isDigit(s.charAt(j));
        }
        return isNumeric;
    }



    @Override
    public boolean subscriptionUpdate(String topic, byte[] data) {
        String topicValue = new String(data);
        SUBSCRIPTION_LOG.info(topic + ": " + topicValue);

        REGEX_LOG.info("match " + OUTPUTS_PATTERN.pattern() + ", " + topic);
        Matcher matcher = OUTPUTS_PATTERN.matcher(topic);
        boolean found = matcher.find();
        REGEX_LOG.info(OUTPUTS_PATTERN.pattern() + " matched: " + found);
        if (found) {
//                    System.out.println("group count: " + matcher.groupCount());
//                    for (int i = 0; i < matcher.groupCount(); i++) {
//                        System.out.println("group: " + matcher.group(i));
//                    }
            String groupString = matcher.group(1);
            System.out.println(groupString + " indexOf: " + _mqttGroups.indexOf(groupString));

            int channelMask = 0;
            String outputString = matcher.group(2);
            if (isNumeric(outputString)) {
                int channel = Integer.parseInt(outputString);
                channelMask = (1 << (channel - 1));
            } else {
                channelMask = Config.getOutputMask(outputString);
            }
            System.out.println(String.format("channelMask: 0x%d", channelMask));
            int statesMask = channelMask;

            REGEX_LOG.info("match " + STATE_PATTERN.pattern() + ", " + topic);
            matcher = STATE_PATTERN.matcher(topic);
            found = matcher.find();
            REGEX_LOG.info(STATE_PATTERN.pattern() + " matched: " + found);
            if (found) {
                String[] parts = StringUtils.split(topicValue, " ");
                System.out.println("parts.length: " + parts.length);

                boolean newState = Boolean.valueOf(parts[0]);
                if (!newState) statesMask ^= 0xfff;
                statesMask &= channelMask;
                System.out.println(String.format("set channelMask to 0x%d", statesMask));

                if (1 < parts.length) {
                    try {
                        double duration = Double.valueOf(parts[1].trim());
                        JANOS.setOutputPulsed(statesMask, channelMask, (int) (duration * 1000));
                        return true;
                    } catch (IOException ex) {
                        ex.printStackTrace();
                    }
                } else {
                    try {
                        System.out.println(String.format("set 0x%d to 0x%d", channelMask, statesMask));
                        JANOS.setOutputStates(statesMask, channelMask);
                        return true;
                    } catch (IOException ex) {
                        ex.printStackTrace();
                    }
                }
            }
        }

        return false;
    }



    @Override
    public void registryKeyUpdated(String registryKeyName) {
        try {
            REGEX_LOG.info("match " + REGISTRY_OUTPUTS_PATTERN.pattern() + ", " + registryKeyName);
            Matcher matcher = REGISTRY_OUTPUTS_PATTERN.matcher(registryKeyName);
            boolean found = matcher.find();
            REGEX_LOG.info(REGISTRY_OUTPUTS_PATTERN.pattern() + " matched: " + found);
            if (found) {
                try {
                    int channel = Integer.parseInt(matcher.group(1));
                    String regValue = JANOS.getRegistryString(registryKeyName, "");

                    if (registryKeyName.endsWith("$HourMeter")) {
                        _mqttClient.publish(MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/outputs/" + channel + "/usagemeter", regValue.getBytes());
                    }
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            }
        } catch (Exception ex) {
            AppLog.error("error handling registry key update for " + registryKeyName + " in digital outputs", ex);
        }
    }
}

