package com.integ.mqtt;

import com.integ.registry.RegistryListener;
import com.integ.iolog.DigitalInputEvent;
import com.integ.iolog.DigitalIoEvent;
import com.integ.iolog.IoLogListener;
import com.integpg.janoslib.io.AppLog;
import com.integpg.janoslib.logger.FileLog;
import com.integpg.janoslib.logger.Log;
import com.integpg.janoslib.logger.LogOptions;
import com.integpg.janoslib.mqtt.MqttClient;
import com.integpg.janoslib.system.Application;
import com.integpg.janoslib.system.UnitConfig;
import com.integpg.system.JANOS;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DigitalInputs implements IoLogListener, RegistryListener {

    protected static final Log REGEX_LOG
            = FileLog.getLog(new LogOptions("temp/" + Application.getAppName() + "_regexpattern.log", 128 * 1024));

    // this pattern is evaluated when there is a change to a registry key.  it is used to determine if the key 
    // is associated with the digital inputs.  these keys are expected to change with there is a change to the 
    // $hourmeter key.
    private static final Pattern REGISTRY_INPUTS_PATTERN = Pattern.compile("io/inputs/din(\\d+)/.*", Pattern.CASE_INSENSITIVE);

    private final MqttClient _mqttClient;



    public DigitalInputs(MqttClient mqttClient) {
        _mqttClient = mqttClient;
    }



    public void reportCurrentStates() {

        // report the current states
        int inputStates = JANOS.getInputStates();
        for (int i = 1; i <= UnitConfig.getInputCount(); i++) {

            try {
                boolean state = (1 == (inputStates & 1));
                _mqttClient.publish(MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/inputs/" + i + "/state",
                        String.valueOf(state).getBytes(), true);

                long usageMeter = JANOS.getUsageMeter((i - 1));
                _mqttClient.publish(MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/inputs/" + i + "/usagemeter",
                        String.valueOf(usageMeter / 3600000.0).getBytes(), true);

                int counter = JANOS.getInputCounter(i - 1);
                _mqttClient.publish(
                        MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/inputs/" + i + "/counter",
                        String.valueOf(counter).getBytes(), true);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
    }



    @Override
    public void onIoEvent(DigitalIoEvent ioEvent) {
        try {
            System.out.println("input changed");
            DigitalInputEvent digitalInputEvent = (DigitalInputEvent) ioEvent;

            if (null != _mqttClient) {
                _mqttClient.publish(
                        MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/inputs/" + digitalInputEvent.Channel + "/state",
                        String.valueOf(digitalInputEvent.State).getBytes(), true);
                _mqttClient.publish(
                        MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/inputs/" + digitalInputEvent.Channel + "/usagemeter",
                        String.valueOf(digitalInputEvent.UsageMeter / 3600000.0).getBytes(), true);

                // only if the input went high
                if (digitalInputEvent.State) {
                    _mqttClient.publish(
                            MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/inputs/" + digitalInputEvent.Channel + "/counter",
                            String.valueOf(digitalInputEvent.Counter).getBytes(), true);
                }
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }



    @Override
    public void onIoEventsProcessed() {
        // do nothing
    }



    @Override
    public void registryKeyUpdated(String registryKeyName) {
        try {
            REGEX_LOG.info("match " + REGISTRY_INPUTS_PATTERN.pattern() + ", " + registryKeyName);
            Matcher matcher = REGISTRY_INPUTS_PATTERN.matcher(registryKeyName);
            boolean found = matcher.find();
            REGEX_LOG.info(REGISTRY_INPUTS_PATTERN.pattern() + " matched: " + found);
            if (found) {
                try {
                    int channel = Integer.parseInt(matcher.group(1));
                    String regValue = JANOS.getRegistryString(registryKeyName, "");

                    if (registryKeyName.endsWith("$HourMeter")) {
                        _mqttClient.publish(
                                MqttMain.DEVICE_STATUS_TOPIC_PREFIX + "digital/inputs/" + channel + "/usagemeter",
                                regValue.getBytes());
                    }
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            }
        } catch (Exception ex) {
            AppLog.error("error handling registry key update for " + registryKeyName + " in digital inputs", ex);
        }
    }

}

