package com.integ.messagepumpengine;

import com.integpg.janoslib.io.AppLog;
import com.integpg.janoslib.logger.FileLog;
import com.integpg.janoslib.logger.Log;
import com.integpg.janoslib.logger.LogOptions;
import com.integpg.janoslib.messaging.MessageLoopTypes;
import com.integpg.janoslib.system.Application;
import com.integpg.system.MessagePump;
import com.integpg.system.SystemMsg;
import java.util.Vector;

public class MessagePumpEngine implements Runnable {

    public static final Log LOG = FileLog.getLog(new LogOptions("temp/" + Application.getAppName() + "_messagepumpengine.log", 64 * 1024));

    private static final MessagePumpEngine THIS = new MessagePumpEngine();
    private static final MessagePump MESSAGE_PUMP = new MessagePump();
    private static final Vector<MessagePumpListener> LISTENERS = new Vector<>();

    private static Thread _thread;



    /**
     * Singleton constructor
     */
    private MessagePumpEngine() {
    }



    /**
     * adds a listener that will be alerted of all received messages. The listener will be responsible for determining
     * if the message has meaning to them
     *
     * @param listener
     */
    public static void addListener(MessagePumpListener listener) {
        synchronized (LISTENERS) {
            LISTENERS.addElement(listener);
        }
    }



    /**
     * starts our message pump engine.
     */
    public static void start() {
        if (null == _thread) {
            _thread = new Thread(THIS);
            _thread.setName("message-pump-engine");
            _thread.setDaemon(true);
            _thread.start();
        }
    }



    @Override
    public void run() {
        System.out.println("open the message pump and start monitoring.\r\n");
        MESSAGE_PUMP.open();

        for (;;) {
            // read all messages from the message pump
            SystemMsg systemMsg = MESSAGE_PUMP.getMessage();

            // we must repost as fast as we can
            MESSAGE_PUMP.postMessage(systemMsg);

            String contents = "";
            if (0 < systemMsg.msg.length) {
                contents = ": " + new String(systemMsg.msg).trim();
            }
            String s = String.format("    (%d) %s%s",
                    systemMsg.type,
                    MessageLoopTypes.getMessageDescription(systemMsg.type),
                    contents);
            LOG.info(s);

            // notify all of our listeners
            synchronized (LISTENERS) {
                for (int i = 0; i < LISTENERS.size(); i++) {
                    try {
                        MessagePumpListener listener = LISTENERS.elementAt(i);
                        listener.messageReceived(systemMsg);
                    } catch (Throwable throwable) {
                        AppLog.error(throwable);
                    }
                }
            }

        }
    }



    /**
     * Exposes the postMesssage method of the System MessagePump
     *
     * @param systemMsg the message the will be sent to the system message pump
     */
    public static void postMessage(SystemMsg systemMsg) {
        MESSAGE_PUMP.postMessage(systemMsg);

        String contents = "";
        if (0 < systemMsg.msg.length) {
            contents = ": " + new String(systemMsg.msg).trim();
        }
        String s = String.format("<-- (%d) %s%s",
                systemMsg.type,
                MessageLoopTypes.getMessageDescription(systemMsg.type),
                contents);
        LOG.info(s);
    }



    @Override
    protected void finalize() {
        MessagePumpEngine.LOG.info("mesage pump finalizing");
        // make sure we close the message pump
        MESSAGE_PUMP.close();
        MessagePumpEngine.LOG.info("mesage pump closed");
    }
}

