using System;
using System.Collections.Generic;
using System.Text;
using System.Net.Sockets;
using System.IO;
using System.Threading;
using System.Collections;

namespace JniorProtocol
{
    class JniorSession
    {
        public TcpClient m_socket;
        private BinaryReader m_br;
        private BinaryWriter m_bw;
        private Listener m_listener;
        private Thread m_listenThread;

        protected InternalInput[] m_inputs = new InternalInput[8];
        protected InternalOutput[] m_outputs = new InternalOutput[8];

        private int m_uid = -1;

        private bool m_enumerationReceived = false;
        private long[] m_devices = null;
        private Hashtable awaitingDevices = new Hashtable();

        public delegate void DeviceHandler(Device device);
        public event DeviceHandler OnDeviceUpdate;

        private Hashtable m_devicesById = new Hashtable();


        public JniorSession()
        {
            for (int i = 1; i <= 8; i++)
            {
                m_inputs[i - 1] = new InternalInput(i * 0x100 + 0xff);
                m_outputs[i - 1] = new InternalOutput(i * 0x100 + 0x100ff);
            }
        }

        public InternalInput GetInput(int channel)
        {
            return m_inputs[channel - 1];
        }

        public InternalOutput GetOutput(int channel)
        {
            return m_outputs[channel - 1];
        }

        public Device GetDeviceById(long id)
        {
            if (!m_devicesById.ContainsKey(id))
            {
                switch (id & 0xff)
                {
                    case 0xff:
                        int channel = (int)((id >> 8) & 0xff);
                        switch (id >> 16)
                        {
                            case 0:
                                m_devicesById.Add(id, GetInput(channel));
                                break;
                            case 1:
                                m_devicesById.Add(id, GetOutput(channel));
                                break;
                        }
                        break;

                    case 0xfc:
                        m_devicesById.Add(id, new TypeFC(id));
                        break;


                    case 0xfd:
                        m_devicesById.Add(id, new TypeFD(id));
                        break;


                    case 0xfe:
                        m_devicesById.Add(id, new TypeFE(id));
                        break;
                }
            }

            return (Device)m_devicesById[id];
        }

        public bool Connect(string host, int port)
        {
            /** try to connect */
            try
            {
                m_socket = new TcpClient();
                m_socket.Connect(host, port);

                /** get the binary input and output streams */
                m_br = new BinaryReader(m_socket.GetStream());
                m_bw = new BinaryWriter(m_socket.GetStream());

                /** tell the listener thread to start listening */
                m_listener = new Listener(m_br, this);
                m_listener.OnExternalRead += new Listener.ExternalDeviceHandler(m_listener_OnExternalRead);
                m_listenThread = new Thread(new ThreadStart(m_listener.Receive));
                m_listenThread.Start();

                return true;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
                return false;
            }
        }

        public void Disconnect()
        {
            /* close and nullify the reader, writer, and socket */
            if (m_br != null)
            {
                m_br.Close();
                m_br = null;
            }
            if (m_bw != null)
            {
                m_bw.Close();
                m_bw = null;
            }
            if (m_socket != null)
            {
                m_socket.Close();
                m_socket = null;
            }
        }

        public int Login(string username, string password)
        {
            byte[] sendBytes = new byte[512];
            /* always start at pos 5 to leave room for the header */
            int pos = 5;

            /** message type 126 */
            sendBytes[pos++] = (byte)126;

            /** send the username */
            sendBytes[pos++] = (byte)(username.Length);
            for (int i = 0; i < username.Length; i++)
                sendBytes[pos++] = (byte)(username.ToCharArray()[i]);

            /** send the password */
            sendBytes[pos++] = (byte)(password.Length);
            for (int i = 0; i < password.Length; i++)
                sendBytes[pos++] = (byte)(password.ToCharArray()[i]);

            m_listener.OnLogin += new Listener.LoginHandler(m_listener_OnLogin);
            m_uid = -1;
            send(sendBytes, 0, pos);

            long start = DateTime.Now.Ticks / 10000;
            while (m_uid == -1 && ((DateTime.Now.Ticks / 10000) - start < 10000))
            {
                Thread.Sleep(100);
            }

            return m_uid;
        }

        private void m_listener_OnLogin(int uid)
        {
            Console.WriteLine("Login Acknowlegement = " + uid);
            m_uid = uid;
        }

        /// <summary>
        /// This method will allow the user to set wether the monitor packet is 
        /// enabled or not.  An enabled monitor packet will be sent any time there 
        /// is a change to the I/O
        /// </summary>
        /// <param name="enabled">wether or not the monitor packet is enabled</param>
        public void EnableMonitorPacket(bool enabled)
        {
            byte[] sendBytes = new byte[512];
            /* always start at pos 5 to leave room for the header */
            int pos = 5;

            /** message type 5 - request packet */
            sendBytes[pos++] = (byte)5;

            if (enabled)
            {
                /** enable monitor packet */
                sendBytes[pos++] = 0;
                sendBytes[pos++] = (byte)5;
            }
            else
            {
                /** disable monitor packet */
                sendBytes[pos++] = 0;
                sendBytes[pos++] = (byte)4;
            }

            send(sendBytes, 0, pos);
        }

        /// <summary>
        /// This method will read the given devices.
        /// </summary>
        /// <param name="devices">an array of devices to subscribe to</param>
        public void ReadDevices(long[] devices)
        {
            byte[] sendBytes = new byte[512];
            /* always start at pos 5 to leave room for the header */
            int pos = 5;

            /** message type 21 - device read packet */
            sendBytes[pos++] = (byte)21;

            /** device count */
            sendBytes[pos++] = (byte)((devices.Length >> 8) & 0xff);
            sendBytes[pos++] = (byte)devices.Length;

            /** set the device id */
            foreach (long device in devices)
            {
                sendBytes[pos++] = (byte)((device >> 56) & 0xff);
                sendBytes[pos++] = (byte)((device >> 48) & 0xff);
                sendBytes[pos++] = (byte)((device >> 40) & 0xff);
                sendBytes[pos++] = (byte)((device >> 32) & 0xff);
                sendBytes[pos++] = (byte)((device >> 24) & 0xff);
                sendBytes[pos++] = (byte)((device >> 16) & 0xff);
                sendBytes[pos++] = (byte)((device >> 8) & 0xff);
                sendBytes[pos++] = (byte)(device & 0xff);
            }

            send(sendBytes, 0, pos);
        }

        /// <summary>
        /// This method will subscribe to the given devices.  A subscription to a 
        /// device means that a device read packet will be sent anytime there is a 
        /// measurable change to that device.
        /// </summary>
        /// <param name="devices">an array of devices to subscribe to</param>
        public void SubscribeDevices(long[] devices)
        {
            byte[] sendBytes = new byte[512];
            /* always start at pos 5 to leave room for the header */
            int pos = 5;

            /** message type 25 - device subscription packet */
            sendBytes[pos++] = (byte)25;

            /** device count */
            sendBytes[pos++] = (byte)((devices.Length >> 8) & 0xff);
            sendBytes[pos++] = (byte)devices.Length;

            /** set the device id */
            foreach (long device in devices)
            {
                sendBytes[pos++] = (byte)((device >> 56) & 0xff);
                sendBytes[pos++] = (byte)((device >> 48) & 0xff);
                sendBytes[pos++] = (byte)((device >> 40) & 0xff);
                sendBytes[pos++] = (byte)((device >> 32) & 0xff);
                sendBytes[pos++] = (byte)((device >> 24) & 0xff);
                sendBytes[pos++] = (byte)((device >> 16) & 0xff);
                sendBytes[pos++] = (byte)((device >> 8) & 0xff);
                sendBytes[pos++] = (byte)(device & 0xff);
            }

            send(sendBytes, 0, pos);
        }

        public void ToggleOutput(int channel)
        {
            byte[] sendBytes = new byte[512];
            /* always start at pos 5 to leave room for the header */
            int pos = 5;

            /** message type 10 - command packet */
            sendBytes[pos++] = (byte)10;

            /** toggle action */
            sendBytes[pos++] = (byte)3;

            /** set the channel */
            sendBytes[pos++] = (byte)((channel >> 8) & 0xff);
            sendBytes[pos++] = (byte)(channel & 0xff);

            send(sendBytes, 0, pos);
        }

        public void ClearInputCounter(int channel)
        {
            byte[] sendBytes = new byte[512];
            /* always start at pos 5 to leave room for the header */
            int pos = 5;

            /** message type 10 - command packet */
            sendBytes[pos++] = (byte)10;

            /** clear input counter */
            sendBytes[pos++] = (byte)5;

            /** set the channel */
            sendBytes[pos++] = (byte)((channel >> 8) & 0xff);
            sendBytes[pos++] = (byte)(channel & 0xff);

            send(sendBytes, 0, pos);
        }

        public long[] EnumerateExternal()
        {
            byte[] sendBytes = new byte[512];
            /* always start at pos 5 to leave room for the header */
            int pos = 5;

            /** message type 26 */
            sendBytes[pos++] = (byte)26;

            /** external only flag */
            sendBytes[pos++] = (byte)2;

            m_listener.OnEnumeration += new Listener.EnumerateHandler(m_listener_OnEnumeration);
            m_enumerationReceived = false;
            send(sendBytes, 0, pos);

            long start = DateTime.Now.Ticks / 10000;
            while (!m_enumerationReceived && ((DateTime.Now.Ticks / 10000) - start < 10000))
            {
                Thread.Sleep(100);
            }

            return m_devices;
        }

        private void m_listener_OnEnumeration(int flag, int count, long[] devices)
        {
            Console.WriteLine("Enumeration Received, " + count + " devices");
            m_devices = devices;

            for (int i = 0; i < count; i++)
            {
                GetDeviceById(devices[i]);
            }

            m_enumerationReceived = true;
        }

        public TypeFD[] ReadExternalTenVoltModules(long[] devices)
        {
            TypeFD[] ret = new TypeFD[devices.Length];

            awaitingDevices = new Hashtable();

            byte[] sendBytes = new byte[512];
            /* always start at pos 5 to leave room for the header */
            int pos = 5;

            /** message type 21 */
            sendBytes[pos++] = (byte)21;

            int count = devices.Length;
            Array.Copy(StreamUtils.ToBigEndian(count, 2), 0, sendBytes, pos, 2);
            pos += 2;

            for (int i = 0; i < count; i++)
            {
                //sendBytes[pos++];
                Array.Copy(StreamUtils.ToBigEndian(devices[i], 8), 0, sendBytes, pos, 8);
                pos += 8;

                awaitingDevices.Add(devices[i], null);
            }

            send(sendBytes, 0, pos);

            long start = DateTime.Now.Ticks / 10000;
            while (((DateTime.Now.Ticks / 10000) - start < 10000))
            {
                bool done = true;

                ICollection values = awaitingDevices.Values;
                int i = 0;
                foreach (Device value in values)
                {
                    if (value == null)
                    {
                        done = false;
                        break;
                    }
                    else
                    {
                        ret[i] = (TypeFD)value;
                    }
                    i++;
                }

                if (done) break;

                Thread.Sleep(100);
            }

            return ret;
        }

        private void m_listener_OnExternalRead(Device device)
        {
            awaitingDevices[device.address] = device;
        }

        public void WriteTenVoltModule(TypeFD device, int channel, int value)
        {
            byte[] sendBytes = new byte[512];
            /* always start at pos 5 to leave room for the header */
            int pos = 5;

            /** message type 26 - command packet */
            sendBytes[pos++] = (byte)23;

            /** count */
            int count = 1;
            Array.Copy(StreamUtils.ToBigEndian(count, 2), 0, sendBytes, pos, 2);
            pos += 2;

            Array.Copy(StreamUtils.ToBigEndian(device.address, 8), 0, sendBytes, pos, 8);
            pos += 8;

            if (channel == 1)
            {
                device.output1 = (short)value;
            }
            else if (channel == 2)
            {
                device.output2 = (short)value;
            }

            byte[] bytes = device.GetBytes();
            Array.Copy(StreamUtils.ToBigEndian(bytes.Length, 2), 0, sendBytes, pos, 2);
            pos += 2;
            Array.Copy(bytes, 0, sendBytes, pos, bytes.Length);
            pos += bytes.Length;

            send(sendBytes, 0, pos);

        }

        private void send(byte[] bytes, int offset, int length)
        {
            /** start of header byte */
            bytes[0] = 0x01;

            /** length **/
            int len = length - 5;
            bytes[1] = (byte)((len >> 8) & 0xff);
            bytes[2] = (byte)(len & 0xff);

            /** CRC, we are passing 0xffff for now */
            bytes[3] = 0xff;
            bytes[4] = 0xff;

            m_bw.Write(bytes, offset, length);
        }







        class Listener
        {
            private BinaryReader m_br;
            private JniorSession m_parent;

            public Listener(BinaryReader br, JniorSession parent)
            {
                m_br = br;
                m_parent = parent;
            }

            public delegate void LoginHandler(int uid);
            public event LoginHandler OnLogin;

            public delegate void EnumerateHandler(int flag, int count, long[] devices);
            public event EnumerateHandler OnEnumeration;

            public delegate void ExternalDeviceHandler(Device device);
            public event ExternalDeviceHandler OnExternalRead;

            public void Receive()
            {
                byte[] b = null;
                int count;

                try
                {

                    while (true)
                    {
                        int syncByte = m_br.ReadByte();
                        Console.WriteLine("Sync Byte = " + syncByte);
                        if (syncByte == 0x01)
                        {
                            int length = (int)StreamUtils.FromBigEndian(m_br.ReadBytes(2), 0, 2);
                            int crc = (int)StreamUtils.FromBigEndian(m_br.ReadBytes(2), 0, 2);
                            int msgType = m_br.ReadByte();
                            Console.WriteLine("Packet Type = " + msgType);

                            switch (msgType)
                            {
                                /** monitor packet */
                                case 1:
                                    Console.WriteLine("Monitor Packet");

                                    int versionSize = m_br.ReadByte();
                                    b = m_br.ReadBytes(versionSize);
                                    String version = System.Text.Encoding.ASCII.GetString(b);

                                    /** read the I/O states */
                                    for (int i = 0; i < 8; i++)
                                    {
                                        m_parent.m_inputs[i].Parse(m_br);
                                        m_parent.OnDeviceUpdate(m_parent.m_inputs[i]);
                                    }
                                    for (int i = 0; i < 8; i++)
                                    {
                                        m_parent.m_outputs[i].Parse(m_br);
                                        m_parent.OnDeviceUpdate(m_parent.m_outputs[i]);
                                    }

                                    long timestamp = StreamUtils.FromBigEndian(m_br.ReadBytes(8), 0, 8);

                                    break;

                                case 22:
                                    Console.WriteLine("Device Read Response");

                                    count = (int)StreamUtils.FromBigEndian(m_br.ReadBytes(2), 0, 2);

                                    for (int i = 0; i < count; i++)
                                    {
                                        long address = StreamUtils.FromBigEndian(m_br.ReadBytes(8), 0, 8);
                                        Console.WriteLine(String.Format("{0:x16}", address));

                                        int dataLength = (int)StreamUtils.FromBigEndian(m_br.ReadBytes(2), 0, 2);

                                        Device device = m_parent.GetDeviceById(address);
                                        if (device != null)
                                        {
                                            /** here we use the extra parameter 
                                             * indicating that the parse is coming 
                                             * from the device read packet.  this 
                                             * is used because the InternalInput 
                                             * and InternalOutput classes we be 
                                             * parsed differently depending if the 
                                             * Parse is called from the 
                                             * MonitorPacket or the DeviceRead
                                             */
                                            device.Parse(m_br, true);
                                            m_parent.OnDeviceUpdate(device);


                                            if (OnExternalRead != null)
                                            {
                                                OnExternalRead(device);
                                            }
                                        }
                                    }

                                    break;

                                case 27:
                                    Console.WriteLine("Enumeration Response");

                                    int flag = m_br.ReadByte();
                                    count = (int)StreamUtils.FromBigEndian(m_br.ReadBytes(2), 0, 2);

                                    long[] devices = null;

                                    if (count > 0)
                                    {
                                        devices = new long[count];
                                        for (int i = 0; i < count; i++)
                                        {
                                            devices[i] = StreamUtils.FromBigEndian(m_br.ReadBytes(8), 0, 8);
                                        }
                                    }

                                    OnEnumeration(flag, count, devices);

                                    break;

                                /** login acknowlegement */
                                case 125:
                                    Console.WriteLine("Login Acknowlegement");

                                    int uid = m_br.ReadByte();
                                    OnLogin(uid);

                                    break;

                                /** handle all other cases gracefully */
                                default:
                                    Console.WriteLine("Other Packet = " + msgType + ", length=" + length);
                                    m_br.ReadBytes(length - 1);
                                    break;
                            }
                        }
                    }

                }
                catch (Exception ex)
                {
                    Console.WriteLine(ex.ToString());
                }

            }

        }

    }
}
