using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using JniorDll.Net;
using System.Threading;

namespace JniorDll.Set.Clock
{
    public partial class Form1 : Form
    {
        // our jnior connection
        private Jnior _jnior = new Jnior();
        // flag indicating that the JNIOR connection is ready
        private bool _isReady = false;


        public Form1()
        {
            InitializeComponent();
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            // set up our connection properties object to connect to 10.0.0.146.  
            // We define the connection to connect to port 0.  this forces the 
            // DLL to query the JNIOR over the web port, port 80, for the correct 
            // JNIOR protocol port.  If this option is not appealing to you then 
            // you can define the port number.  tell the connection that if the 
            // connection cannot be established of it the connection is lost that 
            // it can retry 10 times on a 15 second interval.
            ConnectionProperties cp = new ConnectionProperties("10.0.0.135", 0, 10, 15000);
            // connect.  this is a blocking call.  operation of the application 
            // will be interrupted until this call returns.  there is a connect 
            // async function available.  please refer to the JNIOR DLL 
            // documentation for examples on how to use that feature
            _jnior.Connect(cp);

            // make sure the connection was a success
            if (_jnior.IsConnected())
            {
                // use the default login credentials
                LoginProperties lp = new LoginProperties("jnior", "jnior");
                // now log in.  again this is a blocking call and there is an 
                // asynchronous call available
                _jnior.Login(lp);

                // make sure we logged in with admin privileges
                if (_jnior.IsLoggedIn() && _jnior.GetLoginLevel() >= jnior_dll_calls.LOGIN_ADMIN)
                {
                    _isReady = true;
                    _jnior.AddMonitorCallback(new jnior_dll_calls.MonitorCallback(MonitorReceived));
                }
            }
        }

        private void btnSetTime_Click(object sender, EventArgs e)
        {
            if (!_isReady) return;

            // build a time object from our UI
            DateTime dt = dtpJniorTime.Value;
            dt = dt.AddHours((int)updnJniorHour.Value);
            dt = dt.AddMinutes((int)updnJniorMinute.Value);
            dt = dt.AddSeconds((int)updnJniorSecond.Value);

            // make sure we send it in universal time
            dt = dt.ToUniversalTime();

            // set the Jnior Time
            _jnior.SetClock(dt);
        }

        // the user wants to refresh our clock
        private void btnRefresh_Click(object sender, EventArgs e)
        {
            if (_jnior != null)
            {
                // this method requests the time BUT it will block until the time 
                // is received possibly causing the gui to hang for a second if it 
                // is called from the gui thread
                UpdateTime(_jnior.RequestClock());

                // request the jnior to respond with the monitor packet.  passing 
                // a 0 for the argument tell the jnior that this is a one time 
                // request.
                //_jnior.RequestMonitorPacket(0);
            }
        }

        // this will be called when a monitor packet is received by the DLL.  Note,
        // this method will be called from a non gui thread.  care must be taken to 
        // ensure non cross threaded operations.  To do this we will make a delegate
        // that will be assigned the Updatetime function.  We will check to see if 
        // the delegate needs to be used.  If it does we will call the invoke method 
        // to update the time.
        private void MonitorReceived(Jnior jnior, MonitorArgs args)
        {
            // get the JNIOR time
            DateTime jniorTime = _jnior.GetKnownJniorTime();

            Console.WriteLine(args.Monitor.jniorTime);

            if (this.InvokeRequired)
            {
                this.Invoke(new UpdateTimeDelegate(UpdateTime), new object[] { jniorTime });
            }
            else
            {
                UpdateTime(jniorTime);
            }
        }

        // Updates the time
        private delegate void UpdateTimeDelegate(DateTime time);
        private void UpdateTime(DateTime time)
        {
            lblJniorTicks.Text = (time.Ticks / 10000).ToString();

            // make sure it is in local time
            // the new .net dll will convert this for us
            // time = time.ToLocalTime();

            // set our ui object with the known jnior time
            lblJniorTime.Text = "Time: " + time.ToString();
            dtpJniorTime.Value = time.Date;
            updnJniorHour.Value = time.Hour;
            updnJniorMinute.Value = time.Minute;
            updnJniorSecond.Value = time.Second;
        }

        private void tmr_Tick(object sender, EventArgs e)
        {
            lblNowInfo.Text = "Using the DateTime.Now object, TZ: " + DateTime.Now.Kind;
            lblNowUnspecifiedTime.Text = DateTime.Now.ToString();
            lblNowUnspecifiedTicks.Text = (DateTime.Now.Ticks / 10000).ToString();
            lblNowGmtTime.Text = DateTime.Now.ToUniversalTime().ToString();
            lblNowGmtTicks.Text = (DateTime.Now.ToUniversalTime().Ticks / 10000).ToString();
            lblNowLocalTime.Text = DateTime.Now.ToLocalTime().ToString();
            lblNowLocalTicks.Text = (DateTime.Now.ToLocalTime().Ticks / 10000).ToString();

            DateTime dt = new DateTime(DateTime.Now.Ticks);
            lblDateTimeInfo.Text = "Using a new DateTime object, TZ: " + dt.Kind;
            lblUnspecifiedTime.Text = dt.ToString();
            lblUnspecifiedTicks.Text = (dt.Ticks/10000).ToString();
            lblGmtTime.Text = dt.ToUniversalTime().ToString();
            lblGmtTicks.Text = (dt.ToUniversalTime().Ticks / 10000).ToString();
            lblLocalTime.Text = dt.ToLocalTime().ToString();
            lblLocalTicks.Text = (dt.ToLocalTime().Ticks / 10000).ToString();
        }

    }
}