
/*---------------------------------------------------------------------------
 * Copyright (C) 1999,2000 Dallas Semiconductor Corporation, All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Dallas Semiconductor
 * shall not be used except as stated in the Dallas Semiconductor
 * Branding Policy.
 *---------------------------------------------------------------------------
 */

package com.integpg.io;



import com.integpg.system.JANOS;
import java.util.*;
import java.io.CharConversionException;


/**
 * <p>This class defines an interface to allow conversion of characters
 * to bytes for a particular encoding scheme.  Encoding converters
 * should reside in the <code>com.integpg.io</code> package.</p>
 *
 * <p>Many encoding schemes need to take state into account in the
 * conversion process.  That is, the conversion to a <code>byte</code> might depend on
 * the character sequence converted before it.  To accommodate this, the
 * <code>CharToByteConverter</code> has the ability to remember state between conversions
 * (between calls to <code>convert()</code>.  Therefore, the caller
 * should call the <code>flush()</code> method to finalize the conversion
 * and reset the converter's internal state.</p>
 *
 * <p>Subclasses of this <code>abstract</code> class need to implement
 * <code>getMaxByteCount()</code>, <code>convert()</code>,
 * <code>flush()</code>, and <code>getName()</code>.</p>
 *
 * <p>Programs should not call into a converter directly.  A
 * better method of executing character conversions is to use the
 * <code>java.lang.String.getBytes(String)</code> method.</p>
 *
 * <pre><code>
 *      ...
 *      String str = new String("this is a test");
 *      byte[] convertedString = str.getBytes("UTF8");
 *      ...
 * </code></pre>
 * <p>This will convert the <code>String</code> "this is a test" into a <code>byte</code>
 * array according to the UTF8 encoding scheme.</p>
 *
 * @see ByteToCharConverter
 */
public abstract class CharToByteConverter
{
   private static Hashtable converters  = null;
   private static String    prepend     = null;
   private static String    defaultName = null;

   /**
    * Dynamically loads a <code>CharToByteConverter</code> for the specified encoding scheme.
    * All converters should be placed in the <code>com.integpg.io package</code>, and have class
    * name CharToByteNAME, where NAME is the encoding scheme.  For example, the UTF8
    * <code>CharToByteConverter</code> is called <code>com.integpg.io.CharToByteUTF8</code>.
    *
    * @param name the name of the encoding scheme, such as "UTF8"
    * @return converter for the specified encoding scheme, or <code>null</code>
    * if none could be found
    */
   public static CharToByteConverter getConverter (String name)
   {}

   /**
    * Returns the default <code>CharToByteConverter</code> for the system.  The name
    * of the default encoding scheme is stored in the system property
    * "file.encoding".  This method finds the name of the default
    * encoding scheme, and calls <code>getConverter()</code> with that name
    * as its argument.
    *
    * @return converter that corresponds to the system's default
    * file encoding property, or <code>null</code> if the converter could
    * not be found
    * @see #getConverter(java.lang.String)
    */
   public static CharToByteConverter getDefaultConverter ()
   {}

   /**
    * Returns the number of bytes that the specified character
    * sequence will require for encoding.  For instance, in UTF8
    * encoding, a character can be encoded as one, two, or three
    * bytes.  This method should always be called before the
    * <code>convert()</code> method.  The value returned may not be the actual
    * number of converted bytes that will be produced due to
    * conversion errors, but it will be the maximum that will be
    * produced.
    *
    * @param forThis contains the character sequence that will be encoded
    * to determine the number of bytes required
    * @param start offset into the character array to begin processing
    * @param end ending offset in the character array to end processing.
    * The number of processed characters will then be <code>(end-start)</code>.
    * @return The number of bytes required to encode the specified character sequence.
    * @see #convert(char[],int,int,byte[],int,int)
    */
   public abstract int getMaxByteCount (char[] forThis, int start, int end);

   /**
    * Converts the specified <code>char</code> array into a <code>byte</code> array based on
    * this <code>CharToByteConverter</code>'s encoding scheme.  <code>getMaxByteCount()</code> should
    * always be called first to find out how much room is required in the
    * destination byte array.
    *
    * @param src the same character array passed to <code>getMaxByteCount()</code>
    * @param srcStart the same starting offset as passed to <code>getMaxByteCount()</code>
    * @param srcEnd the same ending offset as passed to <code>getMaxByteCount()</code>
    * @param dst the destination <code>byte</code> array
    * @param dstStart offset to begin storing converted characters in the
    * destination array
    * @param dstEnd  the ending location for storing converted characters into the
    * destination array.  This argument may usually be ignored, as
    * the algorithm may choose to continue converting characters until
    * finished.
    *
    * @return number of bytes created and stored from this character sequence
    *
    * @exception java.io.CharConversionException If an illegal character is encountered that cannot be converted
    *
    * @see #getMaxByteCount(char[],int,int)
    * @see #flush(byte[],int,int)
    */
   public abstract int convert (char[] src, int srcStart, int srcEnd,
                                byte[] dst, int dstStart, int dstEnd)
      throws CharConversionException;

   /**
    * Tells the <code>CharToByteConverter</code> to convert
    * any unconverted data it has internally stored.
    * Some <code>CharToByteConverter</code>'s will store state between
    * calls to <code>convert()</code>.  Since the converter may be left in
    * an unknown state, the converter should be flushed to
    * notify it that no more input will be received.  The converter
    * can handle any unfinished conversions before its output is
    * used.
    *
    * @param buff the destination byte array
    * @param start the next available offset into the destination array
    * @param end offset in the destination array to stop placing data
    * (may be ignored by some algorithms)
    * @return number of bytes that were stored in the destination array
    * from this call to flush().
    * @exception java.io.CharConversionException if an illegal character is encountered that cannot be converted.
    * @see #convert(char[],int,int,byte[],int,int)
    */
   public abstract int flush (byte[] buff, int start, int end)
      throws CharConversionException;

   /**
    * Returns the name of this encoding scheme.  For example, "UTF8".
    *
    * @return the name of this encoding scheme
    */
   public abstract String getName ();
}
