//	JNIOR Automation Network Operating System (JANOS)
//  CopyRight (C) 2012-2022 INTEG process group inc. All Rights Reserved.

/* ---------------------------------------------------------------------------
 * This software is INTEG process group inc proprietary. This source code
 * is for internal company use only and any other use without prior
 * consent of INTEG process group inc is prohibited.
 *
 * @author Bruce Cloutier
 * Inception Date: 
 * -------------------------------------------------------------------------*/

package com.integpg.net;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.StringReader;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 *
 * @author Bruce Cloutier
 */
public class FTPClient {

    private Socket socket = null;
    private BufferedReader in = null;
    private PrintWriter out = null;
    private int so_timeout = 10000;
    private StringBuffer logb = null;
    private boolean passive = false;
    private boolean secure = false;
    
    /**
     * Establishes the FTP connection. This retains the banner announcement.
     * 
     * @param server URL or IP address of the target FTP Server
     * @return TRUE if the connection is properly established.
     */
    public boolean connect(String server) {}
    
    /**
     * Closes the FTP connection.
     */
    public void close() {}
    
    /**
     * Sets timeout in milliseconds used during data transfers. Default is 10 seconds (10000).
     * 
     * @param to timeout in milliseconds
     */
    public void setSoTimeout(int to) {}

    /**
     * Used to set passive or active mode. By default active transfers are used.
     * 
     * @param mode true if passive transfers are to be used.
     */
    public void setPassive(boolean mode) {}
    
    /**
     * Resets the cumulative log. This creates a new instance of the log.
     */
    public void resetLog() {}
    
    /**
     * Enables the cumulative log. This releases an existing log and if logging is enabled
     * creates a new instance. Logging is disabled by default.
     * @param enable set true if logging is to be enabled
     */
    public void enableLog(boolean enable) {}
    
    /**
     * Obtains the cumulative log content.
     * 
     * @return string containing the log
     */
    public String getLog() {}

    /**
     * Perform user login.
     * 
     * @param user username
     * @param pass password
     * @return TRUE if the user has successfully logged in
     */
    public boolean login(String user, String pass) {}
    
    /**
     * Establish a secure connection. This issues the AUTH TLS command and negotiates a TLS
     * secured connection if a positive response is received.
     * 
     * @return true if a secure connection is at least attempted
     */
    public boolean auth() {}

    /**
     * Issues CDUP command. Makes the parent of the current directory the working directory.
     * 
     * @return server's response
     */
    public String cdup() {}

    /**
     * Change working directory
     * 
     * @param dir string containing the directory
     * @return server's response
     */
    public String cwd(String dir) {}

    /**
     * Deletes the given file on the remote host.
     * 
     * @param file remote filename
     * @return true if successful
     */
    public boolean delete(String file) {}

    /**
     * Issues FEAT command
     * 
     * @return server's response
     */
    public String feat() {}

    /**
     * Issues the LIST command.
     * 
     * @param nlst set true to return list of files only
     * @return server's response
     */
    public String list(boolean nlst) {}

    /**
     * Returns the last-modified time of the given file on the remote host. The format is
     * "YYYYMMDDhhmmss": YYYY is the four-digit year, MM is the month from 01 to 12, DD 
     * is the day of the month from 01 to 31, hh is the hour from 00 to 23, mm is the 
     * minute from 00 to 59, and ss is the second from 00 to 59.
     * 
     * @param file name of the remote file
     * @return string containing the server's response
     */
    public String mdtm(String file) {}

    /**
     * Set remote file last modified timestamp
     * 
     * @param file filename on the remote system
     * @param filedate timestamp in YYYYMMDDHHMMSS format
     * @return response string
     */
    public String mfmt(String file, String filedate) {}

    /**
     * Creates the named directory on the remote host.
     * 
     * @param dir directory name
     * @return true if successful
     */
    public boolean mkdir(String dir) {}

    /**
     * Return standardized facts for contents of the directory.
     * 
     * @param dir remote directory
     * @return response string
     */
    public String mlsd(String dir) {}

    /**
     * Return standardized facts about a single item.
     * 
     * @param file the item of interest (current directory if absent).
     * @return server's response in its entirety.
     */
    public String mlst(String file) {}

    /**
     * Issues NOOP command
     * 
     * @return server's response
     */
    public String noop() {}

    /**
     * Issues PWD command.
     * 
     * @return current working directory
     */
    public String pwd() {}

    /**
     * Issue PORT command.
     * 
     * @param addr local IP address
     * @param portnum local port number
     * @return server's response
     */
    public String port(InetAddress addr, int portnum) {}

    /**
     * Issues QUIT command
     * 
     * @return server's response
     */
    public String quit() {
        return command("QUIT");
    }
    

    /**
     * Renames a remote file.
     * 
     * @param from remote file to be renamed
     * @param to new name for the file
     * @return true if successful
     */
    public boolean rename(String from, String to) {
        if (socket == null)
            return false;

        // send RNFR command
        String resp = command("RNFR " + from);
        if (responseCode(resp) != 350)
            return false;

        // send RNTO command
        resp = command("RNTO " + to);
        return (responseCode(resp) == 250);
    }

    /**
     * Deletes the named directory on the remote host.
     * 
     * @param dir directory name
     * @return true if successful
     */
    public boolean rmdir(String dir) {
        if (socket == null)
            return false;

        // send RMD command
        String resp = command("RMD " + dir);
        return (responseCode(resp) == 250);
    }

    /**
     * Returns the size of the remote file. Syntax: SIZE remote-filename
     * 
     * @param file the remote filename
     * @return integer size or 0 on error
     */
    public int size(String file) {
        if (socket == null)
            return 0;

        // send SIZE command
        String resp = command("SIZE " + file);

        // Obtain size
        if (responseCode(resp) == 213)
            return Integer.parseInt(resp.substring(4));
        
        return 0;
    }

    /**
     * Issues SYST command
     * 
     * @return server's response
     */
    public String syst() {}

    /**
     * Issues TYPE command with the provided parameter.
     * 
     * @param stype ASCII type character ("I: for binary)
     * @return server's response
     */
    public String type(String stype) {
        if (socket == null)
            return null;

        // send TYPE command
        return command("TYPE " + stype);
    }

    /**
     * Issue PASV command and retrieve connect parameters
     * 
     * @return byte array containing addressing octets
     */
    public int[] pasv() {
        int[] params = new int[6];
        
        if (socket == null)
            return null;

        // send USER command
        String resp = command("PASV");
        if (responseCode(resp) != 227)
            return null;

        // decode addressing from the response
        //  This assumes nothing and simply expects 6 groups of digits representing
        //  byte values.
        int n = 0;
        for (int p = 4; n < 6 && p < resp.length(); ) {

            // skip to digits
            if (!Character.isDigit(resp.charAt(p))) {
                p++;
                continue;
            }

            // acquire value
            char ch;
            int b = 0;
            while (Character.isDigit(ch = resp.charAt(p++)))
            {
                b = 10 * b + (ch - '0');
                if (b >= 256)
                    return null;
            }

            // save value and continue
            params[n++] = b;
        }

        return params;
    }

    /**
     * Obtain the directory. Since FTP serves can provide vastly differing directory
     *  list formats this method returns the listing in its entirety. This may be 
     *  parsed as needed subsequently.
     * 
     * @param nlst set true to return list of filenames only
     * @return string containing the directory listing in its entirety
     */
    public String dir(boolean nlst) {
        String content = "";
        TextCollector dc = null;
        ServerSocket port = null;
        
        try {
            if (passive) {
                // prepare to run in passive mode
                int[] params = pasv();
                if (params == null) 
                    return content;

                int address = (params[0] << 24) | (params[1] << 16) | (params[2] << 8) | params[3];
                InetAddress addr = new InetAddress(address);
                int portnum = (params[4] << 8) | params[5];

                // open a data thread
                dc = new TextCollector(addr, portnum, so_timeout, secure);

            } else {
                // open a port for an incoming data connection (active mode)
                port = new ServerSocket(0);

                InetAddress address = InetAddress.getLocalHost();
                int portnum = port.getLocalPort();

                // issue PORT command
                port(address, portnum);

                // open a data thread
                dc = new TextCollector(port, so_timeout, secure);
            }
            
            // Start the data collection and make sure it is ready befopre issuing the command
            new Thread(dc).start();
            while (!dc.hasStarted())
                sleep(50);
            
            // issue LIST command
            String resp = list(nlst);

            // wait for data
            while (responseStatus(resp) == 1)
                resp = getresponse();

            // Wait for data to be collected
            while (dc.isRunning())
                sleep(50);
                
            // include log information
            if (dc.getLog() != null)
                logprint(dc.getLog());

            // close port
            if (port != null)
                port.close();

            content = dc.getContent();
        }
        catch (IOException ioe) {
            logprintln(ioe);
        }
        
        return content;
    }

    /**
     * Returns an array of FileEntry containing the details for each file or folder in the directory.
     * 
     * @param listing The directory listing text
     * @return array of file entries
     */
    public FileEntry[] parseDir(String listing) {
        int format = -1;
        int n;
        int year = -1;
        FileEntry[] list = new FileEntry[0];
        
        BufferedReader str = new BufferedReader(new StringReader(listing));
        try {
            String line;
            while ((line = str.readLine()) != null) {

                // remove excess spaces
                line = line.trim();
                int len = tokenlen(line, 0);
                
                // check for formats (done one time)
                if (format == -1) {

                    // check for unix format
                    for (n = 0; n < len; n++)
                        if ("-drwx".indexOfChar(line.charAt(n), 0) == -1)
                            break;
                    if (n == len) {
                        format = 1; // unix
                        
                        // obtain current year as the year may be omitted from the listing
                        Date dt = new Date();
                        year = dt.getYear() + 1900;
                    }
                    else {

                        // check for date
                        for (n = 0; n < len; n++)
                            if ("-/0123456789".indexOfChar(line.charAt(n), 0) == -1)
                                break;
                        if (n == len)
                            format = 2;     // date first
                        else
                            continue;
                    }
                }

                // create an entry for this item
                FileEntry fe = new FileEntry();
                
                // parse unix format
                if (format == 1) {
                    if (line.charAt(0) == 'd')
                        fe.directory = true;
                    n = nexttoken(line, 0); // skip permissions
                    n = nexttoken(line, n); // skip link count
                    n = nexttoken(line, n); // skip owner
                    n = nexttoken(line, n); // skip group
                    fe.size = Integer.parseInt(line.substring(n, n + tokenlen(line, n)));
                    
                    // get date and time specification
                    n = nexttoken(line, n);     // beginning of month
                    len = nexttoken(line, n);   // beginning of day
                    len = nexttoken(line, len); // beginning of time/year
                    len += tokenlen(line, len); // end of timestamp
                    fe.modified = Date.parse(line.substring(n, len));

                    // get filename (balance of line)
                    n = nexttoken(line, len);
                    fe.name = line.substring(n);
                }
                else {
                    // date first format - obtain the timestamp
                    len = nexttoken(line, 0);   // beginning of time
                    len += tokenlen(line, len);   // end of timestamp
                    fe.modified = Date.parse(line.substring(0, len));
                    
                    n = nexttoken(line, len);   // beginning of size or directory flag
                    String ssize = line.substring(n, n + tokenlen(line, n));
                    if (ssize.toLowerCase().indexOf("dir") >= 0)
                        fe.directory = true;
                    else
                        fe.size = Integer.parseInt(ssize);
                    
                    n = nexttoken(line, n); // beginning of file name
                    fe.name = line.substring(n);
                }
                
                // append entry
                FileEntry[] copy = list;
                list = new FileEntry[copy.length + 1];
                for (n = 0; n < copy.length; n++)
                    list[n] = copy[n];
                list[n] = fe;
            }
        } catch (IOException ex) {
            logprintln(ex);
        }
        
        return list;
    }

    /**
     * Locate next token in string.
     * 
     * @param str string to be parsed
     * @param pos position of the current token
     * @return index of the next token
     */
    private int nexttoken(String str, int pos) {
        
        int end = str.length();
        
        // skip token
        while (pos < end && str.charAt(pos) != ' ')
            pos++;
        
        // skip white space
        while (pos < end && str.charAt(pos) == ' ')
            pos++;
        
        // return -1 if no more tokens
        if (pos == end)
            pos = -1;
        
        return pos;
    }
    
    /**
     * Obtain the length of the current token.
     * 
     * @param str string being parsed
     * @param pos position of the current token
     * @return the length of the current token (balance of token)
     */
    private int tokenlen(String str, int pos) {
        
        int end = str.length();
        int ptr = pos;
        
        // skip to enbd of token
        while (pos < end && str.charAt(ptr) != ' ')
            ptr++;
        
        return ptr - pos;
    }
    
    /**
     * Transfers file content from a remote host.
     * 
     * @param dstfile destination file on the local system
     * @param remotefile source filename on the remote host
     * @return true is the transfer succeeds
     */
    public boolean retrieve(String dstfile, String remotefile) {
        FileRetrieve fr;
        ServerSocket port = null;
        
        try {
            if (passive) {
                // prepare to run in passive mode
                int[] params = pasv();
                if (params == null) 
                    return false;

                int address = (params[0] << 24) | (params[1] << 16) | (params[2] << 8) | params[3];
                InetAddress addr = new InetAddress(address);
                int portnum = (params[4] << 8) | params[5];

                // open a transfer thread
                fr = new FileRetrieve(dstfile, addr, portnum, secure);
                
            } else {
                // open a port for an incoming data connection (active mode)
                port = new ServerSocket(0);

                InetAddress address = InetAddress.getLocalHost();
                int portnum = port.getLocalPort();

                // issue PORT command
                port(address, portnum);

                // open a transfer thread
                fr = new FileRetrieve(dstfile, port, secure);
            }

            // need to start thread before requesting the file
            new Thread(fr).start();
            while (!fr.hasStarted())
                sleep(50);

            // issue STOR command
            String resp = command("RETR " + remotefile);
            if (responseStatus(resp) == 5) 
                return false;

            // wait for transfer to complete
            while (responseStatus(resp) == 1)
                resp = getresponse();

            // wait for transfer to complete
            while (fr.isRunning())
                sleep(50);

            // include log information
            if (fr.getLog() != null)
                logprint(fr.getLog());

            // close port
            if (port != null)
                port.close();
            
            // obtain the remote file's timestamp
            resp = command("MDTM " + remotefile);
            if (responseCode(resp) == 213 && resp.length() >= 18) {
                int yr = Integer.parseInt(resp.substring(4, 8)) - 1900;
                int mon = Integer.parseInt(resp.substring(8, 10)) - 1;
                int day = Integer.parseInt(resp.substring(10, 12));
                int hr = Integer.parseInt(resp.substring(12, 14));
                int min = Integer.parseInt(resp.substring(14, 16));
                int sec = Integer.parseInt(resp.substring(16, 18));
                Date dt = new Date(yr, mon, day, hr, min, sec);
                
                // set destination file date
                File file = new File(dstfile);
                file.setLastModified(dt.getTime());
            }

        } catch (IOException ioe) {
            logprintln(ioe);
        }

        return true;
    }
    
    /**
     * Transfers file content to a remote host.
     * 
     * @param srcfile source file on the local system
     * @param remotefile destination filename on the remote host
     * @return true if the transfer succeeds
     */
    public boolean store(String srcfile, String remotefile) {
        FileStore fs;
        ServerSocket port = null;
        
        try {
            if (passive) {
                // prepare to run in passive mode
                int[] params = pasv();
                if (params == null) 
                    return false;

                int address = (params[0] << 24) | (params[1] << 16) | (params[2] << 8) | params[3];
                InetAddress addr = new InetAddress(address);
                int portnum = (params[4] << 8) | params[5];

                // open a transfer thread
                fs = new FileStore(srcfile, addr, portnum, secure);
                
            } else {
                // open a port for an incoming data connection (active mode)
                port = new ServerSocket(0);

                InetAddress address = InetAddress.getLocalHost();
                int portnum = port.getLocalPort();

                // issue PORT command
                port(address, portnum);

                // open a transfer thread
                fs = new FileStore(srcfile, port, secure);
            }

            // issue STOR command
            String resp = command("STOR " + remotefile);
            if (responseStatus(resp) == 5) 
                return false;

            // initiate the outgoing transfer
            new Thread(fs).start();
            while (!fs.hasStarted())
                sleep(50);

            // wait for transfer to complete
            while (responseStatus(resp) == 1)
                resp = getresponse();

            // wait for transfer to complete
            while (fs.isRunning())
                sleep(50);

            // include log information
            if (fs.getLog() != null)
                logprint(fs.getLog());

            // close port
            if (port != null)
                port.close();
            
            // obtain the timestamp from the source file
            File file = new File(srcfile);
            long timestamp = file.lastModified();
            Date dt = new Date(timestamp);
            SimpleDateFormat format = new SimpleDateFormat("yyyyMMddHHmmss");
            
            command("MFMT " + format.format(dt) + " " + remotefile);
            
        } catch (IOException ioe) {
            logprintln(ioe);
        }

        return true;
    }

    /**
     * Issues supplied command and logs the response. The response
     * is not interpreted.
     * 
     * @param cmd string containing the complete command.
     * @return string containing the server's response
     */
    public String command(String cmd) {
        try {
            if (socket == null)
                return "";
            
            // log the command
            logprintln(cmd);

            // send command
            out.println(cmd);
            out.flush();

            // Obtain server response
            return getresponse();
        }
        catch (IOException ioe) {
            logprintln(ioe);
        }
        
        return "";
    }

    /**
     * Gets the server response.
     * 
     * @return string containing the entire (potentially multi-line) response
     * @throws IOException 
     */
    private String getresponse() throws IOException {
        String response = in.readLine();
        String str;
        
        // exit if there is no response
        if (response == null)
            throw new IOException("no response");
        
        // exit if the response is a single line
        if (response.length() < 4 || response.charAt(3) == ' ')
        {
            logprintln(response);
            return response;
        }

        // collect additional lines
        while ((str = in.readLine()) != null) {
            response = response + "\r\n" + str;
            
            if (str.length() >= 4 && str.substring(0, 3).compareTo(response.substring(0, 3)) == 0 && str.charAt(3) == ' ')
                break;
        }

        logprintln(response);
        return response;
    }
    
    /**
     * Returns the response code in the reply.
     * 
     * @param response the response string returned by the FTP Server
     * @return value of 3-digit response code
     */
    public int responseCode(String response) {
        return Integer.parseInt(response.substring(0, 3));
    }
    
    /**
     * Returns the status of the reply.
     * 
     * 1xx	Positive Preliminary reply
     * 2xx	Positive Completion reply
     * 3xx	Positive Intermediate reply
     * 4xx	Transient Negative Completion reply
     * 5xx	Permanent Negative Completion reply
     * 6xx	Protected reply
     * 
     * @param response the response string returned by the FTP Server
     * @return value of the first digit of the response code
     */
    public int responseStatus(String response) {
        return Integer.parseInt(response.substring(0, 1));
    }

    /**
     * Returns the purpose of the reply.
     * 
     * x0x	Syntax
     * x1x	Information
     * x2x	Connections
     * x3x	Authentication and accounting
     * x4x	Unspecified as of RFC 959.
     * x5x	File system
     * 
     * @param response the response string returned by the FTP Server
     * @return value of the second digit of the response code
     */
    public int responsePurpose(String response) {
        return Integer.parseInt(response.substring(1, 2));
    }

    private void logprintln(Object obj) {
        logprint(obj);
        logprint("\r\n");
    }

    private void logprint(Object obj) {
        if (logb != null)
            logb.append(obj.toString());
    }

    private void sleep(int millis) {
        try {
            Thread.sleep(millis);
        } catch (InterruptedException ie) {
        }
    }

    /** 
     * Check if client connected to server
     * 
     * @return true if connected
     */
    public boolean isConnected() {
        return (socket != null);
    }

}

class TextCollector implements Runnable {
    
    private ServerSocket socket = null;
    private Socket data = null;
    private InetAddress address = null;
    private int port = 0;
    private String content = "";
    private boolean started = false;
    private boolean stopped = false;
    private int so_timeout;
    private String log = "";
    private boolean secure;

    
    TextCollector(ServerSocket port, int to, boolean sec) {
        socket = port;
        so_timeout = to;
        secure = sec;
    }

    TextCollector(InetAddress addr, int portnum, int to, boolean sec) {
        address = addr;
        port = portnum;
        so_timeout = to;
        secure = sec;
    }

    @Override
    public void run() {
        started = true;

        try {
            // obtain data socket by whatever means
            if (socket == null) 
                data = new Socket(address, port);
            else 
                data = socket.accept();
            
            // handle secure connections
            if (secure)
                data.setSecure(true);

            // establish a timeout
            data.setSoTimeout(so_timeout);
            
            BufferedReader in = new BufferedReader(new InputStreamReader(data.getInputStream()));
            String str;
            try {
                while ((str = in.readLine()) != null) 
                    content += str + "\r\n";
            } catch (IOException ste) {
            }
            
            // if we opened the socket we close it
            data.close();
            if (socket != null)
                socket.close();
        }
        catch (IOException ioe) {
            logprintln(ioe);
        }
        
        stopped = true;
    }
    
    /**
     * Provides access to content.
     * 
     * @return string containing content.
     */
    String getContent() {
        return content;
    }

    /**
     * Indicates that the collector has started.
     * 
     * @return true if the collector was started
     */
    boolean hasStarted() {
        return started;
    }
    
    /**
     * Indicates that the collector has terminated.
     * 
     * @return true if the collector has stopped
     */
    boolean hasStopped() {
        return stopped;
    }    
    
    /**
     * Indicates that the collector is busy.
     * 
     * @return true if the collector is running
     */
    boolean isRunning() {
        return started ^ stopped;
    }
    
    /**
     * Resets the cumulative log.
     */
    void resetLog() {
        log = "";
    }
    
    /**
     * Disables the cumulative log. Log output will go to stdout.
     */
    void disableLog() {
        log = null;
    }
    
    /**
     * Obtains the cumulative log content.
     * 
     * @return string containing the log
     */
    String getLog() {
        return log;
    }

    private void logprintln(Object obj) {
        logprint(obj);
        logprint("\r\n");
    }

    private void logprint(Object obj) {
        if (log == null)
            System.out.print(obj);
        else
            log += obj.toString();
    }

}

class FileStore implements Runnable {

    private ServerSocket socket = null;
    private Socket data = null;
    private InetAddress address = null;
    private int port = 0;
    private boolean started = false;
    private boolean stopped = false;
    private String log = "";
    private String filename;
    private boolean secure;

    FileStore(String file, ServerSocket port, boolean sec) {
        filename = file;
        socket = port;
        secure = sec;
    }

    FileStore(String file, InetAddress addr, int portnum, boolean sec) {
        filename = file;
        address = addr;
        port = portnum;
        secure = sec;
    }

    @Override
    public void run() {
        char[] bufr = new char[1400];
        started = true;

        try {
            FileReader file = new FileReader(filename);
        
            // obtain data socket by whatever means
            if (socket == null) 
                data = new Socket(address, port);
            else 
                data = socket.accept();
            
            // handle secure connection
            if (secure)
                data.setSecure(true);

            // transfer the file content
            BufferedWriter out = new BufferedWriter(new OutputStreamWriter(data.getOutputStream()));
            int nread;
            while ((nread = file.read(bufr)) > 0) {
                out.write(bufr, 0, nread);
                out.flush();
            }
            out.close();
            file.close();
            
            data.close();
            if (socket != null)
                socket.close();
 
        } catch (IOException ioe) {
            logprintln(ioe);
        }

        stopped = true;
    }

    /**
     * Indicates that the collector has started.
     * 
     * @return true if the collector was started
     */
    boolean hasStarted() {
        return started;
    }
    
    /**
     * Indicates that the collector has terminated.
     * 
     * @return true if the collector has stopped
     */
    boolean hasStopped() {
        return stopped;
    }    
    
    
    /**
     * Indicates that the collector is busy.
     * 
     * @return true if the transfer is running
     */
    boolean isRunning() {
        return started ^ stopped;
    }
    
    /**
     * Resets the cumulative log.
     */
    void resetLog() {
        log = "";
    }
    
    /**
     * Disables the cumulative log. Log output will go to stdout.
     */
    void disableLog() {
        log = null;
    }
    
    /**
     * Obtains the cumulative log content.
     * 
     * @return string containing the log
     */
    String getLog() {
        return log;
    }

    private void logprintln(Object obj) {
        logprint(obj);
        logprint("\r\n");
    }

    private void logprint(Object obj) {
        if (log == null)
            System.out.print(obj);
        else
            log += obj.toString();
    }
}

class FileRetrieve implements Runnable {

    private ServerSocket socket = null;
    private Socket data = null;
    private InetAddress address = null;
    private int port = 0;
    private boolean started = false;
    private boolean stopped = false;
    private String log = "";
    private String filename;
    private boolean secure;

    FileRetrieve(String file, ServerSocket port, boolean sec) {
        filename = file;
        socket = port;
        secure = sec;
    }

    FileRetrieve(String file, InetAddress addr, int portnum, boolean sec) {
        filename = file;
        address = addr;
        port = portnum;
        secure = sec;
    }

    @Override
    public void run() {
        started = true;
        char[] bufr = new char[8192];

        try {
            FileWriter file = new FileWriter(filename);
        
            // obtain data socket by whatever means
            if (socket == null) 
                data = new Socket(address, port);
            else 
                data = socket.accept();
            
            // handle secure connection
            if (secure)
                data.setSecure(true);

            // transfer the file content
            BufferedReader in = new BufferedReader(new InputStreamReader(data.getInputStream()));
            int nread;
            try {
                while ((nread = in.read(bufr)) > 0) 
                    file.write(bufr, 0, nread);
            } catch (IOException ioe) {
            }

            file.close();
            in.close();
            data.close();
            if (socket != null)
                socket.close();
 
        } catch (IOException ioe) {
            logprintln(ioe);
        }
        
        stopped = true;
    }

    /**
     * Indicates that the collector has started.
     * 
     * @return true if the collector was started
     */
    boolean hasStarted() {
        return started;
    }
    
    /**
     * Indicates that the collector has terminated.
     * 
     * @return true if the collector has stopped
     */
    boolean hasStopped() {
        return stopped;
    }    
    
    /**
     * Indicates that the collector is busy.
     * 
     * @return true if the transfer is running
     */
    boolean isRunning() {
        return started ^ stopped;
    }
    
    /**
     * Resets the cumulative log.
     */
    void resetLog() {
        log = "";
    }
    
    /**
     * Disables the cumulative log. Log output will go to stdout.
     */
    void disableLog() {
        log = null;
    }
    
    /**
     * Obtains the cumulative log content.
     * 
     * @return string containing the log
     */
    String getLog() {
        return log;
    }

    private void logprintln(Object obj) {
        logprint(obj);
        logprint("\r\n");
    }

    private void logprint(Object obj) {
        if (log == null)
            System.out.print(obj);
        else
            log += obj.toString();
    }
}