//	JNIOR Automation Network Operating System (JANOS)
//  CopyRight (C) 2012-2022 INTEG process group inc. All Rights Reserved.

/* ---------------------------------------------------------------------------
 * This software is INTEG process group inc proprietary. This source code
 * is for internal company use only and any other use without prior
 * consent of INTEG process group inc is prohibited.
 *
 * @author Bruce Cloutier
 * Inception Date: 
 * -------------------------------------------------------------------------*/

package com.integpg.system;

import com.integpg.sensor.SensorPort;
import java.io.IOException;
import java.util.Hashtable;

/**
 * Provides operating system information, configuration and operation.
 * 
 */
public class JANOS {
    
    /**
     * Class cannot be instantiated.
     */
    private JANOS() throws Exception {}

    /**
     * SYSLOG_EMERGENCY Optional RFC 5424 Severity Level specified in System Logging. This would be used
     * if the event potentially renders the JNIOR unusable. The severity level is only reported to a
     * REMOTE_SYSLOG facility and is otherwise ignored.
     */
    public static final int SYSLOG_EMERGENCY = 0;	// Emergency: system is unusable

    /**
     * SYSLOG_ALERT Optional RFC 5424 Severity Level specified in System Logging. This would indicate
     * the the event required some immediate action. The severity level is only reported to a
     * REMOTE_SYSLOG facility and is otherwise ignored.
     */
    public static final int SYSLOG_ALERT = 1;   	// Alert: action must be taken immediately

    /**
     * SYSLOG_CRITICAL Optional RFC 5424 Severity Level specified in System Logging. This would indicate
     * that the event affects some critical element of the system. The severity level is only reported to a
     * REMOTE_SYSLOG facility and is otherwise ignored.
     */
    public static final int SYSLOG_CRITICAL = 2;    // Critical: critical conditions

    /**
     * SYSLOG_ERROR Optional RFC 5424 Severity Level specified in System Logging. The event represents
     * an error. The severity level is only reported to a REMOTE_SYSLOG facility and is otherwise ignored.
     */
    public static final int SYSLOG_ERROR = 3;   	// Error: error conditions

    /**
     * SYSLOG_WARNING Optional RFC 5424 Severity Level specified in System Logging. The event is a warning.
     *  The severity level is only reported to a REMOTE_SYSLOG facility and is otherwise ignored.
     */
    public static final int SYSLOG_WARNING = 4; 	// Warning: warning conditions

    /**
     * SYSLOG_NOTICE Optional RFC 5424 Severity Level specified in System Logging. This indicates that the event
     * is a normal condition but one that might require some awareness. The severity level is only reported to a
     * REMOTE_SYSLOG facility and is otherwise ignored.
     */
    public static final int SYSLOG_NOTICE = 5;  	// Notice: normal but significant condition

    /**
     * SYSLOG_INFO Optional RFC 5424 Severity Level specified in System Logging. This indicates that the log
     * entry is for information purposes. The condition is normal. This is the default severity level. The 
     * severity level is only reported to a REMOTE_SYSLOG facility and is otherwise ignored.
     */
    public static final int SYSLOG_INFO = 6;    	// Informational: informational messages

    /**
     * SYSLOG_DEBUG Optional RFC 5424 Severity Level specified in System Logging. The log entry is for debugging.
     *  The severity level is only reported to a REMOTE_SYSLOG facility and is otherwise ignored.
     */
    public static final int SYSLOG_DEBUG = 7;   	// Debug: debug-level messages
    
    /**
     * JNIOR_SYSLOG Used in system logging to specify the destination of the log entry. The entry will be
     * made in the jniorsys.log file. Multiple destinations may be specified by OR'ing these flags. This is 
     * combined with REMOTE_SYSTEM by default.
     */
    public static final int JNIOR_SYSLOG = 1;       // send message to jniorsys.log file

    /**
     * REMOTE_SYSLOG Used in system logging to specify the destination of the log entry. The entry will be
     * transmitted to the remote logging facility. Multiple destinations may be specified by OR'ing these flags.
     * This is combined with JNIOR_SYSLOG by default. Note that an optional severity level may be
     * reported. The default is SYSLOG_INFO.
     */
    public static final int REMOTE_SYSLOG = 2;      // send message to remote syslog server

    /**
     * BOOT_SYSLOG Used in system logging to specify the destination of the log entry. The entry will be
     * made in the jniorboot.log file. Multiple destinations may be specified by OR'ing these flags.
     */
    public static final int BOOT_SYSLOG = 4;        // append message to jniorboot.log file
    
    /**
     * TAB_STD Accepts TAB as an input character from the console.
     */
    public static final int TAB_STD = 0;
    
    /**
     * TAB_AUTOFILL Performs auto-fill as it would from the command line.
     */
    public static final int TAB_AUTOFILL = 1;
    
    /**
     * TAB_ENTER Terminates input on TAB. TAB is equivalent to ENTER.
     */
    public static final int TAB_ENTER = 2;
    

//*************************************************************************
    // Boot
    
    /**
     * Retrieves the Power-On-Reset (POR) count. Each time JANOS boots the POR counter is advanced.
     *
     * @return power-on-reset count
     */
    public static native int getPOR();

    
    
    
    //*************************************************************************
    // Versioning

    /**
     * Retrieves the JANOS Firmware Version as a formatted string. The format is "v[MAJOR].[MINOR].[UPDATE]".
     *  If you are running a Release Candidate (rc), Beta (b) or Alpha (a) version of JANOS there will be additional
     *  information affixed to the version string following a dash '-'. This will include [BUILD].[COMPILE] values.
     *  For example "v1.0.8-rc2.1" indicating build 2 of a release candidate.
     *
     * @return version string formatted by the operating system
     * @see #getVersionNum() 
     */
    public static native String getVersion();

    /**
     * Retrieves the JANOS version information. When formatted a version typically appears as "v[MAJOR].[MINOR].[UPDATE]".
     *  If you are running a Release Candidate (rc), Beta (b) or Alpha (a) version of JANOS there will be additional
     *  information affixed to the version string following a dash '-' See {@link #getVersion() }. This will include 
     *  [BUILD].[COMPILE] values. For example "v1.0.8-rc2.1" indicating build 2 of a release candidate. The STATUS field 
     *  indicates the type of build where ALPHA=0, BETA=1, RELEASE CANDIDATE=2 and RELEASE=3. The BUILD and COMPILE 
     *  numbers are valid for all versions but not normally displayed with released versions.
     *
     * @return integer array containing the major, minor, update, status, build, compile values in that order.
     * @see #getVersion() 
     */
    public static native int[] getVersionNum();

    
    
    
    //*************************************************************************
    // Identification

    /**
     * Retrieves the unit's Serial Number.
     *
     * @return Returns the integer serial number or -1 if there is a problem.
     */
    public static native int getSerialNumber();

    
    
    
    //*************************************************************************
    // User

    /**
     * Returns legacy 128 always. Applications execute with administrator rights.
     *  Series 3 systems returns the permission level as user ID.
     *
     * @return legacy user ID
     * @deprecated Use {@link User#getCurrentUID()} if applications need to respond based upon
     * user permissions.
     */
    @Deprecated
    public static byte getCurrentUID() {};

    
    
    //*************************************************************************
    // Environment

    /**
     * Returns a copy of the current environment.  The environment contains parameters specific to each
     * process instance. The current environment is specific to the current process. This is different 
     * than the Registry which applies dynamically to all processes.
     *
     * @return Hashtable containing the current environment
     */
    public static Hashtable getCurrentEnvironment() {}

    /**
     * Gets the value of the given key from the current environment.
     *
     * @param key <code>String</code> containing a key to query
     *
     * @return The value of the key, or <code>null</code> if the key is not set in the current environment.
     */
    public static String getFromCurrentEnvironment(String key) {}

    /**
     * Sets the value of the key in the current environment.
     *
     * @param key <code>String</code> containing a key to set
     * @param value <code>String</code> containing the new value for the key
     *
     */
    public static void setInCurrentEnvironment(String key, String value) {}
    
    /**
     * Returns a copy of the system environment. The environment contains parameters specific to each
     * process instance. The system environment is that set of parameters defined generally as
     * default for each new process. This is different than the Registry which applies dynamically 
     * to all processes. An application program should refer to the current environment using
     * {@link #getCurrentEnvironment()}
     * @see #getCurrentEnvironment() 
     *
     * @return Hashtable containing the system environment
     */
    public static Hashtable getSystemEnvironment() {}
    
    private static Hashtable getenviron(int mode) {}
    private static native byte[] getCurrEnv(int mode);    
    private static native void setCurrEnv(String key, String value);
    
    
    //*************************************************************************
    // Command

    /**
     * Executes a command. This method provided limited interaction with the executing command. This should not be
     * used if access to the command output or status is required. It is recommend that {@link Runtime#exec} be used instead.
     *
     * @param commandLine The command line including any parameters needed for that command.
     * @param in Unused. Can be NULL.
     * @param out Unused. Can be NULL.
     * @param err Unused. Can be NULL.
     * @param env Unused. Can be NULL.
     * @param blocking Set true if we are to wait for command completion.
     */
    @Deprecated
    public static void execute(String commandLine, Object in,
            Object out, Object err, Object env, boolean blocking) {}
    private static native void exec(String cmd, boolean block);

    
    
    
    //*************************************************************************
    // Memory

    /**
     * Returns the amount of Heap memory remaining. Programs require an amount for free heap space
     * in order to execute reliably. This can be used to verify that a sufficient amount of memory
     * remains available for continued program execution.
     *
     * @return available free space in bytes
     */
    public static int getFreeRAM() {}
    private static native int getFreeRAMNative();

    /**
     * Returns the total amount of Heap memory.
     *
     * @return size of Heap memory.
     */
    public static int getTotalRAM() {}
    private static native int getTotalRAMNative();

    /**
     * Returns the amount of Flash memory remaining.
     *
     * @return available Flash space in bytes
     */
    public static int getFreeFlash() {}
    private static native int gFreeFlash();


    
    
    
    //*************************************************************************
    // Clock

    /**
     * Gets the number of milliseconds since system boot. This can be used for general timing.
     *
     * @return number of milliseconds since boot
     * @see java.lang.System#currentTimeMillis() 
     */
    public static native long uptimeMillis();

    /**
     * Returns the current timezone.
     *
     * @return The timezone abbreviation (without Daylight Savings aliasing)
     */
    public static native String getTimeZone();

    
    
    
    //*************************************************************************
    // Logging

    /**
     * Makes a timestamped entry in the jniorsys.log file. If configured the
     *  report is also sent to the remote syslog server with a SYSLOG_INFO (4)
     *  severity level.
     *
     * @param message <code>String</code> containing the text to be logged.
     */
    public static void syslog(String message) {}
    
    /**
     * Makes a timestamped entry in the jniorsys.log file. If configured the
     *  report is also sent to the remote syslog server with the specified
     *  severity level.
     *
     * @param message <code>String</code> containing the text to be logged.
     * @param level one of 8 SYSLOG severity levels
     */
    public static void syslog(String message, int level) {}
    
    /**
     * Makes a timestamped entry in either the jniorboot.log file or the
     *  jniorsys.log file if selected. If configured the report can also be
     *  sent to the remote syslog server with the specified severity level.
     *  Note that anything logged to jniorsys.log will also be transmitted
     *  to the remote syslog server if one is configured.
     *
     * @param message <code>String</code> containing the text to be logged.
     * @param level one of 8 SYSLOG severity levels
     * @param logs bitmapped log destination specifier
     */
    public static void syslog(String message, int level, int logs) {}
    
    private static native void logmessage(String message, int level, int logs);

    /**
     * Makes a timestamped entry in the specified file. If NULL then the string is
     *  directed to the jniorsys.log file.
     *
     * @param filename <code>String</code> specifying the destination file.
     * @param message <code>String</code> containing the text to be logged.
     */
    public static void logfile(String filename, String message) {}
    
    /**
     * Makes a timestamped entry in the specified file. If NULL then the string is
     *  directed to the jniorsys.log file. The message is also logged externally 
     *  using the appropriate SYSLOG level.
     *
     * @param filename <code>String</code> specifying the destination file.
     * @param message <code>String</code> containing the text to be logged.
     * @param level <code>int</code> defines SYSLOG level for remote logging.
     */
    public static native void logfile(String filename, String message, int level);
    
    
    //*************************************************************************
    // Email

    /**
     * Sends the defined email message. The email block must define a valid ToAddress.
     *
     * @param sEmailBlock valid Registry key string specifying the email block
     * @param sSubject Default subject line to use if none is defined by the email block.
     * @param sMessage Default message content to use if none is defined by the email block.
     */
    public static native void sendMail(String sEmailBlock, String sSubject, String sMessage);
    
    
    
    
    //*************************************************************************
    // IO

    /**
     * Use this method to determine if internal I/O has changed. The I/O signature will increment any time there is a change.
     * A comparison of two I/O signatures at two points in time if different will signify that there was a change in I/O.
     * This will not alert you about the nature of the change, only that has occurred.
     *
     * @return The I/O signature value
     */
    public static native long getIoSignature();

    /**
     * Obtains the state of all digital inputs as a single bit-mapped integer.
     *
     * @return An integer where the lease significant 8 bits each correspond to the digital input. A '1' is reported for inputs
     * that are ON and a '0' for those that are OFF. The least significant bit corresponds to DIN1. For the Model 410
     * bit D0-D7 of the returned value represent DIN1 through DIN8 respectively. For the Model 412 only bits D0-D3 are 
     * valid for channels DIN1 through DIN4.
     */
    public static native int getInputStates();

    /**
     * Obtains the state of all relay outputs as a single bit-mapped integer.
     * <p>
     * For the Model 410 the least significant 8 bits D0-D7 represent the state of ROUT1 through ROUT8. Bits D8-D11
     * will reflect the status of an option external 4ROUT module for relay outputs ROUT9 through ROUT12. A second
     * external 4ROUT module if present is represented by bits D12-D15 for ROUT13 through ROUT16.
     * <p>
     * For the Model 412 the least significant 12 bits D0-D11 represent the state of ROUT1 through ROUT12. Bits D12-D15
     * will reflect the status of an option external 4ROUT module for relay outputs ROUT13 through ROUT16.
     * <p>
     * Note that some relays may be hardware jumper configured for Normally-Closed operation. These are set to Normally-Open 
     * by default. If a relay is set for normally-closed operation a '1' will indicate the OPRN condition.
     * 
     * @return An integer where bits each correspond to relay outputs. A '1' is reported for outputs
     * that are CLOSED and a '0' for those that are OPEN. The least significant bit corresponds to ROUT1.
     */
    public static native int getOutputStates();

    /**
     * Establishes the condition for the specified output relays. The selected relays are changed to the indicated state.
     *
     * @param bits Integer where the bits indicate the desired output condition. A '1' resulting in a
     * closed relay
     * @param mask Integer where the indicate which output relays are to be affected by the command. Those relays 
     * selected by a '1' will change to the state indicated by the bits parameter. Those corresponding
     * with '0' bits will remain unchanged.
     * @throws java.io.IOException if port is open
     */
    public static void setOutputStates(int bits, int mask) throws IOException {}

    /**
     * Establishes a temporary condition for the specified output relays. The selected relays are changed to the indicated state
     * for a duration specified in milliseconds. When this time period passes the relays are returned to their original states.
     * This is used to pulse outputs with 1 millisecond resolution.
     *
     * @param bits Integer where the least significant eight bits indicate the desired output condition. A '1' resulting in a
     * closed relay
     * @param mask Integer where the least significant eight bits indicate which output relays are to be affected by the
     * command. Those relays selected by a '1' will change to the state indicated by the data parameter. Those corresponding
     * with '0' bits will remain unchanged.
     * @param milliseconds Integer specifying the duration that the new condition will be held. The pulse duration is to within
     * 1 millisecond +0.2/-0.8
     * @return Returns <code>true</code> if the pulse is successfully executed and <code>false</code> otherwise.
     * @throws java.io.IOException if port is open
     */
    public static boolean setOutputPulsed(int bits, int mask, int milliseconds) throws IOException {}

    /**
     * Obtains the state of a specified Digital Input.
     *
     * @return <code>true</code> if the physical Digital Input is ON. This indicates that there is more than 2VDC 
     * across the input terminals. The associated LED will be illuminated. Returns <code>false</code> otherwise.
     * <p>
     * The channel is 0-based with 0 specifying DIN1, 1 for DIN2, etc. The Model 410 uses channels 0-7 as it 
     * supports DIN1 through DIN8. The Model 412 has only 4 digital inputs which are specified as channels 0-3
     * 
     * @param chan 0-based value specifying a digital input channel
     */
    public static native boolean isInputSet(int chan);

    /**
     * Obtains the state of a specified Relay Output.
     * <p>
     * The channel is 0-based with 0 specifying ROUT1, 1 for ROUT2, etc. The Model 410 uses channels 0-7 as it 
     * supports ROUT1 through ROUT8. The Model 412 has 12 relay outputs which are specified as channels 0-11.
     * <p>
     * Additional relay outputs may be available if external 4ROUT modules are present. For the Model 410 two
     * external 4ROUT modules may be used. These can be addressed here as channels 8-15 representing ROUT9 through
     * ROUT16. The Model 412 can use a single external module providing channels 12-15 for ROUT13 through ROUT16.
     *
     * @param chan 0-based value specifying a relay output channel
     * @return <code>true</code> if the relay output is CLOSED. Returns <code>false</code> otherwise.
     */
    public static native boolean isOutputSet(int chan);

    /**
     * Establish the state of the specified output relay.
     * <p>
     * The channel is 0-based with 0 specifying ROUT1, 1 for ROUT2, etc. The Model 410 uses channels 0-7 as it 
     * supports ROUT1 through ROUT8. The Model 412 has 12 relay outputs which are specified as channels 0-11.
     * <p>
     * Additional relay outputs may be available if external 4ROUT modules are present. For the Model 410 two
     * external 4ROUT modules may be used. These can be addressed here as channels 8-15 representing ROUT9 through
     * ROUT16. The Model 412 can use a single external module providing channels 12-15 for ROUT13 through ROUT16.
     *
     * @param chan 0-based value specifying a relay output channel
     * @param state <code>true</code> closes the associated relay illuminating the corresponding LED. <code>false</code>
     * opens the relay. This opposite would be the case if a relay is hardware jumper configured for Normally-Closed 
     * operation.
     * @throws java.io.IOException if port is open
     */
    public static void setOutputRelay(int chan, boolean state) throws IOException {}

    /**
     * Obtains the latched state for the specified Digital Input.
     * <p>
     * The channel is 0-based with 0 specifying DIN1, 1 for DIN2, etc. The Model 410 uses channels 0-7 as it 
     * supports DIN1 through DIN8. The Model 412 has only 4 digital inputs which are specified as channels 0-3
     *
     * @param chan 0-based value specifying a digital input channel
     * @return Returns <code>true</code> if the digital input has been latched. The <code>isLatching</code> method 
     * is used to determine if the <code>isInputSet</code> or <code>isInputLatched</code> best describes the desired 
     * digital input status. The returned state reflects the latching process regardless of the latching 
     * configuration setting for the digital input.
     */
    public static native boolean isInputLatched(int chan);

    /**
     * Clears the latched state of a digital input.
     * <p>
     * The channel is 0-based with 0 specifying DIN1, 1 for DIN2, etc. The Model 410 uses channels 0-7 as it 
     * supports DIN1 through DIN8. The Model 412 has only 4 digital inputs which are specified as channels 0-3
     *
     * @param chan 0-based value specifying a digital input channel
     */
    public static native void resetLatchedInput(int chan);
    
    /**
     * Obtains the counter for the specified Digital Input.
     *
     * @return By default this returns the raw count of positive transitions on the selected input channel. If a SampleTime is
     * configured this returns the number of counts over the most recent period. If a Multiplier is defined then the counter
     * value is scaled. Returns -1 on error.
     * @param chan selected Digital Input (0-7)
     */
    public static native int getInputCounter(int chan);

    /**
     * Resets the counter for the specified Digital Input.
     *
     * @param chan selected Digital Input (0-7)
     */
    public static native void resetInputCounter(int chan);

    /**
     * Sets the counter for the specified Digital Input.
     *
     * @param chan selected Digital Input (0-7)
     * @param value desired value of the counter. 0 - 2^31
     */
    public static native void setInputCounter(int chan, int value);

    /**
     * Increments counter for the specified Digital Input.
     *
     * @param chan selected Digital Input (0-7)
     */
    public static native void incInputCounter(int chan);

    /**
     * Obtains the Usage for the specified Digital Input or Relay Output.
     *
     * @return By default this returns the raw count of milliseconds for the selected channel. This is the total time the input
     * is ON or the relay is CLOSED. Returns -1 on error.
     * @param chan selected Digital Input (0-7) or Relay Output (8-15) where 8 corresponds to Relay Output 1 and so forth.
     */
    public static native long getUsageMeter(int chan);

    /**
     * Resets the usage meter for the specified channel.
     *
     * @param chan selected Digital Input (0-7) or Relay Output (8-15) where 8 corresponds to Relay Output 1 and so forth.
     */
    public static native void resetUsageMeter(int chan);

    
    
    //*************************************************************************
    // Registry
    
    /**
     * Reads the primary string associated with the specified Registry key.
     *
     * @return The primary string associated with the Registry key. Each Registry entry can consist of numerous parameters
     * separated by commas. The primary string is the very first parameter and often the only parameter assigned to the Registry
     * entry. If there is no corresponding entry in the Registry then the default string is returned.
     * @param key valid Registry key string.
     * @param sDefault a default string that will be returned when the specified key does not exist.
     */
    public static native String getRegistryString(String key, String sDefault);

    /**
     * Returns the boolean state of the specified Registry key. Registry keys contain strings and often these may be interpreted
     * in a boolean sense where values such as "enabled", "true", "on", "yes" and "1" indicate a 'true' status. Any other
     * content indicates 'false'.
     *
     * @return the boolean interpretation of the Registry entry.
     * @param key the Registry key.
     * @param bDefault the default boolean condition to be returned when the Registry key is not defined.
     */
    public static native boolean getRegistryBoolean(String key, boolean bDefault);

    /**
     * Returns the integer value from the specified Registry key. Registry keys contain strings and often these may be
     * interpreted as an integer.
     *
     * @return Returns the integer value of the Registry entry or nDefault if the key is not defined or upon error.
     * @param key the Registry key.
     * @param nDefault the default integer value to be returned when the Registry key is not defined.
     */
    public static native int getRegistryInt(String key, int nDefault);

    /**
     * Returns the double value from the specified Registry key. Registry keys contain strings and often these may be
     * interpreted as a floating point value.
     *
     * @return Returns the double value of the Registry entry or dDefault if the key is not defined or upon error.
     * @param key the Registry key.
     * @param dDefault the default floating point (double) value to be returned when the Registry key is not defined.
     */
    public static native double getRegistryDouble(String key, double dDefault);
    
    /**
     * Reads the complete string array associated with the specified Registry key.
     *
     * @return The string array associated with the Registry key. Each Registry entry can consist of numerous parameters
     * separated by commas. The returned array contains each comma-separated field.
     * @param key valid Registry key string.
     */
    public static native String[] getRegistryKey(String key);

    /**
     * This lists the content of a node in the Registry tree.
     *
     * @return Returns a String[] listing of the contents of the specified Registry node. The returned strings represent either
     * the name of the key whose value is defined at this node or the leaf associated with nodes within additional substructure.
     * In the latter case the string will end with a '/' character. This would differentiate between a key defined at this node
     * and a leaf of the same name as both may legally exist in the Registry. A zero-length String[] is returned if the node
     * does not exist. A node cannot be empty since by definition it then does not exist.
     *
     * To then access the content of a key defined at the current node one must combine the Registry Key prefix supplied with
     * the returned name string to construct the complete Registry Key for the value.
     * @param key Registry Key prefix specifying the node to list. This is specified complete from the root of the Registry
     * (without prefix '/' or suffix '/').
     */
    public static native String[] getRegistryList(String key);

    /**
     * This method returns the last modified time of a given registry key or registry node. The registry node last modified time
     * will reflect the latest revision time of any of that node's immediate children.
     *
     * @param key <code>String</code> containing the registry key or node.
     * @return The UTC time as a <code>long</code> integer.
     */
    public static native long getRegistryLastModified(String key);

    /**
     * Defines the content of a Registry key.
     *
     * @return Returns 'true' if the Registry key was properly defined or redefined. A 'false' usually indicates that the
     * Registry key string is not valid.
     * @param key the Registry key.
     * @param value the new string to be assigned to the Registry key.
     */
    public static native boolean setRegistryString(String key, String value);

    /**
     * Defines the content of a Registry key.
     *
     * @return Returns 'true' if the Registry key was properly defined or redefined.
     * @param key the Registry key.
     * @param values a string array defining each comma-separated entry to be assigned to the Registry key.
     */
    public static native boolean setRegistryKey(String key, String[] values);

    
    
    
    //*************************************************************************
    // Configuration
    
    /**
     * Obtains the latching mode set for the specified Digital Input.
     * <p>
     * The channel is 0-based with 0 specifying DIN1, 1 for DIN2, etc. The Model 410 uses channels 0-7 as it 
     * supports DIN1 through DIN8. The Model 412 has only 4 digital inputs which are specified as channels 0-3
     *
     * @return Returns <code>true</code> if the digital input is configured for Latching.
     * @param chan 0-based value specifying a digital input channel
     */
    public static native boolean isLatching(int chan);

    /**
     * The digital input is processed unchanged.
     */
    public static final int INPUT_COND_NONE = 0;

    /**
     * The digital input is inverted. The input will be considered to be ON when no voltage is applied (input LED will be off).
     */
    public static final int INPUT_COND_INVERT = 1;

    /**
     * The digital input is forced to the OFF condition. The input will report the OFF condition regardless of the 
     * actual status (LED on/off).
     */
    public static final int INPUT_COND_0 = 2;

    /**
     * The digital input is forced to the ON condition. The input will report the ON condition regardless of the 
     * actual status (LED on/off).
     */
    public static final int INPUT_COND_1 = 3;

    /**
     * Retrieves the current input conditioning. In addition to the normal input function each individual input may be
     * inverted of forced to either a ON or OFF state.
     * <p>
     * The channel is 0-based with 0 specifying DIN1, 1 for DIN2, etc. The Model 410 uses channels 0-7 as it 
     * supports DIN1 through DIN8. The Model 412 has only 4 digital inputs which are specified as channels 0-3
     *
     * @param chan 0-based value specifying a digital input channel
     * @return INPUT_COND_NONE (0) - none, INPUT_COND_INVERT (1) - invert, INPUT_COND_0 (2) - OFF, 
     * INPUT_COND_1 (3) - ON.
     */
    public static native int getInputConditioning(int chan);

    /**
     * Sets the desired input conditioning. In addition to the normal input function each individual input may be
     * inverted of forced to either a ON or OFF state.
     * <p>
     * The channel is 0-based with 0 specifying DIN1, 1 for DIN2, etc. The Model 410 uses channels 0-7 as it 
     * supports DIN1 through DIN8. The Model 412 has only 4 digital inputs which are specified as channels 0-3
     * <p>
     * Note that a change in conditioning for an input causes an update to the monitor packet. This does not 
     * affect counters and usage meters.
     *
     * @param chan 0-based value specifying a digital input channel
     * @param mode INPUT_COND_NONE (0) - none, INPUT_COND_INVERT (1) - invert, INPUT_COND_0 (2) - OFF, 
     * INPUT_COND_1 (3) - ON.
     */
    public static native void setInputConditioning(int chan, int mode);


    
    //*************************************************************************
    // Externals
    
    /**
     * 
     * 
     * @param chan channel
     * @return value
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#getTemp(int)} instead.
     */
    @Deprecated
    public static double getTemp(int chan) throws IOException {}

    /**
     * This method is used to read the value of one 10 volt module input.
     *
     * @param chan the channel of the 10 volt input. You can have 2 external 10 volt modules giving you 8 inputs. The channel
     * must be 0 - 7.
     * @return a scaled value based on the range that was configured using the Main JNIOR Web Page
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#get10vIn(int) } instead.
     */
    @Deprecated
    public static double get10vIn(int chan) throws IOException {}

    /**
     * This method is used to read the value of one 10 volt module output.
     *
     * @param chan the channel of the 10 volt output. You can have 2 external 10 volt modules giving you 4 outputs. The channel
     * must be 0 - 3.
     * @return a scaled value based on the range that was configured using the Main JNIOR Web Page
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#get10vOut(int) } instead.
     */
    @Deprecated
    public static double get10vOut(int chan) throws IOException {}

    /**
     * Sets the value of the output of a 10v module at the given channel.
     *
     * @param chan channel 0 or 1
     * @param value percentage (0.0 to 100.0) of output level
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#set10vOut(int, double) } instead.
     */
    @Deprecated
    public static void set10vOut(int chan, double value) throws IOException {}

    /**
     * Reads one of the 4-20ma module channels.
     * 
     * @param chan the channel of the 4-20ma input. You can have 2 external 4-20ma modules giving you 8 inputs. The channel
     * must be 0 - 7.
     * @return  scaled value of the channel. Scaling can be set through Registry entries and configuration.
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#get420In(int) } instead.
     */
    @Deprecated
    public static double get420In(int chan) throws IOException {}

    /**
     * reads one of the 4-20ma module outputs.
     * 
     * @param chan  the channel of the 4-20ma output. You can have 2 external 4-20ma modules giving you 4 outputs. The channel
     * must be 0 - 3.
     * @return percentage (0.0 to 100.0) of the output level against full scale
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#get420Out(int) } instead.
     */
    @Deprecated
    public static double get420Out(int chan) throws IOException {}

    /**
     * Sets the value of the output of a 4-20ma module at the given channel.
     *
     * @param chan the channel of the 4-20ma input. You can have 2 external 4-20ma modules giving you 8 inputs. The channel
     * must be 0 - 7.
     * @param value percentage of full scale (0.0 to 100.0)
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#set420Out(int, double) } instead.
     */
    @Deprecated
    public static void set420Out(int chan, double value) throws IOException {}

    /**
     * This method will return the RTD value for the given channel.
     *
     * @param chan ?
     * @return ?
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#getRtdIn(int) } instead.
     */
    @Deprecated
    public static double getRtdIn(int chan) throws IOException {}

    /**
     * This method requests the enumeration of external devices. The addresses returned are used to read/write these devices.
     *
     * @return A long[] of device addresses.
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#getExternalDeviceList() } instead.
     */
    @Deprecated
    public static long[] getExternalDeviceList() throws IOException {}

    /**
     * This method requests the enumeration of external devices in the alarm state. The addresses returned are used to
     * read/write these devices.
     *
     * @return A long[] of device addresses.
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#getAlarmingDeviceList() } instead.
     */
    @Deprecated
    public static long[] getAlarmingDeviceList() throws IOException {}

    /**
     * Obtains the external device type from a device address.
     *
     * @param addr the device address as returned by enumeration.
     *
     * @return The lease significant byte of the device address which indicates the device type.
     * @deprecated Use {@link SensorPort#getDeviceType(long) } instead.
     */
    @Deprecated
    public static int getDeviceType(long addr) {}

    /**
     *
     * Obtain 16-bit raw values from a Type 20/FE/FD Quad Analog module.
     *
     * @return An int[] containing four values 0-65535 corresponding to input channels 1 through 4.
     * @param addr the device address as returned by enumeration. Note that full scale is 65520
     * for Type FD &amp; FE 12-bit modules.
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#readAnalogInputs(long) } instead.
     */
    @Deprecated
    public static int[] readAnalogInputs(long addr) throws IOException {}

    /**
     * Obtains the Read Device Block from the external device. The format is defined by the v2 JNIOR Communications Protocol.
     *
     * @return A byte[] containing the Read Device Block as defined for the specific device type in the v2 JNIOR Communications
     * Protocol.
     * @param addr the device address as obtained by enumeration.
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#readDeviceBlock(long) } instead.
     */
    @Deprecated
    public static byte[] readDeviceBlock(long addr) throws IOException {}
        
    /**
     * Writes the Write Device Block to the external device. The format is defined by the v2 JNIOR Communications Protocol.
     *
     * @param addr the device address as obtained by enumeration.
     * @param block the Write Device Block as defined in the v2 JNIOR Communications Protocol for the specific device type.
     * @throws java.io.IOException if port is open
     * @deprecated Use {@link SensorPort#writeDeviceBlock(long, byte[]) } instead.
     */
    @Deprecated
    public static void writeDeviceBlock(long addr, byte[] block) throws IOException {}

    //*************************************************************************
    // CRC
    
    /**
     * Perform CRC8 on an array of data elements based on the provided seed. CRC8 is based on the polynomial = X^8 + X^5 + X^4 +
     * 1. As used on the Dallas/Maxim 1-Wire bus
     *
     * @param data - byte array of data elements
     * @param offset - starting offset in the byte array
     * @param count - count of data elements to be included in the calculation
     * @param seed - initial CRC8 value
     * @return - CRC8 calculation
     */
    public static native int CRC8(byte[] data, int offset, int count, int seed);

    /**
     * Perform CRC16 on an array of data elements based on the provided seed. CRC16 is based on the polynomial = X^16 + X^15 +
     * X^2 + 1.
     *
     * @param data data byte array
     * @param offset starting offset in the byte array
     * @param count count of data elements to be included in the calculation
     * @param seed initial CRC16 value
     * @return CRC16 calculation
     */
    public static native int CRC16(byte[] data, int offset, int count, int seed);

    /**
     * Perform CRC32 on an array of data elements based on the provided seed.
     *
     * @param data data byte array
     * @param offset starting offset in the byte array
     * @param count count of data elements to be included in the calculation
     * @param seed - initial CRC32 value
     * @return CRC32 calculation
     */
    public static native int CRC32(byte[] data, int offset, int count, int seed);

    /**
     * Perform ADLER32 on an array of data elements based on the provided seed.
     *
     * @param data data byte array
     * @param offset starting offset in the byte array
     * @param count count of data elements to be included in the calculation
     * @param seed - initial ADLER32 value
     * @return ADLER32 calculation
     */
    public static native int ADLER32(byte[] data, int offset, int count, int seed);

    
    
    
    //*************************************************************************
    // Process
    
    /**
     * Returns the process ID for the current process.
     *
     * @return current process ID
     */
    public static native int getProcessID();

    /**
     * Obtain active process ID list.
     * 
     * @return array of process IDs
     */
    public static native int[] getProcessIDs();

    /**
     * Returns the process identification string for the process ID provided.
     *
     * @param procid process ID
     * @return associated process name
     */
    public static native String getProcessName(int procid);

    /**
     * Provides Process Table detail.
     * 
     * @return String[] with process information.
     */
    public static native String[] getProcessTable();

    /**
     * Used to determine if the process associated with the supplied ID is
     * actually running.
     * 
     * @param pid process id
     * @return true if it is running
     */
    public static native boolean isProcessRunning(int pid);

    /**
     * Terminates the process associated with the supplied ID.
     * 
     * @param pid process ID
     */
    public static native void killProcess(int pid);

    /**
    * Turns on and off the ability to interrupt the process using Ctrl-C. This
    * overrides any existence of a 'nobreak' parameter in the local environment.
    * 
     * @param enab true (default) if interrupt is to be allowed. false to disable.
    */
    public static native void setCtrlC (boolean enab);

    //*************************************************************************
    // LED
    
    /*
     * Indicates that the status LED is in use
     */
    /**
     *
     * @return ?
     */
    public static native boolean isbusyStatusLED();

    /**
     * Flashes that status LED (yellow/orange LED)
     *
     * Extends the flash sequence for the status LED. This issues a 250ms flash every 3s ONLY if the LED is otherwise inactive.
     */
    public static native void flashStatusLED();

    /**
     * Pulses the status LED
     *
     * @param ontime ?
     * @param offtime ?
     * @param count ?
     */
    public static native void pulseStatusLED(int ontime, int offtime, int count);

    /**
     * Display Morse Code message. Note that only characters A-Z (uppercase) and digits 0-9 are signaled. You define the number
     * of times that the message repeats.
     * <pre>
     * 0 --- --- --- --- --- 
     * 1 - --- --- --- --- 
     * 2 - - --- --- --- 
     * 3 - - - --- --- 
     * 4 - - - - --- 
     * 5 - - - - - 
     * 6 --- - - - - 
     * 7 --- --- - - - 
     * 8 --- --- --- - - 
     * 9 --- --- --- --- - 
     * 
     * A - --- 
     * B --- - - - 
     * C --- - --- - 
     * D --- - - 
     * E - 
     * F - - --- - 
     * G --- --- - 
     * H - - - - 
     * I - - 
     * J - --- --- --- 
     * K --- - --- 
     * L - --- - - 
     * M --- --- 
     * N --- - 
     * O --- --- --- 
     * P - --- --- - 
     * Q --- --- - --- 
     * R - --- - 
     * S - - - 
     * T --- 
     * U - - --- 
     * V - - - --- 
     * W - --- --- 
     * X --- - - --- 
     * Y --- - --- ---
     * Z --- --- - -
     * </pre>
     * @param msg ?
     * @param repeat ?
     */
    public static native void morseStatusLED(String msg, int repeat);

    

    //*************************************************************************
    // DEPRECATED
    
    /**
     * Defines the content of a System Registry key (those whose name starts with '$'). System Registry Keys are not preserved
     * through reboot.
     *
     * @param key the Registry key.
     *
     * @param value the new string to be assigned to the Registry key.
     * @deprecated Use {@link #setRegistryString(java.lang.String, java.lang.String) }. This will be removed in a
     *  future release.
     */
    @Deprecated
    public static void setSystemKey(String key, String value) {}

    /**
     * Returns the task table ID list.
     *
     * @return int array with size equal to number of tasks
     * @deprecated Use {@link #getProcessIDs() }. This will be removed in a future release.
     */
    @Deprecated
    public static int[] getTaskTableIDs() {}

    /**
     * Returns the task table.
     *
     * @return string array with size equal to number of tasks
     * @deprecated Use {@link #getProcessTable() }. This will be removed in a future release.
     */
    @Deprecated
    public static String[] getTaskTable() {}

    /**
     * Stops and removes task from system.
     *
     * @param taskID Task ID of task to kill
     *
     * @throws IllegalArgumentException if task does not exist.
     * @deprecated Use {@link #killProcess(int) }. This will be removed in a future release.
     */
    @Deprecated
    public static void killTask(int taskID) {}

    /**
     * Stops and removes task from system.
     *
     * @param taskID Task ID of task to kill
     * @param force false to request, true to demand kill.
     *
     * @throws IllegalArgumentException if task does not exist.
     * @deprecated Use {@link #killProcess(int) }. This will be removed in a future release.
     */
    @Deprecated
    public static void killTask(int taskID, boolean force) {}

    /**
     * Checks to see if task with specified ID is running
     *
     * @param taskid ID of task
     *
     * @return true if task is running
     * @deprecated Use {@link #isProcessRunning(int) }. This will be removed in a future release.
     */
    @Deprecated
    public static boolean isTaskRunning(int taskid) {}
    
    /**
     * Registers a GUID for the current process with the operating system.
     * 
     * Each process may have a GUID assigned. This method atomically assigns the provided
     * GUID to the current process and returns the number of processes with the given GUID.
     * If this number is not 1 then there is already 1 or more processes running with the 
     * same GUID. This can be used to force exclusive execution.
     * 
     * @param guid String of any length that is considered unique to the process
     * @return number of processes with the given GUID
     */
    public static native int registerProcess(String guid);
    
    /**
     * Checks for processes with the registered GUID.
     * 
     * Each process may have a GUID assigned. This method returns the number of processes with
     * the given GUID.
     * 
     * @param guid String of any length that is considered unique to the process
     * @return number of processes with the given GUID
     */
    public static native int countRegistered(String guid);

    /**
     * Identifies a process with the registered GUID.
     * 
     * Each process may have a GUID assigned. This method returns the Process IDs of processes with
     * the given GUID.
     * 
     * @param guid String of any length that is considered unique to a process
     * @return array of process IDs
     */
    public static native int[] getRegistered(String guid);
    
    /**
     * Invokes the command line entry routine for input.This provides consistent 
     * entry functionality for strings supporting the TAB auto-fill as well as 
     * Ctrl-U Unicode UTF8 capabilities.
     *
     * @param tabflag one of TAB_STD, TAB_ENTER or TAB_AUTOFILL
     * @return Entered string or the empty string.
     */
    public static native String readCommand(int tabflag);
}
