
/*---------------------------------------------------------------------------
 * Copyright (C) 1999,2000 Dallas Semiconductor Corporation, All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Dallas Semiconductor
 * shall not be used except as stated in the Dallas Semiconductor
 * Branding Policy.
 *---------------------------------------------------------------------------
 */

package com.integpg.io;



import com.integpg.system.JANOS;
import java.util.*;
import java.io.CharConversionException;


/**
 * <p>This class defines an interface to allow conversion of bytes
 * to characters for a particular encoding scheme.  Encoding converters
 * should reside in the <code>com.integpg.io</code> package.</p>
 *
 * <p>Many of the encoding schemes need to take state into account in the
 * conversion process.  That is, the conversion to a char might depend on
 * the byte sequence converted before it.  To accommodate this, the
 * ByteToCharConverter has the ability to remember state between conversions
 * (between calls to <code>convert()</code>.  Therefore, the caller
 * should call the <code>flush()</code> method to finalize the conversion
 * and reset the converter's internal state.</p>
 *
 * <p>Subclasses of this abstract class need to implement <code>getMaxCharCount()</code>,
 * <code>convert()</code>, <code>flush()</code>, and <code>getName()</code>.</p>
 *
 * <p>Programs should not call into a converter directly.  A
 * better method of executing <code>byte</code> array conversions
 * is to use the <code>java.lang.String(byte[],String)</code> constructor.</p>
 *
 * <pre><code>
 *      ...
 *      //byte[] preConvertedBytes is previously declared and
 *      //has a sequence of UTF8 encoded bytes
 *      String str = new String(preConvertedBytes,"UTF8");
 *      ...
 * </code></pre>
 * <p>This will convert the bytes stored in <code>preConvertedBytes</code>
 * into a <code>String</code> according to the UTF8 encoding scheme.</p>
 *
 * @see CharToByteConverter
 */
public abstract class ByteToCharConverter
{
   private static Hashtable<String, ByteToCharConverter> converters  = null;
   private static String    prepend     = null;
   private static String    defaultName = null;

   /**
    * Dynamically loads a <code>ByteToCharConverter</code> for the specified encoding scheme.
    * All converters should be placed in the <code>com.integpg.io package</code>, and have class
    * name ByteToCharNAME, where NAME is the encoding scheme.  For example, the UTF8
    * <code>ByteToCharConverter</code> is called <code>com.integpg.io.ByteToCharUTF8</code>.
    *
    * @param name the name of the encoding scheme
    *
    * @return converter for the specified encoding scheme, or <code>null</code>
    * if none could be found
    */
   public static ByteToCharConverter getConverter (String name)
   {}

   /**
    * Returns the default <code>ByteToCharConverter</code> for the system.  The name
    * of the default encoding scheme is stored in the system property
    * "file.encoding".  This method finds the name of the default
    * encoding scheme, and calls <code>getConverter()</code> with that name
    * as its argument.
    *
    * @return converter for the system's default file encoding property,
    * or <code>null</code> if the converter could not be found
    *
    * @see #getConverter(java.lang.String)
    */
   public static ByteToCharConverter getDefaultConverter ()
   {}

   /**
    * Returns the number of characters that the specified <code>byte</code>
    * sequence will require for encoding.  For instance, in UTF8
    * encoding, a one, two, or three byte sequence may encode to
    * one <code>char</code>.  This method should always be called before the
    * <code>convert()</code> method.  The value returned may not be the actual
    * number of converted characters that will be produced due to
    * conversion errors, but it will be the maximum that will be
    * produced.
    *
    * @param forThis the byte sequence to determine the
    * required encoding size
    * @param start offset into the byte array to begin processing
    * @param end the ending offset in the byte array to stop processing.
    * The number of processed bytes will then be <code>(end-start)</code>.
    *
    * @return The number of bytes required to encode the specified <code>byte</code> sequence
    *
    * @see #convert(byte[],int,int,char[],int,int)
    */
   public abstract int getMaxCharCount (byte[] forThis, int start, int end);

   /**
    * Converts the specified <code>byte</code> array into a <code>char</code> array based on
    * this <code>ByteToCharConverter</code>'s encoding scheme.  <code>getMaxCharCount()</code> should
    * always be called first to find out how much room is required in the
    * destination <code>char</code> array.
    *
    * @param src the same byte array passed to <code>getMaxCharCount()</code>
    * @param srcStart the same starting offset as passed to <code>getMaxCharCount()</code>
    * @param srcEnd the same ending offset as passed to <code>getMaxCharCount()</code>
    * @param dst the destination character array.
    * @param dstStart the offset to begin storing converted bytes in the
    * destination array
    * @param dstEnd the ending location for storing converted bytes into the
    * destination array.  This argument may usually be ignored, as
    * the algorithm may choose to continue converting bytes until
    * finished.
    *
    * @return number of characters created and stored from this character sequence
    *
    * @exception java.io.CharConversionException if an illegal byte sequence is encountered
    * that cannot be converted
    *
    * @see #getMaxCharCount(byte[],int,int)
    * @see #flush(char[],int,int)
    */
   public abstract int convert (byte[] src, int srcStart, int srcEnd,
                                char[] dst, int dstStart, int dstEnd)
      throws CharConversionException;

   /**
    * Tells the <code>ByteToCharConverter</code> to convert
    * any unconverted data it has internally stored.
    * Some <code>ByteToCharConverter</code>'s will store state between
    * calls to <code>convert()</code>.  Since the converter may be left in
    * an unknown state, the converter should be flushed to
    * notify it that no more input will be received.  The converter
    * can handle any unfinished conversions before its output is
    * used.
    *
    * @param buff the destination character array
    * @param start the next available offset into the destination array
    * @param end offset in the destination array to stop placing data
    * (may be ignored by some algorithms)
    *
    * @return number of bytes that were stored in the destination array
    * from this call to <code>flush()</code>
    *
    * @exception java.io.CharConversionException if an illegal character is encountered that cannot be converted
    *
    * @see #convert(byte[],int,int,char[],int,int)
    */
   public abstract int flush (char[] buff, int start, int end)
      throws CharConversionException;

   /**
    * Returns the name of this encoding scheme.  For example, "UTF8".
    *
    * @return String representing the name of this encoding scheme.
    */
   public abstract String getName ();
}
