//	JNIOR Automation Network Operating System (JANOS)
//  CopyRight (C) 2012-2022 INTEG process group inc. All Rights Reserved.

/* ---------------------------------------------------------------------------
 * This software is INTEG process group inc proprietary. This source code
 * is for internal company use only and any other use without prior
 * consent of INTEG process group inc is prohibited.
 *
 * @author Bruce Cloutier
 * Inception Date: 
 * -------------------------------------------------------------------------*/

package com.integpg.system;

/**
 * This class implements methods to manage arrays. This includes comparison, copying, filling and inserting and removing simple
 * data types into byte arrays. Simple data types are stored in byte arrays in big-endian format.
 */
public class ArrayUtils {

    /**
     * Copies <code>length</code> elements from the <code>src</code> array at index <code>src_position</code> to the
     * <code>dst</code> array at index <code>dst_position</code>. The <code>src</code> and <code>dst</code> arrays must be of
     * the same type. This is similar to <code>java.lang.System.arraycopy()</code>, but more restrictive. This allows it to be
     * slightly faster, and calls one less method than the same array copy performed through the <code>java.lang.System</code>
     * class.
     *
     * @param src source array
     * @param src_position starting index in source array
     * @param dst target array
     * @param dst_position starting index in target array
     * @param length number of elements to copy
     *
     * @return 0 under normal operation. If both <code>dst</code> and <code>src</code> are arrays of references, and are not the
     * same type, returns -1.
     */
    public static native int arraycopy(Object src, int src_position,
            Object dst, int dst_position,
            int length);

    /**
     * Copies <code>length</code> common lower element bytes from <code>fromArray</code> at <code>fromOffset</code> into
     * <code>toArray</code> at <code>toOffset</code>. <br>
     * If the destination element has more bytes than the source element, the upper bytes of each element are set to zero. <br>
     * If the destination element has less bytes than the source element, the upper bytes are truncated. <br>
     *
     * @param fromArray source array
     * @param fromOffset starting index in source array
     * @param toArray target array
     * @param toOffset starting index in target array
     * @param length number of elements to copy
     */
    public static native void arrayCopyUnequal(Object fromArray, int fromOffset,
            Object toArray, int toOffset,
            int length);

    /**
     * Compares <code>length</code> elements in <code>array1</code> at <code>offset1</code> with the elements in
     * <code>array2</code> at <code>offset2</code>.
     *
     * @param array1 source array
     * @param offset1 starting index in source array
     * @param array2 target array
     * @param offset2 starting index in target array
     * @param length number of elements to compare
     *
     * @return false if all elements are not the same
     */
    public static native boolean arrayComp(Object array1, int offset1,
            Object array2, int offset2,
            int length);

    /**
     * Fills <code>thisArray</code> from <code>fromIndex</code> to <code>toIndex</code> with <code>fillValue</code>.
     *
     * @param thisArray target array
     * @param fromIndex starting index for fill, inclusive
     * @param toIndex ending index for fill, exclusive
     * @param fillValue value to fill array
     */
    public static native void arrayFill(byte[] thisArray, int fromIndex,
            int toIndex, byte fillValue);

    /**
     * Writes a short value <code>value</code> to byte array <code>thisArray</code> starting at offset <code>offset</code>. The
     * values are stored in the array in Big Endian form.
     *
     * @param thisArray array that will be modified
     * @param offset starting offset
     * @param value value to be written
     *
     * @return array reference to <code>thisArray</code>
     */
    public static native byte[] setShort(byte[] thisArray, int offset,
            short value);

    /**
     * Writes an int value <code>value</code> to byte array <code>thisArray</code> starting at offset <code>offset</code>. The
     * values are stored in the array in Big Endian form.
     *
     * @param thisArray array that will be modified
     * @param offset starting offset
     * @param value value to be written
     *
     * @return array reference to <code>thisArray</code>
     */
    public static native byte[] setInt(byte[] thisArray, int offset,
            int value);

    /**
     * Writes a long value <code>value</code> to byte array <code>thisArray</code> starting at offset <code>offset</code>. The
     * values are stored in the array in Big Endian form.
     *
     * @param thisArray array that will be modified
     * @param offset starting offset
     * @param value value to be written
     *
     * @return array reference to <code>thisArray</code>
     */
    public static native byte[] setLong(byte[] thisArray, int offset,
            long value);

    /**
     * Reads a short value from a byte array. The values extracted are assumed to be in Big Endian Form.
     *
     * @param thisArray array containing the value
     * @param offset starting offset
     *
     * @return value read
     */
    public static native short getShort(byte[] thisArray, int offset);

    /**
     * Reads an int value from a byte array. The values extracted are assumed to be in Big Endian Form.
     *
     * @param thisArray array containing the value
     * @param offset starting offset
     *
     * @return value read
     */
    public static native int getInt(byte[] thisArray, int offset);

    /**
     * Reads a long value from a byte array. The values extracted are assumed to be in Big Endian Form.
     *
     * @param thisArray array containing the value
     * @param offset starting offset
     *
     * @return value read
     */
    public static native long getLong(byte[] thisArray, int offset);

    /**
     * Reverses the byte order of a value. A little endian value is returned as big endian (system default).
     * A system default big endian value is returned in little endian representation. This is very useful
     * in dealing with protocol byte order differences.
     *
     * @param value short value (2 bytes)
     * @return short value with byte order reversed
     */
    public static short swapEndian(short value) {}
    private static native int sendian(int value);

    /**
     * Reverses the byte order of a value. A little endian value is returned as big endian (system default).
     * A system default big endian value is returned in little endian representation. This is very useful
     * in dealing with protocol byte order differences.
     *
     * @param value int value (4 bytes)
     * @return int value with byte order reversed
     */
    public static int swapEndian(int value) {}
    private static native int iendian(int value);

    /**
     * Reverses the byte order of a value. A little endian value is returned as big endian (system default).
     * A system default big endian value is returned in little endian representation. This is very useful
     * in dealing with protocol byte order differences.
     * 
     * @param value long value (8 bytes)
     * @return long value with byte order reversed
     */
    public static long swapEndian(long value) {}
    private static native long lendian(long value);

    /* --- private section --- */
    private ArrayUtils() {}

}
