package com.integ.common.logging;

import com.integ.common.system.Application;
import com.integ.common.utils.ExceptionUtils;
import com.integpg.system.JANOS;
import java.text.QuickDateFormat;

public class AppLog {

//    private static final boolean REQUIRED_OS_VERSION = MinimumVersion.atLeast("SysLog", "1.3-b10");
    private static QuickDateFormat DateFormat = new QuickDateFormat("MM/dd/yy HH:mm:ss zzz");

    private static Logger _log;
    private static Logger _exceptionLog;
//    private static boolean _warnSysLog = true;

    /**
     * We only want to log that an error has occurred to the application log file and that the
     * errors log should be
     * checked. This way multiple errors do no fill up the application log file.
     */
    private static boolean _logErrorOccured = true;



    private static synchronized void ensureLogFile() {
        try {
            String appName = Application.getAppInstanceName(); //JANOS.getProcessName(JANOS.getProcessID());
            if (_log == null && appName != null) {
                _log = RollingFileLog.getLog(new LogOptions(appName + ".log"));
            }
            if (_exceptionLog == null && appName != null) {
                _exceptionLog = RollingFileLog.getLog(new LogOptions(appName + "_exception.log"));
            }
        } catch (Exception ex) {
            ex.printStackTrace();

            StringBuilder sb = new StringBuilder();
            sb.append(DateFormat.format(System.currentTimeMillis()));
            sb.append(", ");
            sb.append(ex.toString());
            System.err.println(sb.toString());
        }
    }



    public static void setDateFormat(String dateFormat) {
        if (!dateFormat.equals(DateFormat.getDateFormatString())) {
            info("Change applog date format to " + dateFormat);
            DateFormat = new QuickDateFormat(dateFormat);
        }
    }



    public static Logger getLog() {
        return _log;
    }



    public static void setAppLog(Logger log) {
        _log = log;
    }



    public static void setAppExceptionLog(Logger log) {
        _exceptionLog = log;
    }



    public static void fatal(String line) {
        ensureLogFile();
        if (_log != null) _log.fatal(line);
        JANOS.syslog(line, JANOS.SYSLOG_CRITICAL, JANOS.JNIOR_SYSLOG);
    }



    public static void warn(String line) {
        ensureLogFile();
        if (_log != null) _log.warn(line);
        sendToSyslogServer(line, JANOS.SYSLOG_WARNING);
    }



    public static void error(String error) {
        ensureLogFile();
        _log.error(error);
        sendToSyslogServer(error, JANOS.SYSLOG_ERROR);
    }



    public static void error(Throwable ex) {
        String stackTrace = ExceptionUtils.getStackTrace(ex);

        ensureLogFile();
        anErrorHasOccuredEntry();
        if (_exceptionLog != null) {
            _exceptionLog.error(stackTrace);
        } else {
            ex.printStackTrace(System.out);
        }
        sendToSyslogServer(ex.toString(), JANOS.SYSLOG_ERROR);
    }



    public static void error(String description, Throwable ex) {
        String stackTrace = ExceptionUtils.getStackTrace(ex);

        ensureLogFile();
        anErrorHasOccuredEntry();
        if (_exceptionLog != null) {
            _exceptionLog.error(description + ": " + stackTrace);
        } else {
            ex.printStackTrace(System.out);
        }
        sendToSyslogServer(description + ": " + ex.toString(), JANOS.SYSLOG_ERROR);
    }



    private static void anErrorHasOccuredEntry() {
        if (null != _log) {
            synchronized (_log) {
                if (_log != null && _logErrorOccured) {
                    _log.error("An ERROR has occured.  Please check the application exception log");
                    _logErrorOccured = false;
                }
            }
        }
    }



    private static void sendToSyslogServer(String message, int level) {
//        if (REQUIRED_OS_VERSION) {
        JANOS.syslog(message, level, JANOS.REMOTE_SYSLOG);
//        } else {
//            if (_warnSysLog) {
//                _log.info("!! Sending to the syslog server requires v1.3 or later");
//                _warnSysLog = false;
//            }
//        }
    }



    public static void debug(String line) {
        ensureLogFile();
        if (_log != null) _log.debug(line);

        sendToSyslogServer(line, JANOS.SYSLOG_INFO);
    }



    public static void info(String line) {
        ensureLogFile();
        if (_log != null) _log.info(line);

        sendToSyslogServer(line, JANOS.SYSLOG_INFO);
    }



    public static void vital(String line) {
        ensureLogFile();
        if (_log != null) _log.vital(line);

        sendToSyslogServer(line, JANOS.SYSLOG_INFO);
    }

}

