package com.integ.common.collections;

import java.io.IOException;

/**
 * Implementation of a simple FIFO circular queue
 *
 * @author kcloutier
 */
public class CircularQueue<E> {

    private final Object[] _items;
    private final int _capacity;

    private int _head = 0;
    private int _tail = 0;



    /**
     * Instantiates a circular queue with specified capacity
     *
     * @param capacity
     */
    public CircularQueue(int capacity) {
        _capacity = capacity;
        _items = new Object[capacity];
    }



    /**
     * is the queue full?
     *
     * @return
     */
    public boolean isFull() {
        synchronized (_items) {
            if (_head > _tail) return (_head - _tail) == (_capacity - 1);
            else return _tail == _head + 1;
        }
    }



    /**
     * is the queue empty?
     *
     * @return
     */
    public boolean isEmpty() {
        synchronized (_items) {
            return _head == _tail;
        }
    }



    /**
     * the number of elements in the queue
     *
     * @return
     */
    public int size() {
        synchronized (_items) {
            if (_head >= _tail) return _head - _tail;
            else return _capacity - _tail + _head;
        }
    }



    /**
     * the capacity of the queue
     *
     * @return
     */
    public int capacity() {
        return _capacity;
    }



    /**
     * adds an object to the queue. if the queue is full we pluck one off the tail to make room.
     *
     * @param object
     * @throws IOException when the queue is full.
     */
    public void enqueue(E object) throws IOException {
        synchronized (_items) {
            if (isFull()) {
                dequeue();
            }

            _items[_head++] = object;
            if (_head >= _capacity) _head = 0;
        }
    }



    /**
     * removes and returns an object form the queue
     *
     * @return
     * @throws IOException if the queue is empty
     */
    public E dequeue() throws IOException {
        if (!isEmpty()) {
            synchronized (_items) {
                Object o = _items[_tail++];
                if (_tail >= _capacity) _tail = 0;
                return (E) o;
            }
        } else throw new IOException("Queue is empty");
    }



    /**
     * returns the first object from the queue. The object is not removed.
     *
     * @return
     * @throws IOException if the queue is empty
     */
    public E peek() throws IOException {
        if (!isEmpty()) {
            synchronized (_items) {
                return (E) _items[_tail];
            }
        }
        throw new IOException("Queue is empty");
    }
}

