package com.integ.common.system;

import com.integ.common.CommonLib;
import com.integ.common.logging.AppLog;
import com.integ.common.utils.FileUtils;
import com.integpg.system.JANOS;
import java.io.File;
import java.io.IOException;
import java.text.QuickDateFormat;
import java.util.ArrayList;
import java.util.Json;

public class Application {

    private static AssemblyBase _assemblyBase;
    private static String _instanceName = null;
    private static String _registryRoot = "";
    private static boolean _isBackground = false;



    // 2.0
    public static void init(AssemblyBase assemblyInfo) {
        init(assemblyInfo, null);
    }



    // 2.0
    public static void init(AssemblyBase assemblyInfo, String instanceName) {
        _assemblyBase = assemblyInfo;
        _instanceName = instanceName;

        _registryRoot = "AppData/" + _assemblyBase.getName() + "/";
        if (null != _instanceName) _registryRoot += _instanceName + "/";

        JANOS.setRegistryString(_registryRoot + "$Version", _assemblyBase.getVersion());
        JANOS.setRegistryString(_registryRoot + "$BuildTime", _assemblyBase.getBuildTime());
        JANOS.setRegistryString(_registryRoot + "$CommonLibraryVersion", CommonLib.getVersion());
        JANOS.setRegistryString(_registryRoot + "$CommonLibraryBuildTime", CommonLib.getBuildTime());

        _isBackground = setIsBackground();

        Application.logBanner();

        try {
            Json releaseNotesJson = getReleaseNotesJson(assemblyInfo);
            if (null != releaseNotesJson) {
                FileUtils.writeAllBytes(Application.getAppName().toLowerCase() + "_releasenotes.json", releaseNotesJson.toString().getBytes());
            }
        } catch (Exception ex) {
            AppLog.warn("error writing release notes");
            ex.printStackTrace();
        }
    }



    public static String getAppName() {
        return _assemblyBase.getName();
    }



    public static String getAppInstanceName() {
        String name = _assemblyBase.getName();
        if (null != _instanceName) name += "-" + _instanceName;
        return name;
    }



    public static String getAppVersion() {
        return _assemblyBase.getVersion();
    }



    public static String getInstanceName() {
        return _instanceName;
    }



    public static String getUuid() {
        return _assemblyBase.getUUID();
    }



    public static String getRegistryRoot() {
        return _registryRoot;
    }



    public static boolean isBackground() {
        return _isBackground;
    }



    /**
     * this method will use the process table to determine if this process is running in the
     * background or the
     * foreground. the logic to determine this is as follows:
     * 1. a process that starts with 'Run' was executed by the OS during boot and is running in the
     * background
     * 2. a process parent starts with something other than 'Command' was executed by another
     * executing process
     * and therefore executed in the background
     * 3. we run though the process table and look at the other process names. if the process parent
     * is the
     * same as one of the other processes then it was executed by a user at the command line with
     * the '&'
     * parameter and therefore running in the background.
     * 4. if all of the above fail then the process is running in the foreground.
     *
     * @since 5.3
     */
    private static boolean setIsBackground() {
        int thisProcessId = JANOS.getProcessID();
        String thisProcessName = JANOS.getProcessName(thisProcessId);
        boolean isBackground = thisProcessName.startsWith("Run");

        if (!isBackground) {
            int lastSlashPos = thisProcessName.indexOf("/", 8);
            String thisProcessParent = thisProcessName.substring(0, lastSlashPos);

            if (!thisProcessParent.startsWith("Command")) {
                isBackground = true;
            } else {
                int[] processIds = JANOS.getProcessIDs();
                for (int i = 0; i < processIds.length && !isBackground; i++) {
                    int processId = processIds[i];
                    if (thisProcessId != processId) {
                        String processName = JANOS.getProcessName(processId);
                        if (processName.length() >= lastSlashPos) {
                            isBackground = thisProcessParent.equals(processName);
                        }
                    }
                }
            }
        }

        return isBackground;
    }



    // 5.0
    public static String printReleaseNotes(AssemblyBase assemblyBase) {
        StringBuilder sb = new StringBuilder();

        sb.append("Version: " + assemblyBase.getVersion() + "\r\n\r\n");

        ReleaseInfo[] releaseNotes = AssemblyBase.getReleaseNotes();
        if (null != releaseNotes) {
            for (ReleaseInfo releaseInfo : releaseNotes) {
                String releaseInfoString = releaseInfo.getVersion() + "          ";
                releaseInfoString = releaseInfoString.substring(0, 10) + releaseInfo.getDate();
                sb.append(releaseInfoString + "\r\n");

                for (String note : releaseInfo.getReleaseNotes()) {
                    // break up long lines
                    int maxLineLength = 90;
                    for (int pos = 0, endPos = 0; pos < note.length(); pos = endPos) {
                        endPos += maxLineLength;

                        // make sure we dont go beyond the end of the stirng
                        if (note.length() < endPos) {
                            endPos = note.length();
                        } else {
                            // find the last space before our max line length.  If there isnt one then we use 
                            // the previous calculated endPos
                            int lastSpacePos = note.lastIndexOf(" ", endPos);
                            if (pos < lastSpacePos) endPos = lastSpacePos + 1;
                        }

                        if (0 == pos) sb.append("        * ");
                        else sb.append("          ");
                        sb.append(note.substring(pos, endPos) + "\r\n");
                    }
                }
                sb.append("\r\n");
            }
        }

        return sb.toString();
    }



    private static Json getReleaseNotesJson(AssemblyBase assemblyBase) {
        if (null != assemblyBase.getReleaseNotes()) {
            Json json = new Json();
            json.put("CurrentVersion", assemblyBase.getVersion());
            json.put("BuildTime", assemblyBase.getBuildTime());

            ArrayList<Json> releaseInfoJsonArray = new ArrayList<>();
            System.out.println("release notes: " + assemblyBase.getReleaseNotes());
            for (ReleaseInfo releaseInfo : assemblyBase.getReleaseNotes()) {
                Json releaseInfoJson = new Json();
                releaseInfoJson.put("Version", releaseInfo.getVersion());
                releaseInfoJson.put("Date", releaseInfo.getDate());
                releaseInfoJson.put("Notes", (String[]) releaseInfo.getReleaseNotes().toArray());
                releaseInfoJsonArray.add(releaseInfoJson);
            }
            json.put("ReleaseNotes", (Json[]) releaseInfoJsonArray.toArray());

            return json;
        }
        return null;
    }



    private static void logBanner() {
        String versionString = getAppInstanceName() + " v" + _assemblyBase.getVersion();
        boolean isForeground = !_isBackground;

        AppLog.vital(versionString);
        if (!_assemblyBase.getBuildTime().equals("")) {
            AppLog.vital(" build-time: " + _assemblyBase.getBuildTime());
        }
        AppLog.vital(" using Common Lib v" + CommonLib.getVersion());
        AppLog.vital(" running on JANOS " + JANOS.getVersion());
        AppLog.vital(String.format(" run as process %d%s",
                JANOS.getProcessID(), (isForeground ? " in FOREGROUND" : "")));
        AppLog.vital("Serial Number: " + UnitConfig.SERIAL_NUMBER + ", Model: " + UnitConfig.MODEL);

        JANOS.syslog(String.format("-- %s%s built %s",
                versionString, (isForeground ? " (foreground)" : ""),
                _assemblyBase.getBuildTime()));

        updateBootLog();
    }



    // 5.0
    private static void updateBootLog() {
        try {
            QuickDateFormat quickDateFormat = new QuickDateFormat("MM/dd/yy HH:mm:ss zzz");

            String entry = String.format("%s, %s v%s (built %s) using JanosLib v%s (built %s) running on JANOS %s\r\n",
                    quickDateFormat.format(System.currentTimeMillis()),
                    getAppInstanceName(), _assemblyBase.getVersion(), _assemblyBase.getBuildTime(),
                    CommonLib.getVersion(), CommonLib.getBuildTime(),
                    JANOS.getVersion());

            File bootlogFile = new File(getAppName().toLowerCase() + "_boot.log");
            if (16 * 1024 < bootlogFile.length() + entry.length()) {
                File bakFile = new File(bootlogFile.getPath() + ".bak");
                if (bakFile.exists()) {
                    bakFile.delete();
                }
                bootlogFile.renameTo(bakFile);
            }
            FileUtils.appendAllBytes(bootlogFile.getPath(), entry.toString().getBytes());
        } catch (IOException ex) {
            ex.printStackTrace(System.err);
        }
    }



    // 2.0
    public static void enableApplicationWatchdog(int watchdogAction) {
        ApplicationWatchdog applicationWatchdog = new ApplicationWatchdog(watchdogAction);
        applicationWatchdog.start();
    }



    public static void sleepForever() {
        try {
            System.out.println("sleep forever");
            Thread.sleep(Integer.MAX_VALUE);
        } catch (InterruptedException ex) {
            ex.printStackTrace();
        }
    }



    public static void exit(int status) {
        System.exit(status);
    }
}

