package com.integ.common.logging;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * used when buffering full lines of data
 */
public class BufferedLineOutputStream extends BufferedOutputStream {

    /**
     * overridden constructor to use defined OutputStream as underlying stream
     *
     * @param out
     */
    public BufferedLineOutputStream(OutputStream out) {
        super(out);
    }



    /**
     * overridden constructor to use defined OutputStream as underlying stream where our buffer size is the given size.
     *
     * @param out
     * @param size
     */
    public BufferedLineOutputStream(OutputStream out, int size) {
        super(out, size);
    }



    /**
     * overrides the BufferedOutputStream write method. We only want to commit full lines to our logs. At this point
     * this method ASSUMES that only full lines are being written. So if the new line would fill the buffer, the buffer
     * is committed before the new line is buffered.
     *
     * @param bytes
     * @param offset
     * @param len
     * @throws IOException
     */
    @Override
    public synchronized void write(byte bytes[], int offset, int len) throws IOException {
        // if the length being written is larger than the buffer 
        if (len >= buf.length) {
            // commit whatis currently and the buffer and bypass the buffer and commit the new data immediately.
            super.out.write(buf, 0, count);
            count = 0;
            super.out.write(bytes, offset, len);
        } else {
            // calculate the remaining bytes left in the buffer to see if the new data will fill the buffer.  if 
            // the length of the new data is greater than or equal to the remaining bytes then we must commit the 
            // current buffer.
            int remainingBytes = buf.length - count;
            if (len >= remainingBytes) {
                super.out.write(buf, 0, count);
                count = 0;
            }
            
            // buffer the new data
            System.arraycopy(bytes, offset, buf, count, len);
            count += len;
        }
    }
}

