//	JNIOR Automation Network Operating System (JANOS)
//  CopyRight (C) 2012-2016 INTEG process group inc. All Rights Reserved.

/* ---------------------------------------------------------------------------
 * This software is INTEG process group inc proprietary. This source code
 * is for internal company use only and any other use without prior
 * consent of INTEG process group inc is prohibited.
 *
 * Author: Bruce Cloutier
 * Inception Date: 
 * -------------------------------------------------------------------------*/

package com.integpg.system;

/**
 *
 * @author Bruce Cloutier
 */
public class User {
    
    /**
     * This permissions flag is set if the user is an administrator.
     * 
     * @since 1.4.1
     */
    public static final int PERMS_ADMIN     = 0x80;
    
    /**
     * This permissions flag is set if the user can control outputs. Note that
     * an administrator can also control outputs.
     * 
     * @since 1.4.1
     */
    public static final int PERMS_CONTROL   = 0x02;
    
    /**
     * This permissions flag is set if the user account is disabled.
     * 
     * @since 1.4.1
     */
    public static final int PERMS_DISABLED  = 0x01;

    /*
     **  Cannot be instantiated
     */
    private User() {}

    /**
     * Query the current user. Programs executed at startup run as administrator using
     * the account with the lowest ID. Programs executed from the command line run
     * under the current login.
     * 
     * @return the current user's ID
     * @since 1.4.1
     */
    public static native int getCurrentUID();
    
    /**
     * Returns the name of the current user account.
     *
     * @return String containing the current user name
     * @since 1.4.1
     */
    public static String getCurrentUserName() {}

    /**
     * Queries the existence of a user.
     * 
     * @param username user's account (e.g. "jnior")
     * @return TRUE if an account exists for the specified user
     */
    public static native boolean isUser(String username);

    /**
     * Obtains the permissions byte for the specified user.
     *
     * @param username user's account (e.g. "jnior")
     * @return permissions or -1 if the user does not exist
     */
    public static native int perms(String username);

    /**
     * Retrieves the user account name for a specific ID.
     * 
     * @param username user's account name (e.g. "jnior")
     * @return account ID number or -1 if the user does not exist
     */
    public static native int getUserID(String username);

    /**
     * Retrieves the user account name for the specified ID.
     * 
     * @param id account ID
     * @return user's account name or NULL if the account does not exist
     */
    public static native String getUserName(int id);
    
    /**
     * Indicates whether or not the current user has administrator privileges.  Administrators
     * have access to all commands and functions.
     *
     * @return  true if the user has administrative rights
     * @since v1.5.1
     */
    public static boolean isCurrentUserAdmin() {}

    /**
     * Determines if the given user ID has administrative rights.
     *
     * @param uid user ID for the query
     *
     * @return  true if the supplied user is an administrator
     * @since v1.5.1
     */
    public static boolean isAdmin(int uid) {}

    /**
     * Determines if the given user ID has been disabled.
     *
     * @param uid user ID for the query
     *
     * @return  true if the supplied user account has been disabled
     * @since v1.5.1
     */
    public static boolean isDisabled(int uid) {}

    /**
     * Check for control access.
     * 
     * @return TRUE if the current user can control outputs
     * @since 1.4.1
     */
    public static boolean canControl() {}

    /**
     * Check for control access. In order to control outputs the user's account
     * must exist and cannot be disabled. The user must have control permission
     * or be an administrator.
     * 
     * @param username user's account name
     * @return TRUE if the specified user can control outputs
     * @since 1.4.1
     */
    public static boolean canControl(String username) {}

    /**
     * Confirms access to a user account. Requires both a valid account name and
     * password. Does not change the current user.
     * 
     * @param username the user account name (e.g. "jnior")
     * @param password the account password
     * @return the account permissions if the login is successful or -1 otherwise
     */
    public static native int validate(String username, String password);

    /**
     * Calculates the MD5 digest obtained by concatenating the user's password to 
     * the 'prefix' and then concatenating the 'suffix'.
     *
     * @param userid the user's account ID
     * @param prefix String prepended to the user's credentials
     * @param suffix String appended to the user's credentials
     * @return 16-byte array containing the MD5 digest or NULL if the user does not exist
     */
    public static native byte[] digestMD5(int userid, String prefix, String suffix);

    /**
     * Calculates the SHA1 digest obtained by concatenating the user's password to
     * the 'prefix' and then concatenating the 'suffix'.
     *
     * @param userid the user's account ID
     * @param prefix String prepended to the user's credentials
     * @param suffix String appended to the user's credentials
     * @return 16-byte array containing the SHA1 digest or NULL if the user does not exist
     */
    public static native byte[] digestSHA1(int userid, String prefix, String suffix);
}
