package java.text;

import com.integpg.system.ArrayUtils;
import com.integpg.system.Timebase;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * This class constructs a date string with a user defined format. it caches date and time locations to help construct
 * the string faster.
 */
public class QuickDateFormat {

    private static final SimpleDateFormat TIMEZON_DATE_FORMAT = new SimpleDateFormat("z");
    private static final long MILLIS_IN_HOUR = 3600000;
    private static final String[] MONTH_NAME_STRINGS = new String[]{ "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
    private static final byte[][] AM_PM_STRING_BYTES = new byte[][]{ "am".getBytes(), "pm".getBytes() };

    private final Calendar _calendar = Calendar.getInstance();

    private final byte[] _dateStampBytes;
    public final int DATE_STAMP_LENGTH;
    public final int DATE_STAMP_LENGTH_WITH_COMMA;
    private final String _pattern;

    int _weekOfYearPos = -1;
    int _monthPos = -1;
    int _monthNamePos = -1;
    int _dayPos = -1;
    int _2yearPos = -1;
    int _4yearPos = -1;
    int _hourPos = -1;
    int _hourOfDayPos = -1;
    int _minutePos = -1;
    int _secondPos = -1;
    int _millisPos = -1;
    int _timezonePos = -1;
    int _ampmPos = -1;

    private final byte[] _buf = new byte[8];
    private long _lastTimeRef = 0;

    private long _nextHour = System.currentTimeMillis() - System.currentTimeMillis() % MILLIS_IN_HOUR;
    private long _nextMinute = 0;



    public QuickDateFormat() {
        this("MM-dd-yyyy HH:mm:ss.fff zzz");
    }



    public QuickDateFormat(String pattern) {
        _pattern = pattern;

        _dateStampBytes = new byte[pattern.length() + 2];
        ArrayUtils.arraycopy(pattern.getBytes(), 0, _dateStampBytes, 0, pattern.length());
        _dateStampBytes[_dateStampBytes.length - 2] = ',';
        _dateStampBytes[_dateStampBytes.length - 1] = ' ';

        DATE_STAMP_LENGTH = _dateStampBytes.length - 2;
        DATE_STAMP_LENGTH_WITH_COMMA = _dateStampBytes.length;

        _weekOfYearPos = pattern.indexOf("WY");
        _monthNamePos = pattern.indexOf("MMM");
        if (-1 == _monthNamePos) {
            _monthPos = pattern.indexOf("MM");
        }
        _dayPos = pattern.indexOf("dd");
        if (-1 == _dayPos) _dayPos = pattern.indexOf("DD");
        _2yearPos = pattern.indexOf("yy");
        if (-1 == _2yearPos) _2yearPos = pattern.indexOf("YY");
        _4yearPos = pattern.indexOf("yyyy");
        if (-1 == _4yearPos) _4yearPos = pattern.indexOf("YYYY");
        _hourPos = pattern.indexOf("hh");
        _hourOfDayPos = pattern.indexOf("HH");
        _minutePos = pattern.indexOf("mm");
        _secondPos = pattern.indexOf("ss");
        _millisPos = pattern.indexOf("fff");
        if (-1 == _millisPos) _millisPos = pattern.indexOf("SSS");
        _timezonePos = pattern.indexOf("zzz");
        _ampmPos = pattern.indexOf("aa");

        updateTimezone();
    }



    public String getDateFormatString() {
        return _pattern;
    }



    private void updateTimezone() {
        if (-1 != _timezonePos) {
            String timezoneString = Timebase.getTimeZone(); //TIMEZON_DATE_FORMAT.format(new Date());
//            System.out.println("timezoneString: " + timezoneString);
//            System.out.println("_dateStampBytes.legnth: " + _dateStampBytes.length);
//            System.out.println("_timezonePos: " + _timezonePos);
            ArrayUtils.arraycopy(timezoneString.getBytes(), 0, _dateStampBytes, _timezonePos, timezoneString.length());
        }
    }



    // 2.0
    public String format(Date date) {
        return format(date.getTime());
    }



    public String format() {
        return format(System.currentTimeMillis());
    }



    public String format(long time) {
        synchronized (_dateStampBytes) {
            if (time <= 0) time = System.currentTimeMillis();

            if (_nextHour <= time) {
                _nextHour += MILLIS_IN_HOUR;
                updateTimezone();
            }

            if (_lastTimeRef > time || time >= _nextMinute) {
                initRefBytes(time);
            }

            int delta = (int) (time - _lastTimeRef);
            int rSecond = delta / 1000;

            if (-1 != _secondPos) {
                stuffInt(_buf, rSecond + 100, 0, false);
                ArrayUtils.arraycopy(_buf, 1, _dateStampBytes, _secondPos, 2);
            }

            if (-1 != _millisPos) {
                int rMillis = delta - 1000 * rSecond;
                stuffInt(_buf, rMillis + 1000, 0, false);
                ArrayUtils.arraycopy(_buf, 1, _dateStampBytes, _millisPos, 3);
            }

            return new String(_dateStampBytes, 0, DATE_STAMP_LENGTH);
        }
    }



    // 2.0
    public byte[] toByteArray() {
        return _dateStampBytes;
    }



    private void initRefBytes(long time) {
        _calendar.setTimeInMillis(time);

        int second = _calendar.get(Calendar.SECOND);
        int millis = _calendar.get(Calendar.MILLISECOND);

        _lastTimeRef = time - (1000 * second + millis);
        _nextMinute = time + 60000 - (time % 60000);

        synchronized (_dateStampBytes) {
            if (-1 != _weekOfYearPos) {
                int weekOfYear = _calendar.get(Calendar.WEEK_OF_YEAR);
//        System.out.println("woy: " + weekOfYear);
                stuffInt(_buf, weekOfYear + 100, 0, false);
                ArrayUtils.arraycopy(_buf, 1, _dateStampBytes, _weekOfYearPos, 2);
            }

            if (-1 != _monthPos) {
                int month = _calendar.get(Calendar.MONTH);
                stuffInt(_buf, month + 101, 0, false);
                ArrayUtils.arraycopy(_buf, 1, _dateStampBytes, _monthPos, 2);
            }

            if (-1 != _monthNamePos) {
                int month = _calendar.get(Calendar.MONTH);
                String monthName = MONTH_NAME_STRINGS[month];
                ArrayUtils.arraycopy(monthName.getBytes(), 0, _dateStampBytes, _monthNamePos, 3);
            }

            if (-1 != _dayPos) {
                int day = _calendar.get(Calendar.DAY_OF_MONTH);
                stuffInt(_buf, day + 100, 0, false);
                ArrayUtils.arraycopy(_buf, 1, _dateStampBytes, _dayPos, 2);
            }

            if (-1 != _2yearPos || -1 != _4yearPos) {
                int year = _calendar.get(Calendar.YEAR);
                stuffInt(_buf, year, 0, false);
                if (-1 != _2yearPos) ArrayUtils.arraycopy(_buf, 2, _dateStampBytes, _2yearPos, 2);
                if (-1 != _4yearPos) ArrayUtils.arraycopy(_buf, 0, _dateStampBytes, _4yearPos, 4);
            }

            if (-1 != _hourPos) {
                int rHour = _calendar.get(Calendar.HOUR);
                stuffInt(_buf, rHour + 100, 0, false);
                ArrayUtils.arraycopy(_buf, 1, _dateStampBytes, _hourPos, 2);
            }

            if (-1 != _hourOfDayPos) {
                int rHour = _calendar.get(Calendar.HOUR_OF_DAY);
                stuffInt(_buf, rHour + 100, 0, false);
                ArrayUtils.arraycopy(_buf, 1, _dateStampBytes, _hourOfDayPos, 2);
            }

            if (-1 != _minutePos) {
                int minute = _calendar.get(Calendar.MINUTE);
                stuffInt(_buf, minute + 100, 0, false);
                ArrayUtils.arraycopy(_buf, 1, _dateStampBytes, _minutePos, 2);
            }

            if (-1 != _ampmPos) {
                byte[] amPmBytes = AM_PM_STRING_BYTES[_calendar.get(Calendar.AM_PM)];
                ArrayUtils.arraycopy(amPmBytes, 0, _dateStampBytes, _ampmPos, 2);
            }
        }
    }



    private void stuffInt(byte[] array, int value, int dec, boolean rjustified) {
        ArrayUtils.arrayFill(array, 0, array.length, (byte) ' ');

        if (value == 0) {
            array[0] = (byte) '0';
            return;
        }

        int dig;
        int offset = 0;
        boolean leading = true;

        if (value < 0) {
            array[offset++] = (byte) '-';
            value = -value;
            rjustified = false;
        }

        for (dig = 1000000; dig >= 1; dig /= 10) {
            if (leading && value < dig) {
                if (rjustified) {
                    offset++;
                }
                continue;
            }

            array[offset] = (byte) '0';
            while (value >= dig) {
                array[offset]++;
                value -= dig;
            }
            offset++;
            leading = false;
        }

        for (dig = dec; dig > 0; dig--) {
            array[offset] = array[offset - 1];
            if (array[offset] == ' ') {
                array[offset] = '0';
            }
            offset--;

            if (dig == 1) {
                array[offset] = (byte) '.';
                if (array[offset - 1] == ' ') {
                    array[offset - 1] = '0';
                }
            }
        }
    }
}

