package com.integ.common.messagepump;

import com.integ.common.collections.BlockingQueue;
import com.integ.common.logging.Logger;
import com.integ.common.logging.SystemOutLog;
import com.integpg.system.MessagePump;
import com.integpg.system.SystemMsg;
import java.util.ArrayList;

public class MessagePumpEngine {

    private static final MessagePump MESSAGE_PUMP = new MessagePump();
    private static final BlockingQueue<SystemMsg> BLOCKING_QUEUE = new BlockingQueue<>();
    private static final ArrayList<MessagePumpListener> LISTENERS = new ArrayList<>();

    private static Logger _log = SystemOutLog.getLogger();
    private static Thread _messagePumpThread;
    private static Thread _messageNotifierThread;



    /**
     * Singleton constructor
     */
    private MessagePumpEngine() {
        BLOCKING_QUEUE.setName(this.getClass().getName());
    }



    /**
     * adds a listener that will be alerted of all received messages. The listener will be
     * responsible for determining
     * if the message has meaning to them
     */
    public static void addListener(MessagePumpListener listener) {
        synchronized (LISTENERS) {
            LISTENERS.add(listener);
        }
    }



    public static void setLog(Logger log) {
        _log = log;
    }



    /**
     * starts our message pump engine.
     */
    public static void start() {
        synchronized (MESSAGE_PUMP) {
            if (null == _messagePumpThread) {
                _messagePumpThread = new Thread(new Runnable() {
                    public void run() {
                        System.out.println("open the message pump and start monitoring.\r\n");
                        MESSAGE_PUMP.open();

                        for (;;) {
                            // read all messages from the message pump
                            SystemMsg systemMsg = MESSAGE_PUMP.getMessage();

                            // we must repost as fast as we can
                            MESSAGE_PUMP.postMessage(systemMsg);

                            try {
//                synchronized (BLOCKING_QUEUE) {
//                                LOG.info(".");
                                BLOCKING_QUEUE.put(systemMsg);
//                }
                            } catch (Exception ex) {
                                ex.printStackTrace();
                            }
                        }
                    }
                });
                _messagePumpThread.setName(MessagePumpEngine.class.getName() + ":queue");
                _messagePumpThread.setDaemon(true);;
                _messagePumpThread.start();
            }

            if (null == _messageNotifierThread) {
                _messageNotifierThread = new Thread(new Runnable() {
                    public void run() {
                        for (;;) {
                            try {
                                // get the next available system message from the message pump queue
                                SystemMsg systemMsg = BLOCKING_QUEUE.get();

                                try {
                                    // so we have a message.  what kind is it?
                                    String messageName = SystemMsgTypes.getMessageNameByType(systemMsg.type);
                                    // print out what we know about this message
                                    String contents = "";
//                                    if (0 < systemMsg.msg.length) {
//                                        contents = ": " + new String(systemMsg.msg).trim();
//                                    }
                                    String s = String.format("(%d) %s%s", systemMsg.type, messageName, contents);
                                    _log.info(s);
                                } catch (Exception ex) {
                                    ex.printStackTrace();
                                }

//                                System.out.println(String.format("systemMsg: %d", systemMsg.type));

                                // notify all of our listeners
                                synchronized (LISTENERS) {
                                    for (MessagePumpListener listener : LISTENERS) {
                                        try {
                                            listener.messageReceived(systemMsg);
                                        } catch (Throwable ex) {
                                            ex.printStackTrace();
                                        }
                                    }
                                }

                                Thread.sleep(1);
                            } catch (Exception ex) {
                                ex.printStackTrace();
                            }

                        }
                    }
                });
                _messageNotifierThread.setName(MessagePumpEngine.class.getName() + ":notifier");
                _messageNotifierThread.setDaemon(true);
                _messageNotifierThread.start();
            }
        }
    }



    /**
     * Exposes the postMesssage method of the System MessagePump
     *
     * @param msg the message the will be sent to the system message pump
     */
    public static void postMessage(SystemMsg msg) {
        MESSAGE_PUMP.postMessage(msg);
    }
}

