package com.integ.common.net.protocols.modbus.protocols;

import com.integ.common.logging.AppLog;
import com.integ.common.net.BytesReceivedEvent;
import com.integ.common.net.TcpConnectionListener;
import com.integ.common.net.protocols.modbus.connections.TcpModbusConnection;
import com.integ.common.net.protocols.modbus.functions.ModbusPdu;
import com.integpg.system.ArrayUtils;
import java.io.*;
import java.util.EventObject;

public class ModbusTCPProtocol
        extends ModbusProtocol
        implements TcpConnectionListener {

    private final TcpModbusConnection _modbusConnection;
    private int _transactionIdentifier = 0;
    private int _unitIdentifier = 0;

    private final ByteArrayOutputStream _baos = new ByteArrayOutputStream();
    private final DataOutputStream _dos = new DataOutputStream(_baos);



    public ModbusTCPProtocol(TcpModbusConnection modbusConnection) {
        _modbusConnection = modbusConnection;
        _modbusConnection.addConnectionListener(this);
    }



    public ModbusTCPProtocol setUnitIdentifier(int unitIdentifier) {
        _unitIdentifier = unitIdentifier;
        return this;
    }



    @Override
    public void write(ModbusPdu pdu) throws Exception {
        try {
            byte[] data = pdu.getData();

            _baos.reset();
            _dos.writeShort(++_transactionIdentifier);
            _dos.writeShort(0);                                     // 0 for modbus/tcp 

            _dos.writeShort(data.length + 2);
            _dos.write(_unitIdentifier);

            _dos.write(pdu.getFunctionCode());
            _dos.write(data);

            byte[] bytes = _baos.toByteArray();
//            _log.info("<- " + HexUtils.bytesToHex(bytes, 0, bytes.length));

            _modbusConnection.beginWrite();
            OutputStream outputStream = _modbusConnection.getOutputStream();
            outputStream.write(bytes, 0, bytes.length);
            outputStream.flush();
            _modbusConnection.endWrite(bytes.length);
        } catch (Exception ex) {
            String clientInfo = _modbusConnection.getClientInfo();
            AppLog.error("Error writing to modbus tcp connection (" + clientInfo + ")", ex);
            _modbusConnection.closeConnection();
            throw ex;
        }
    }

    private final byte[] _responseBuffer = new byte[7];



    @Override
    public void read(ModbusPdu pdu) throws Exception {
        try {
            InputStream inputStream = _modbusConnection.getInputStream();

            int pos = 0;
            while (7 > pos) {
                int bytesRead = inputStream.read(_responseBuffer, pos, 7 - pos);
                pos += bytesRead;
                if (0 >= bytesRead) throw new Exception("Unable to read response header.  Not all bytes available.  Read " + pos + " bytes");
            }

            int transactionIdentifier = ArrayUtils.getShort(_responseBuffer, 0) & 0xffff;
            int protocolIdentifier = ArrayUtils.getShort(_responseBuffer, 2) & 0xffff;
            int lengthToFollow = ArrayUtils.getShort(_responseBuffer, 4) & 0xffff;

            //
            // going to be zero
            int unitId = _responseBuffer[6];

//            System.out.println(String.format("transactionIdentifier: %d, protocolIdentifier: %d, lengthToFollow: %d, unitId: %d",
//                    transactionIdentifier, protocolIdentifier, lengthToFollow, unitId));

            byte[] inBytes = new byte[lengthToFollow - 1];
            inputStream.read(inBytes);
//            _log.info("-> " + HexUtils.bytesToHex(inBytes, 0, inBytes.length));
            pdu.setResponse(inBytes);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }



    @Override
    public void connectionAttempt(EventObject evt) {
        _log.info(String.format("attempting connection to %s", _modbusConnection.getClientInfo()));
    }



    @Override
    public void connectionEstablished(EventObject evt) {
        _log.info(String.format("connection established to %s", _modbusConnection.getClientInfo()));
    }



    @Override
    public void connectionClosed(EventObject evt) {
        _log.info(String.format("connection to %s has been closed", _modbusConnection.getClientInfo()));
    }



    @Override
    public void bytesReceived(BytesReceivedEvent evt) {
        // do nothing
    }

}

