package com.integ.common.net.protocols.modbus.protocols;

import com.integ.common.net.protocols.modbus.connections.ModbusConnection;
import com.integ.common.net.protocols.modbus.functions.ModbusPdu;
import com.integ.common.net.protocols.modbus.functions.ReadHoldingRegisters;
import com.integ.common.net.protocols.modbus.functions.ReadInputRegisters;
import com.integ.common.utils.HexUtils;
import com.integpg.system.ArrayUtils;
import com.integpg.system.JANOS;
import java.io.*;

public class ModbusRTUProtocol extends ModbusProtocol {

    private final ModbusConnection _modbusConnection;
    private final int _slaveId;

    private InputStream _inputStream = null;
    private DataInputStream _dataInputStream = null;

    private OutputStream _outputStream = null;
    private DataOutputStream _dataOutputStream;

    private final ByteArrayOutputStream _baos = new ByteArrayOutputStream();
    private final DataOutputStream _dos = new DataOutputStream(_baos);



    public ModbusRTUProtocol(ModbusConnection modbusConnection, int slaveId) {
        _modbusConnection = modbusConnection;
        _slaveId = slaveId;

        try {
            _inputStream = _modbusConnection.getInputStream();
            _dataInputStream = new DataInputStream(_inputStream);
            _inputStream.skip(_inputStream.available());
        } catch (Exception ex) {
            _log.error(ex);
        }
    }



    @Override
    public void write(ModbusPdu pdu) {
        try {
            OutputStream outputStream = _modbusConnection.getOutputStream();
            if (_outputStream != outputStream) {
                _outputStream = outputStream;
                _dataOutputStream = new DataOutputStream(outputStream);
            }

            _baos.reset();
            _dos.writeByte(_slaveId);
            _dos.writeByte(pdu.getFunctionCode());
            _dos.write(pdu.getData());
            int crc = JANOS.CRC16(_baos.toByteArray(), 0, _baos.size(), 0xffff);
            crc = ArrayUtils.swapEndian((short) crc);
            _dos.writeShort(crc);

            _modbusConnection.beginWrite();
            byte[] bytes = _baos.toByteArray();

            String hex = HexUtils.bytesToHex(bytes, 0, bytes.length);
            _log.info("<--  " + hex);

            _modbusConnection.endWrite(bytes, 0, bytes.length);
        } catch (IOException ex) {
            _log.error(ex);
            ex.printStackTrace();
            _modbusConnection.endWrite(-1);
        }
    }



    public ModbusPdu read() {
        try {
//            while (_inputStream.available() < _lastBytesWritten) {
//                System.out.println("read: " + _inputStream.available());
//            }
//            _inputStream.skip(_lastBytesWritten);

            int slaveId = _inputStream.read();
            System.out.println("slaveId: " + slaveId);
            int function = _inputStream.read();
            System.out.println("function: " + function);

            ModbusPdu pdu = null;
            switch (function) {
                case 3: {
                    int address = _dataInputStream.readShort();
                    System.out.println("address: " + address);
                    int quantity = _dataInputStream.readShort();
                    System.out.println("quantity: " + quantity);
                    pdu = new ReadHoldingRegisters(address, quantity);
                    break;
                }

                case 4: {
                    int address = _dataInputStream.readShort();
                    System.out.println("address: " + address);
                    int quantity = _dataInputStream.readShort();
                    System.out.println("quantity: " + quantity);
                    pdu = new ReadInputRegisters(address, quantity);
                    break;
                }

                default:
                    throw new RuntimeException("Unknown PDU function");
            }

            int crc = _dataInputStream.readShort();
            System.out.println("crc: " + Integer.toHexString(crc));

            return pdu;
        } catch (Exception ex) {
            _log.error("error reading modbus response", ex);
            ex.printStackTrace();
        }
        return null;
    }



    @Override
    public void read(ModbusPdu pdu) {
        try {
            _inputStream = _modbusConnection.getInputStream();

            byte[] inbuf = new byte[512];
            _baos.reset();
            long timeout = JANOS.uptimeMillis() + 5000;
            long totalBytes = 0;
            while (JANOS.uptimeMillis() < timeout) {
                int bytesRead = _inputStream.read(inbuf, 0, inbuf.length);
                totalBytes += bytesRead;
                _log.debug("read(pdu): " + bytesRead + "," + totalBytes);
                _baos.write(inbuf, 0, bytesRead);

                if (0 == _inputStream.available()) {
                    if (_baos.size() > 4) {
                        byte[] bytes = _baos.toByteArray();
                        int crc2 = ArrayUtils.getShort(bytes, bytes.length - 2);
                        int calculatedCrc2 = JANOS.CRC16(bytes, 0, bytes.length - 2, 0xffff);
                        calculatedCrc2 = ArrayUtils.swapEndian((short) calculatedCrc2);
//                    System.out.println(crc2 + " " + calculatedCrc2);
                        if (calculatedCrc2 == crc2) {
                            break;
                        } else {
//                        READ_MODBUS_LOG.debug("crc does not match");
                        }
                    }
                }
            }

            byte[] inBytes = _baos.toByteArray();
            String hex = HexUtils.bytesToHex(inBytes, 0, inBytes.length);
            _log.info(" --> " + hex);

            int slaveId = inBytes[0];
            int functionCode = inBytes[1];
            if ((functionCode & 0x80) == 0) {
                byte[] data = new byte[inBytes.length - 4];
                ArrayUtils.arraycopy(inBytes, 2, data, 0, data.length);
                pdu.setResponse(data);
            } else {
                int errorCode = inBytes[2];
                System.out.println("errorCode: " + errorCode);
            }
//            System.out.println("slaveId: " + slaveId + ", functionCode: " + functionCode + ", crc: " + crc);
            int crc = ArrayUtils.getShort(inBytes, inBytes.length - 2);
            int calculatedCrc = JANOS.CRC16(inBytes, 0, inBytes.length - 2, 0xffff);
            calculatedCrc = ArrayUtils.swapEndian((short) calculatedCrc);
            if (calculatedCrc != crc) {
                _log.warn("calculatedCrc: " + Integer.toHexString(calculatedCrc) + ", crc: " + Integer.toHexString(crc));
                throw new Exception("Bad CRC Check");
            }
        } catch (Exception ex) {
            byte[] inBytes = _baos.toByteArray();
            String hex = HexUtils.bytesToHex(inBytes, 0, inBytes.length);
            _log.warn("< " + hex);
            _log.error(ex);

            ex.printStackTrace();
        }
    }

}

