package com.integ.common.logging;

import com.integ.common.utils.FileUtils;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;

public abstract class LogOutputStream extends OutputStream {

    protected final LogOptions _logOptions;
    protected final File _file;
    protected final File _bakFile;
    protected final int _clusterSize = 4096;



    public LogOutputStream(LogOptions logOptions) throws FileNotFoundException {
        _logOptions = logOptions;
        _file = new File(_logOptions.getFilePath());
        _bakFile = new File(_file.getPath() + ".bak");
    }



    /**
     * This must be overridden when extending the OutputStream class. But I dont ever expect this to be called in the
     * current use.
     *
     * @param b
     * @throws IOException
     */
    @Override
    public void write(int b) throws IOException {
        throw new RuntimeException("should not call this method");
    }



    /**
     * writes to file. before writing to the file we call the preWrite method. This allows subclasses to implement
     * special limit the file size. This may be a backup or rolling file.
     *
     * @param b
     * @param off
     * @param len
     * @throws IOException
     */
    @Override
    public void write(byte b[], int off, int len) throws IOException {
        synchronized (_file) {
            if (0 != len) {
                int numberOfBytesConsumedInPreWrite = preWrite(b, off, len);
                if (0 < numberOfBytesConsumedInPreWrite) {
                    off += numberOfBytesConsumedInPreWrite;
                    len -= numberOfBytesConsumedInPreWrite;
                }

                FileUtils.appendBytes(_file.getPath(), b, off, len);

                postWrite(len);
            }
        }
    }



    /**
     * pre write logic. must be overridden by subclass instances.
     *
     * @param bytes byte array to be written the the file
     * @param off offset
     * @param len length
     * @return number of bytes consumed and committed to the backup file. this happens when using a .bak file concept.
     * In that case we will look for the most amount of data that can be used to fill out the .bak. The remaining bytes
     * are written to our current file.
     * @throws IOException
     */
    protected abstract int preWrite(byte[] bytes, int off, int len) throws IOException;



    /**
     * post write logic. None of my subclasses implement this yet.
     *
     * @param len length of bytes that have been written
     */
    private void postWrite(int len) {
//        System.out.println(String.format("%s committed %d bytes. (%.2f KB)", _logOptions.getFilePath(), len, (_file.length() / 1024.0)));
    }



    @Override
    public void flush() {
        // do nothing
    }
}

