package com.integ.common.iolog;

import com.integ.common.logging.Logger;
import com.integ.common.logging.SystemOutLog;
import com.integpg.system.IoEvent;
import com.integpg.system.Iolog;
import com.integpg.system.JANOS;
import java.text.QuickDateFormat;
import java.util.ArrayList;

public abstract class IoLogMonitor implements Runnable {

    protected static final QuickDateFormat QUICK_DATE_FORMAT = new QuickDateFormat("MM/dd/yyyy HH:mm:ss.fff");

    protected final Iolog _iolog = new Iolog();
    protected final ArrayList<IoLogListener> _ioLogEventListeners = new ArrayList<>();
    protected final ArrayList<IoChannelLogListener> _ioChannelLogEventListeners = new ArrayList<>();

    protected Logger LOG = SystemOutLog.getLogger();
    private Thread _thd;
    private long _refreshTimestamp;
    protected static IoEvent _lastIoEvent;
    protected static IoEvent _lastInputEvent;
    protected static IoEvent _lastOutputEvent;



    protected IoLogMonitor() {
    }



    public void start() {
        if (null == _thd) {
            _thd = new Thread((Runnable) this);
            _thd.setName(this.getClass().getName());
            _thd.start();
        }
    }



    public void addIoLogEventListener(IoLogListener ioLogEventListener) {
        _ioLogEventListeners.add(ioLogEventListener);
    }



    public void removeIoLogEventListener(IoLogListener ioLogEventListener) {
        _ioLogEventListeners.remove(ioLogEventListener);
    }



    public void addIoChannelLogEventListener(IoChannelLogListener digitalOutputsLogListener) {
        _ioChannelLogEventListeners.add(digitalOutputsLogListener);
    }



    public void removeIoChannelLogEventListener(IoChannelLogListener digitalOutputsLogListener) {
        _ioChannelLogEventListeners.remove(digitalOutputsLogListener);
    }



    public void setRefreshTimestamp(long refreshTimestamp) {
        _refreshTimestamp = refreshTimestamp;
    }



    protected abstract IoEvent[] getIoEvents();



    public static IoEvent getLastIoEvent() {
        return _lastIoEvent;
    }



    public static IoEvent getLastInputEvent() {
        return _lastInputEvent;
    }



    public static IoEvent getLastOutputEvent() {
        return _lastOutputEvent;
    }



    @Override
    public void run() {
        long _savedSignature = -1;
        while (true) {
            try {
                long signature = JANOS.getIoSignature();
                if (_savedSignature != signature) {

                    _savedSignature = signature;

                    // refresh the iolog object so that we only get new events since the last time we queried it
                    _iolog.refresh(_refreshTimestamp);
                    // get the events for the inputs only
                    IoEvent[] ioEvents = getIoEvents();

                    // make sure we were returned events
                    if (ioEvents.length > 0) {
                        if (1 < ioEvents.length) {
                            LOG.info(ioEvents.length + " events returned since " + QUICK_DATE_FORMAT.format(_refreshTimestamp));
                        }

                        int errorCount = 0;
                        // go through each event and alert listeners
                        for (int index = ioEvents.length - 1; index >= 0; index--) {
                            IoEvent ioEvent = ioEvents[index];

                            if (ioEvent.timestamp < _refreshTimestamp) {
                                errorCount++;
                                continue;
                            }

//                        System.out.println("ioEvent[" + index + "]:"
//                                + " time: " + QUICK_DATE_FORMAT.format(ioEvent.timestamp)
//                                + " changed input mask: " + Long.toHexString(ioEvent.mask)
//                                + " input states mask: " + Long.toHexString(ioEvent.states));

                            alertListenersOfIoEvent(ioEvent);
                        }

                        if (0 != errorCount) {
                            LOG.info("iolog returned " + errorCount + " events earlier than the requested time of "
                                    + QUICK_DATE_FORMAT.format(_refreshTimestamp));
                        }

                        // update our lastConsumedTimestamp with the date of the most recent event
                        _refreshTimestamp = ioEvents[0].timestamp;
//                    System.out.println("_refreshTimestamp: " + QUICK_DATE_FORMAT.format(_refreshTimestamp));

                        alertListenersOfEventsProcessed();
                    }
                }

                Thread.sleep(10);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
    }



    private void alertListenersOfIoEvent(IoEvent ioEvent) {
        try {
            long start = System.currentTimeMillis();

            if (null == _lastIoEvent || _lastIoEvent.timestamp < ioEvent.timestamp) {
                _lastIoEvent = ioEvent;
            }

            if (this instanceof DigitalInputsIoLogMonitor) {
                _lastInputEvent = ioEvent;
            } else if (this instanceof DigitalOutputsIoLogMonitor) {
                _lastOutputEvent = ioEvent;
            }

            for (IoLogListener ioLogListener : _ioLogEventListeners) {
                if (null != ioLogListener) {
                    ioLogListener.onIoEvent(ioEvent);
                }
            }

            long elapsed = System.currentTimeMillis() - start;
            if (100 < elapsed) {
                System.out.println(String.format("process event took %.3f", (elapsed / 1000.0)));
            }

            alertIoChannelListeners(ioEvent);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }



    public abstract void alertIoChannelListeners(IoEvent ioEvent);



    private void alertListenersOfEventsProcessed() {
        try {
            for (IoLogListener ioLogListener : _ioLogEventListeners) {
                if (null != ioLogListener) {
                    long start = System.currentTimeMillis();
                    ioLogListener.onIoEventsProcessed();
                    long elapsed = System.currentTimeMillis() - start;

                    if (100 < elapsed) {
                        System.out.println(String.format("process event took %.3f", (elapsed / 1000.0)));
                    }
                }
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

}

