package com.integ.common.net.protocols.modbus.connections;

import com.integ.common.logging.AppLog;
import com.integ.common.utils.HexUtils;
import com.integpg.comm.*;
import com.integpg.system.ArrayUtils;
import com.integpg.system.JANOS;
import java.io.*;

public class SerialModbusConnection extends ModbusConnection implements SerialPortEventListener {

    private SerialPort _serialPort;
    private boolean _is485 = false;
    private int _lastBytesWritten;
    private byte[] _skipBytes = new byte[128];



    public SerialModbusConnection(String portName, int baudRate, int dataBits, int stopBits, int parity) throws Throwable {
        AppLog.info("Creating MODBUS Connection on " + portName + " port");
        if (portName.equalsIgnoreCase("none")) throw new IllegalArgumentException("Serial port not selected");
        else if (portName.equalsIgnoreCase("aux")) _serialPort = new AUXSerialPort();
        else if (portName.equalsIgnoreCase("com")
                || portName.equalsIgnoreCase("rs232")) _serialPort = new COMSerialPort();

        if (_serialPort != null) {
            try {
                _serialPort.open();
                _serialPort.disableReceiveTimeout();
                _serialPort.setSerialPortParams(baudRate, dataBits, stopBits, parity);

                // well we know that 128 registers could be read at 2 bytes a peiece plus modbus header info.  
                // make the input buffer much larger than neccesary.  512 should be good.
                _serialPort.setInputBufferSize(512);
                AppLog.info("Set Serial Input Buffer to " + _serialPort.getInputBufferSize() + " bytes");

                // listen for events
                _serialPort.addEventListener(this);
                _serialPort.notifyOnFramingError(true);
                _serialPort.notifyOnOverrunError(true);
                _serialPort.notifyOnOverrunError(true);
                _serialPort.notifyOnParityError(true);
                _serialPort.notifyOnBreakInterrupt(true);

                InputStream inputStream = _serialPort.getInputStream();
                inputStream.skip(inputStream.available());
            } catch (Exception ex) {
                throw new RuntimeException("unable to open " + portName + " port")
                        .initCause(ex);
            }
        }
    }



    public void setSerialPortParams(int baudRate, int dataBits, int stopBits, int parity) throws UnsupportedCommOperationException {
        _serialPort.setSerialPortParams(baudRate, dataBits, stopBits, parity);
    }



    public void setRS485(boolean is485) {
        _is485 = is485;
        if (is485) {
            AppLog.info("Setting MODBUS Connection to use RS-485");
            ((AUXSerialPort) _serialPort).setRS485(is485);
//            ((AUXSerialPort) _serialPort).enableDrivers(false);
        }
    }



    public void enableReceiveTimeout(int timeout) throws UnsupportedCommOperationException {
        AppLog.info("Setting MODBUS Connection read timeout to " + timeout);
        if (-1 == timeout) {
            _serialPort.disableReceiveTimeout();
        } else {
            _serialPort.enableReceiveTimeout(timeout);
        }
    }



    public InputStream getInputStream() throws IOException {
        return _serialPort.getInputStream();
    }



    @Override
    public OutputStream getOutputStream() throws IOException {
        return _serialPort.getOutputStream();
    }



    @Override
    public void beginWrite() {
        try {
            _serialPort.enableReceiveTimeout(1000);
        } catch (UnsupportedCommOperationException ex) {
            ex.printStackTrace(System.err);
        }
    }



    @Override
    public void endWrite(int bytesWritten) {
        _serialPort.disableReceiveTimeout();
    }



    @Override
    public void endWrite(byte[] bytes, int offset, int length) {
        try {
            if (_is485) {
                // enable the driver to send to 485 network
                ((AUXSerialPort) _serialPort).enableDrivers(true);
            }

            _lastBytesWritten = length;

            // blast all waiting bytes.  we are gonig to make a new query here
            int bytesAvailableAtStartOfQuery = _serialPort.getInputStream().available();
            if (0 < bytesAvailableAtStartOfQuery) {
                AppLog.warn("there are bytes available at the start of the query: " + bytesAvailableAtStartOfQuery);
                _serialPort.getInputStream().skip(bytesAvailableAtStartOfQuery);
            }

            _serialPort.getOutputStream().write(bytes, 0, length);
            _serialPort.getOutputStream().flush();

            if (_is485) {
                // make sure we received what we sent
//                System.out.println("avail: " + _serialPort.getInputStream().available());
                long expire = System.currentTimeMillis() + 1000;
                while (_serialPort.getInputStream().available() < length) {
//                    System.out.println("avail: " + _serialPort.getInputStream().available());
                    if (System.currentTimeMillis() > expire) {
                        AppLog.info("reading sent bytes expired: " + _serialPort.getInputStream().available());
                        break;
                    }
                }

                // disable the driver after sending to the 485 network
                ((AUXSerialPort) _serialPort).enableDrivers(false);

                if (length <= _serialPort.getInputStream().available()) {
                    _serialPort.getInputStream().read(_skipBytes, 0, length);
                    if (!ArrayUtils.arrayComp(bytes, 0, _skipBytes, 0, length)) {
                        AppLog.info("sent bytes not echoed");
                        AppLog.info("< " + HexUtils.bytesToHex(_skipBytes, 0, length));
                    }
                } else {
                    AppLog.error("sent bytes not echoed");
                }
            }
        } catch (IOException ex) {
            ex.printStackTrace();
        }

    }



    public void serialEvent(SerialPortEvent ev) {
        switch (ev.getEventType()) {
            case SerialPortEvent.PE:
                JANOS.syslog("** parity error");
                break;
            case SerialPortEvent.FE:
                JANOS.syslog("** framing error");
                break;
            case SerialPortEvent.OE:
                JANOS.syslog("** overrun error");
                break;
            case SerialPortEvent.BI:
                JANOS.syslog("** break condition");
                break;
            default:
                JANOS.syslog("** unknown event");
                break;
        }
    }



    @Override
    public void closeConnection() {
        // this is serial.  there is nothing to close
    }



    @Override
    public boolean isConnected() {
        // this is serial.  it is always "connected"
        return true;
    }

}

