package com.integ.common.net.protocols.modbus.protocols;

import com.integ.common.logging.AppLog;
import com.integ.common.net.protocols.modbus.connections.ModbusConnection;
import com.integ.common.net.protocols.modbus.functions.ModbusPdu;
import com.integ.common.utils.HexUtils;
import com.integpg.system.ArrayUtils;
import com.integpg.system.JANOS;
import java.io.*;

public class ModbusAsciiProtocol extends ModbusProtocol {

    private final ModbusConnection _modbusConnection;
    private final int _slaveId;

    private InputStream _inputStream = null;

    private OutputStream _outputStream = null;
    private DataOutputStream _dataOutputStream;

    private final ByteArrayOutputStream _baos = new ByteArrayOutputStream();
    private final DataOutputStream _dos = new DataOutputStream(_baos);



    public ModbusAsciiProtocol(ModbusConnection modbusConnection, int slaveId) {
        _modbusConnection = modbusConnection;
        _slaveId = slaveId;
    }



    public void write(ModbusPdu pdu) {
        try {
            OutputStream outputStream = _modbusConnection.getOutputStream();
            if (_outputStream != outputStream) {
                _outputStream = outputStream;
                _dataOutputStream = new DataOutputStream(outputStream);
            }

            _baos.reset();
            _dos.writeByte(_slaveId);
            pdu.pack(_dos);
            int lrc = getLrc(_baos.toByteArray(), 0, _baos.size());
            _dos.writeByte(lrc);
            String r = HexUtils.bytesToHex(_baos.toByteArray(), 0, _baos.size());

            _baos.reset();
            _dos.writeByte(':');
            _dos.write(r.getBytes());
            _dos.writeByte('\r');
            _dos.writeByte('\n');

            _modbusConnection.beginWrite();
            byte[] bytes = _baos.toByteArray();

            String hex = new String(bytes, 0, bytes.length - 2);
            _log.info("<--  " + hex);

            _modbusConnection.endWrite(bytes, 0, bytes.length);
        } catch (IOException ex) {
            ex.printStackTrace();
            _modbusConnection.endWrite(-1);
        }
    }



    public void read(ModbusPdu pdu) {
        try {
            _inputStream = _modbusConnection.getInputStream();
            _baos.reset();

            boolean crlfReceived = false;
            long timeout = JANOS.uptimeMillis() + 5000;
            while (JANOS.uptimeMillis() < timeout) {
                int b = _inputStream.read();
                _baos.write(b);
//                System.out.print(String.format("%s", Integer.toHexString(b)));
                if (!crlfReceived && '\r' == b) crlfReceived = true;
                if (crlfReceived && '\n' == b) break;
            }
//            System.out.println("");

            byte[] inBytes = _baos.toByteArray();
            String asciiString = new String(inBytes, 0, inBytes.length - 2);
            _log.info(" --> " + asciiString);

            byte[] bytes = HexUtils.hexToBytes(asciiString, 1, asciiString.length() - 1);
//            System.out.println(" --> " + HexUtils.bytesToHex(bytes));

//            String slaveId = new String(inBytes, 0, 2);
//            String functionCode = new String(inBytes, 2, 2);

            int lrc = bytes[bytes.length - 1] & 0xff;
//            System.out.println("slaveId: " + slaveId + ", functionCode: " + functionCode + ", crc: " + lrc);

            int calculatedLrc = getLrc(bytes, 0, bytes.length - 1);
//            System.out.println("calculatedLrc: " + calculatedLrc + ", expected Lrc: " + lrc);

            if (calculatedLrc != lrc) {
                System.out.println("calculatedLrc: " + calculatedLrc + ", expected Lrc: " + lrc);
                throw new Exception("Bad LRC Check");
            }

            int lengthToFollow = bytes[2];
            byte[] data = new byte[lengthToFollow + 2];
            ArrayUtils.arraycopy(bytes, 3, data, 2, lengthToFollow);
//            System.out.println("bytes: " + HexUtils.bytesToHex(bytes));
//            System.out.println(" data: " + HexUtils.bytesToHex(data));

            pdu.setResponse(data);
        } catch (Exception ex) {
            AppLog.error("error reading modbus ascii pdu", ex);
        }
    }



    public int getLrc(byte[] bytes, int offset, int length) {
        int lrc = 0;
        for (int i = 0; i < length; i++) {
            lrc += bytes[offset + i];
        }
        lrc = (((lrc ^ 0xff) + 1) & 0xff);
        return lrc;
    }
}

