//	JNIOR Automation Network Operating System (JANOS)
//  CopyRight (C) 2012-2016 INTEG process group inc. All Rights Reserved.

/* ---------------------------------------------------------------------------
 * This software is INTEG process group inc proprietary. This source code
 * is for internal company use only and any other use without prior
 * consent of INTEG process group inc is prohibited.
 *
 * Author: Bruce Cloutier
 * Inception Date: 
 * -------------------------------------------------------------------------*/

package com.integpg.sensor;

import com.integpg.system.ArrayUtils;
import com.integpg.system.JANOS;
import java.io.IOException;

/**
 *
 * @author Bruce Cloutier
 */
public class SensorPort {
    
    private static Object lock = new Object();
    private static boolean popen = false;
    private boolean open = false;
    private static int extchan = externals();
    private static int extmask = -(1 << extchan);
    
    // Returns the first channel that could be assigned to an external 4ROUT.
    private static int externals() {}
    
    /**
     * Perform CRC8 on an array of data elements based on the provided seed. CRC8 is based on the polynomial = X^8 + X^5 + X^4 +
     * 1. As used on the Dallas/Maxim 1-Wire bus
     *
     * @param data - byte array of data elements
     * @param offset - starting offset in the byte array
     * @param count - count of data elements to be included in the calculation
     * @param seed - initial CRC8 value
     * @return - CRC8 calculation
     */
    public static native int CRC8(byte[] data, int offset, int count, int seed);

    /**
     * Perform CRC16 on an array of data elements based on the provided seed. CRC16 is based on the polynomial = X^16 + X^15 +
     * X^2 + 1.
     *
     * @param data data byte array
     * @param offset starting offset in the byte array
     * @param count count of data elements to be included in the calculation
     * @param seed initial CRC16 value
     * @return CRC16 calculation
     */
    public static native int CRC16(byte[] data, int offset, int count, int seed);
    
    /**
     *
     * This method requests the enumeration of external devices. The addresses returned are used to read/write these devices.
     *
     * @return A long[] of device addresses.
     * @throws IOException The system must have access to the port.
     */
    public long[] getExternalDeviceList() throws IOException {}

    /**
     *
     * This method requests the enumeration of external devices in the alarm state. The addresses returned are used to
     * read/write these devices.
     *
     * @return A long[] of device addresses.
     * @throws IOException The system must have access to the port.
     */
    public long[] getAlarmingDeviceList() throws IOException {}

    /**
     * Obtains the external device type from a device address.
     *
     * @param addr the device address as returned by enumeration.
     *
     * @return The lease significant byte of the device address which indicates the device type.
     */
    public static int getDeviceType(long addr) {}

    /**
     * Opens the Sensor Port providing exclusive access to external devices. You MUST close the port as soon
     * as possible to allow other processes access.
     * 
     * @throws IOException If the port is already open.
     */
    public void open() throws IOException {}
    private static native void sensopn();
    
    /**
     * Closes an open Sensor Port.
     * 
     * @throws java.io.IOException if the port is not open
     */
    public void close() throws IOException {}
    private static native void senscls();

    /**
     * Issues the one-wire RESET pulse.
     * 
     * @throws IOException If the port is not open.
     */
    public void reset() throws IOException {}
    private static native void sensrst();
    
    /**
     * Selects the external device by address.
     * 
     * @param address Device address obtained by enumeration.
     * @throws IOException If the port is not open.
     */
    public void select(long address) throws IOException {}
    private static native void senssel(long address);

    /**
     * Performs 1-wire data exchange.
     * 
     * @param buf byte array with transmit/return data
     * @param ofs offset into byte array
     * @param len length of communications
     * @throws IOException if the port is not open
     */
    public void data(byte[] buf, int ofs, int len) throws IOException {}
    private static native void sensdata(byte[] buf, int ofs, int len);

    /**
     * Returns the temperature value.
     * 
     * @param chan Selects the temperature sensor channel.
     * @return Temperature
     * 
     * @throws java.io.IOException if the port is open
     */
    public static double getTemp(int chan) throws IOException {}
    private static native double gTemp(int chan) throws IOException;

    /**
     * *
     * This method is used to read the value of one 10 volt module input.
     *
     * @param chan the channel of the 10 volt input. You can have 2 external 10 volt modules giving you 8 inputs. The channel
     * must be 0 - 7.
     * @return a scaled value based on the range that was configured using the Main JNIOR Web Page
     * @throws java.io.IOException if the port is open
     */
    public static double get10vIn(int chan) throws IOException {}
    private static native double g10vIn(int chan) throws IOException;

    /**
     * *
     * This method is used to read the value of one 10 volt module output.
     *
     * @param chan the channel of the 10 volt output. You can have 2 external 10 volt modules giving you 4 outputs. The channel
     * must be 0 - 3.
     * @return a scaled value based on the range that was configured using the Main JNIOR Web Page
     * @throws java.io.IOException if the port is open
     */
    public static double get10vOut(int chan) throws IOException {}
    private static native double g10vOut(int chan) throws IOException;

    /**
     * sets the value of the output of a 10v module at the given channel.
     *
     * @param chan ?
     * @param value ?
     * @throws java.io.IOException if the port is open
     */
    public static void set10vOut(int chan, double value) throws IOException {}
    private static native void s10vOut(int chan, double value) throws IOException;

    /**
     * @param chan channel to read
     * @return value
     * @throws java.io.IOException if the port is open
     */
    public static double get420In(int chan) throws IOException {}
    private static native double g420In(int chan) throws IOException;

    /**
     * @param chan channel to read
     * @return value
     * @throws java.io.IOException if the port is open
     */
    public static double get420Out(int chan) throws IOException {}
    private static native double g420Out(int chan) throws IOException;

    /**
     * sets the value of the output of a 10v module at the given channel.
     *
     * @param chan channel to set
     * @param value percentage 
     * @throws java.io.IOException if channel is open
     */
    public static void set420Out(int chan, double value) throws IOException {}
    private static native void s420Out(int chan, double value) throws IOException;

    /**
     * This method will return the RTD value for the given channel.
     *
     * @param chan channel to read
     * @return value
     * @throws java.io.IOException if port is open
     */
    public static double getRtdIn(int chan) throws IOException {}
    private static native double gRtdIn(int chan) throws IOException;

    /**
     *
     * This method requests the enumeration of external devices. The addresses returned are used to read/write these devices.
     *
     * @return A long[] of device addresses.
     * @throws java.io.IOException if port is open
     */
    public static long[] externalDeviceList() throws IOException {}
    private static native long[] gExtDevLst();

    /**
     *
     * This method requests the enumeration of external devices in the alarm state. The addresses returned are used to
     * read/write these devices.
     *
     * @return A long[] of device addresses.
     * @throws java.io.IOException if port is open
     */
    public static long[] alarmingDeviceList() throws IOException {}
    private static native long[] gAlarmDevLst();

    /**
     *
     * Obtain 16-bit raw values from a Type 20/FE/FD Quad Analog module.
     *
     * @return An int[] containing four values 0-65535 corresponding to input channels 1 through 4.
     * @param addr the device address as returned by enumeration. Note that full scale is 65520
     * for Type FD &amp; FE 12-bit modules.
     * @throws java.io.IOException if the device is not an analog module with 4 input channels
     */
    public static int[] readAnalogInputs(long addr) throws IOException {}

    /**
     * Obtains the Read Device Block from the external device. The format is defined by the v2 JNIOR Communications Protocol.
     *
     * @return A byte[] containing the Read Device Block as defined for the specific device type in the v2 JNIOR Communications
     * Protocol.
     * @param addr the device address as obtained by enumeration.
     * @throws java.io.IOException if port is open
     */
    public static byte[] readDeviceBlock(long addr) throws IOException {}
    private static native byte[] rDevBlk(long addr);
        
    /**
     * Writes the Write Device Block to the external device. The format is defined by the v2 JNIOR Communications Protocol.
     *
     * @param addr the device address as obtained by enumeration.
     * @param block the Write Device Block as defined in the v2 JNIOR Communications Protocol for the specific device type.
     * @throws java.io.IOException if port is open
     */
    public static void writeDeviceBlock(long addr, byte[] block) throws IOException {}
    private static native void wDevBlk(long addr, byte[] block);

    /**
     * Obtains the state of all relay outputs as a single bit-mapped integer.
     * <p>
     * For the Model 410 the least significant 8 bits D0-D7 represent the state of ROUT1 through ROUT8. Bits D8-D11
     * will reflect the status of an option external 4ROUT module for relay outputs ROUT9 through ROUT12. A second
     * external 4ROUT module if present is represented by bits D12-D15 for ROUT13 through ROUT16.
     * <p>
     * For the Model 412 the least significant 12 bits D0-D11 represent the state of ROUT1 through ROUT12. Bits D12-D15
     * will reflect the status of an option external 4ROUT module for relay outputs ROUT13 through ROUT16.
     * <p>
     * Note that some relays may be hardware jumper configured for Normally-Closed operation. These are set to Normally-Open 
     * by default. If a relay is set for normally-closed operation a '1' will indicate the OPRN condition.
     * 
     * @return An integer where bits each correspond to relay outputs. A '1' is reported for outputs
     * that are CLOSED and a '0' for those that are OPEN. The least significant bit corresponds to ROUT1.
     */
    public static native int getOutputStates();

    /**
     * Obtains the state of a specified Relay Output.
     * <p>
     * The channel is 0-based with 0 specifying ROUT1, 1 for ROUT2, etc. The Model 410 uses channels 0-7 as it 
     * supports ROUT1 through ROUT8. The Model 412 has 12 relay outputs which are specified as channels 0-11.
     * <p>
     * Additional relay outputs may be available if external 4ROUT modules are present. For the Model 410 two
     * external 4ROUT modules may be used. These can be addressed here as channels 8-15 representing ROUT9 through
     * ROUT16. The Model 412 can use a single external module providing channels 12-15 for ROUT13 through ROUT16.
     *
     * @param chan 0-based value specifying a relay output channel
     * @return <code>true</code> if the relay output is CLOSED. Returns <code>false</code> otherwise.
     */
    public static native boolean isOutputSet(int chan);

    /**
     * Establishes the condition for the specified output relays. The selected relays are changed to the indicated state.
     * 
     * Since external 4ROUT relay modules may be commanded this is synchronized with Sensor Port access.
     *
     * @param bits Integer where the bits indicate the desired output condition. A '1' resulting in a
     * closed relay
     * @param mask Integer where the indicate which output relays are to be affected by the command. Those relays 
     * selected by a '1' will change to the state indicated by the bits parameter. Those corresponding
     * with '0' bits will remain unchanged.
     * @throws java.io.IOException if port is open
     */
    public static void setOutputStates(int bits, int mask) throws IOException {}
    private static native void sOutStates(int bits, int mask);

    /**
     * Establish the state of the specified output relay.
     * 
     * Since external 4ROUT relay modules may be commanded this is synchronized with Sensor Port access.
     * 
     * The channel is 0-based with 0 specifying ROUT1, 1 for ROUT2, etc. The Model 410 uses channels 0-7 as it 
     * supports ROUT1 through ROUT8. The Model 412 has 12 relay outputs which are specified as channels 0-11.
     * 
     * Additional relay outputs may be available if external 4ROUT modules are present. For the Model 410 two
     * external 4ROUT modules may be used. These can be addressed here as channels 8-15 representing ROUT9 through
     * ROUT16. The Model 412 can use a single external module providing channels 12-15 for ROUT13 through ROUT16.
     *
     * @param chan 0-based value specifying a relay output channel
     * @param state <code>true</code> closes the associated relay illuminating the corresponding LED. <code>false</code>
     * opens the relay. This opposite would be the case if a relay is hardware jumper configured for Normally-Closed 
     * operation.
     * @throws java.io.IOException if port is open
     */
    public static  void setOutputRelay(int chan, boolean state) throws IOException {}
    private static native void sOutRelay(int chan, boolean state);

    /**
     * Establishes a temporary condition for the specified output relays. The selected relays are changed to the indicated state
     * for a duration specified in milliseconds. When this time period passes the relays are returned to their original states.
     * This is used to pulse outputs with 1 millisecond resolution.
     * 
     * Since external 4ROUT relay modules may be commanded this is synchronized with Sensor Port access.
     *
     * @param bits Integer where the least significant eight bits indicate the desired output condition. A '1' resulting in a
     * closed relay
     * @param mask Integer where the least significant eight bits indicate which output relays are to be affected by the
     * command. Those relays selected by a '1' will change to the state indicated by the data parameter. Those corresponding
     * with '0' bits will remain unchanged.
     * @param milliseconds Integer specifying the duration that the new condition will be held. The pulse duration is to within
     * 1 millisecond +0.2/-0.8
     * @return Returns <code>true</code> if the pulse is successfully executed and <code>false</code> otherwise.
     * @throws java.io.IOException if port is open
     */
    public static boolean setOutputPulsed(int bits, int mask, int milliseconds) throws IOException {}
    private static native boolean sOutPulsed(int bits, int mask, int milliseconds);


    // We need to be sure that we release the port or bad things will happen.
    protected void finalize() throws Throwable {}

}
