package com.integ.common.logging;

import java.io.PrintStream;

public abstract class Logger {

    /**
     * called on the first instance created of any Log subclass. This will start the FileLogService,
     * which is responsible for making sure that the buffered streams get committed within a
     * reasonable amount of time. Buffered streams only get committed when they reach the buffer
     * limit or a flush() is manually called.  The FileLogService manually calls the flush()
     */
    static {
        FileLogService.start();
    }

    /**
     * the PrintStream instance
     */
    protected PrintStream _printStream;

    /**
     * a flag indicating that the log class has been finalized. once it has been finalized we can no
     * longer rely on the
     * FileLogService to manually flush uncommitted buffered stream data. at this point the stream
     * MUST be flushed once
     * it is written to. This is rare but other finalizers might want to write to the log after the
     * FileLogService has
     * been shutdown.
     */
    protected boolean _finalized = false;



    /**
     * sets the PrintStream instance that will be used.
     *
     * @param printStream
     */
    protected void setPrintStream(PrintStream printStream) {
        _printStream = printStream;
    }



    /**
     * gets the PrintStream instance that is being used.
     *
     * @return PrintStream
     */
    public PrintStream getPrintStream() {
        return _printStream;
    }



    /**
     * creates a log entry with preceeding '##' to denote that this entry is VITAL
     *
     * @param vitalString
     */
    public void vital(String vitalString) {
        println(String.format("## %s", vitalString));
    }



    /**
     * creates a normal log entry
     *
     * @param infoString
     */
    public void info(String infoString) {
        println(infoString);
    }



    /**
     * creates a log entry with preceeding '??' to denote that this entry is DEBUG
     *
     * @param debugString
     */
    public void debug(String debugString) {
        println(String.format("?? %s", debugString));
    }



    /**
     * creates a log entry with preceeding ' *' to denote that this entry is a WARNING
     *
     * @param warningString
     */
    public void warn(String warningString) {
        println(String.format(" * %s", warningString));
    }



    /**
     * creates a log entry with preceeding '**' to denote that this entry is an ERROR
     *
     * @param errorString
     * @param ex
     */
    public void error(String errorString, Throwable ex) {
        if (null != ex) {
//            ex.printStackTrace();

            String exceptionMessage = ex.getMessage();
            if (null == exceptionMessage) exceptionMessage = ex.getClass().getName();
            println(String.format("** %s: %s", errorString, exceptionMessage));
//            AppLog.error(errorString, ex);
        } else {
            error(errorString);
        }
    }



    /**
     * creates a log entry with preceeding '**' to denote that this entry is an ERROR
     *
     * @param errorString
     */
    public void error(String errorString) {
        println(String.format("** %s", errorString));
    }



    /**
     * creates a log entry with preceeding '**' to denote that this entry is an ERROR
     *
     * @param ex
     */
    public void error(Throwable ex) {
        println(String.format("** %s", ex.getMessage()));
//        AppLog.error(ex);
    }



    /**
     * creates a log entry with preceeding '***' to denote that this entry is an FATAL to the
     * programs execution.
     *
     * @param fatalString
     */
    public void fatal(String fatalString) {
        println(String.format("*** %s", fatalString));
    }



    /**
     * prints a line to the PrintStream instance
     *
     * @param s
     */
    protected void println(String s) {
        _printStream.println(s);
    }



    protected void print(String s) {
        _printStream.print(s);
    }



    protected abstract void flush();
}

