package com.integ.common.logging;

import com.integpg.system.JANOS;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;

class RollingLogOutputStream extends LogOutputStream {

    /**
     * creates a RollingLogOutputStream with the specified LogOptions
     *
     * @param logOptions
     * @throws FileNotFoundException
     * @since 5.0
     */
    public RollingLogOutputStream(LogOptions logOptions) throws FileNotFoundException {
        super(logOptions);

        // we need to see if a .bak file exists.  if it does then this indicates that the rolling file logic did 
        // not complete before.
        if (_bakFile.exists()) {
            // if the log file exists then we should remove the .bak file.  if it doesnt then just rename the 
            // .bak file to our log file
            if (_file.exists()) {
                JANOS.syslog("removing orphaned " + _bakFile.getPath() + " file left during rolling log logic");
                _bakFile.delete();
            } else {
                JANOS.syslog("saved orphaned " + _bakFile.getPath() + " file left during rolling log logic");
                _bakFile.renameTo(_file);
            }
        }
    }



    /**
     * determines if the length of bytes to be logged will surpass the maximum file size. If it will then the rolling
     * file logic is invoked.
     *
     * @param bytes
     * @param off
     * @param len
     * @return
     * @throws IOException
     */
    @Override
    protected int preWrite(byte[] bytes, int off, int len) throws IOException {
        // we are going to write some data.  if the length of new data puts us over the maximum size then we 
        // must trim the file to make room for it.
        if (_logOptions.getMaxFileSize() < _file.length() + len) {
            rollFile();
        }
        return 0;
    }



    private void rollFile() throws IOException {
        long start = System.currentTimeMillis();

        _file.renameTo(_bakFile);
        copyfile(_bakFile, _clusterSize, _file, 0);
        _bakFile.delete();

        long elapsed = System.currentTimeMillis() - start;
        System.out.println(String.format("took %lld to roll %s", elapsed, _file.getPath()));
    }



    private void copyfile(File srcFile, int srcPos, File dstFile, int dstPos) throws IOException {
//            long start = System.currentTimeMillis();

        RandomAccessFile in = null;
        RandomAccessFile out = null;

        synchronized (_file) {
            try {
                // a flag indicating if this is the first write block.  if it is then we should find the end of 
                // the first line after the srcPos
                boolean firstWrite = true;

                in = new RandomAccessFile(srcFile, "r");
                in.seek(srcPos);

                out = new RandomAccessFile(dstFile, "rw");
                out.seek(dstPos);

                byte[] buf = new byte[32 * 1024];
                int len;
                while ((len = in.read(buf, 0, buf.length)) > 0) {
                    int pos = 0;
                    if (firstWrite) {
                        firstWrite = false;
                        for (int i = 0; i < len; i++) {
                            if ('\n' == buf[i]) {
                                pos = i + 1;
                                break;
                            }
                        }
                    }
                    out.write(buf, pos, len - pos);
                }
            } catch (Exception ex) {
                System.err.println("Error copying file " + srcFile.getPath() + " to " + dstFile.getPath() + ": " + ex.toString());
            } finally {
                if (null != in) in.close();
                if (null != out) out.close();
            }
        }

//            long elapsed = System.currentTimeMillis() - start;
//            System.out.println("took " + elapsed + " to copy " + srcFile.getAbsolutePath() + " to " + dstFile.getAbsolutePath());
    }

}

