//	JNIOR Automation Network Operating System (JANOS)
//  CopyRight (C) 2012-2016 INTEG process group inc. All Rights Reserved.

/* ---------------------------------------------------------------------------
 * This software is INTEG process group inc proprietary. This source code
 * is for internal company use only and any other use without prior
 * consent of INTEG process group inc is prohibited.
 *
 * Author: Bruce Cloutier
 * Inception Date: 
 * -------------------------------------------------------------------------*/

package com.integpg.system;

/**
 * JANOS supports a message loop (or message pump) supporting interprocess communications.
 * Each process may tap the loop and monitor messages. If the process receives a message 
 * intended for only that process it may consume the message and act upon it as required.
 * If the message is a broadcast meant for all processes or intended for another process
 * the message MUST be reposted to the message loop. All of these message are of the
 * SystemMsg type.
 * 
 * @author Bruce Cloutier
 */
public class SystemMsg
{

    /**
     * SM_SHUTDOWN (0x01) This message is sent by JANOS indicating that the system is shutting down.
     * When received this MUST be reposted. A process then may have as much as 5 seconds to prepare
     * for shutdown. The process should save critical data and exit. The system will terminate all
     * remaining processes before shutdown.
     * <p>
     * The {@code msg} will indicate the reason for the shutdown. This generally will indicate "reboot".
     */
    public static final int SM_SHUTDOWN		= 0x01;

    /**
     * SM_PROBE (0x02) This message is sent by JANOS periodically to monitor the health of the message loop. 
     * All processes tapping the message pump MUST repost this message. If this message is delayed or fails 
     * to circulate the loop the system may terminate the offending process or otherwise reboot.
     */
    public static final int SM_PROBE		= 0x02;

    /**
     * SM_GCRUN (0x10) This message is sent by JANOS to indicate that the JVM Garbage Collector (GC) is 
     * running. All processes tapping the message pump SHOULD repost this message.
     */
    public static final int SM_GCRUN		= 0x10;

    /**
     * SM_WATCHDOG (ox11) This message is sent if an Application Watchdog has been configured to send
     * a system message when it is triggered. The {@code msg} will contain text specified by the application or
     * other identification (may be "unknown"). A process tapping the message pump MUST repost this message. The
     * originator may consume the message if that is the intended use.
     */
    public static final int SM_WATCHDOG		= 0x11;

    /**
     * SM_REGUPDATE (0x40) This message is sent by JANOS whenever a Registry Key is created, updated or removed.
     * The {@code msg} will indicate the key name. All processes tapping the message pump MUST repost this message.
     * A process may then query the Registry for the key's value if needed. 
     */
    public static final int SM_REGUPDATE	= 0x40;

    /**
     * SM_WEBSTARTUP (0x60) This message is sent by the WebServer when the service is started. All processes
     * tapping the message pump MUST repost this message. The WebServer process will shut down after a period 
     * of inactivity. This message will indicate the restoration of services when an external access is attempted.
     */
    public static final int SM_WEBSTARTUP	= 0x60;

    /**
     * SM_WEBSHUTDOWN (0x61) This message is sent by the WebServer when the service is stopped. All processes
     * tapping the message pump MUST repost this message. The WebServer process will shut down after a period
     * of inactivity.
     */
    public static final int SM_WEBSHUTDOWN	= 0x61;

    /**
     * SM_PROTCMDMSG (0x70) This message is sent by the JNIOR Protocol message upon receipt of a Custom
     * Command message type 255. An external client may issue this JNIOR Protocol command. All processes tapping 
     * the message pump MUST repost this message unless the process is specifically intended to process the 
     * custom command. In that case it may consume the message.
     * <p>
     * {@code msg} contains a binary structure as follows:
     * <pre>
     *   index     size    description
     *     0        4      IP address of remote protocol client 
     *     4        2      IP port of remote protocol client
     *     6        N      Variable length Custom Command as supplied by client
     * </pre>
     */
    public static final int SM_PROTCMDMSG	= 0x70;

    /**
     * SM_PROTCMDRESP (0x71) This message will be generated by an application designed to respond to a
     * JNIOR Protocol Custom Command message type 255. All processes tapping the message pump MUST repost
     * this message. If appropriate it will be consumed by the JNIOR Protocol server.
     * <p>
     * {@code msg} must contain a binary structure as follows:
     * <pre>
     *   index     size    description
     *     0        4      IP address of destination protocol client 
     *     4        2      IP port of destination protocol client
     *     6        N      Variable length response
     * </pre>
     * <p>
     * The JNIOR Protocol server will consume this message. It will scan its client list for clients with 
     * matching IP address and port. If the client is active the response will be transmitted.
     * <p>
     * This message should be sent ONLY in response to a SM_PROTCMDMSG reflecting the IP address and port 
     * from that message. Only one response for every custom command should be sent. Unsolicited responses
     * may corrupt the protocol stream.
     */
    public static final int SM_PROTCMDRESP	= 0x71;

    /**
     * SM_PIPEOPEN (0x80) This message is sent by the WebServer when there is a request to open a Websocket pipe.
     * All processes MUST repost this message with the exception of an application designed to process the specific
     * Websocket pipe. In that case the message SHOULD be consumed.
     * <p>
     * The pipe is identified by a numeric "app" parameter supplied in the Websocket URL. Pipe identifiers MUST be
     * an EVEN integer greater than or equal to 1024.
     * <p>
     * {@code msg} will contain a binary structure as follows:
     * <pre>
     *   index     size    description
     *     0        4      IP address of client 
     *     4        2      IP port of client
     *     6        2      Pipe identifier (app number)
     * </pre>
     * <p>
     * Al Websocket message traffic will then be passed through the message pump using the User Message Number
     * (numbers greater than or equal to 1024) defined by the "app" parameter. An application designed to service
     * this pipe will respond using the message number one greater (app + 1). The WebServer will forward responses
     * through the Websocket connection.
     * <p>
     * The format of pipe messages is completely arbitrary and application specific. For instance this might
     * typically contain JSON message structures.
     */
    public static final int SM_PIPEOPEN		= 0x80;

    /**
     * SM_PIPECLOSE (0x81) Sent by the WebServer when a Websocket pipe connection closes. All processes tapping
     * the message pump MUST repost this message with the exception of the application designed to process the specific
     * Websocket pipe. In that case the message SHOULD be consumed.
     * <p>
     * The pipe is identified by the numeric "app" parameter supplied in the original Websocket URL.
     * <p>
     * {@code msg} will contain a binary structure as follows:
     * <pre>
     *   index     size    description
     *     0        4      IP address of client 
     *     4        2      IP port of client
     *     6        2      Pipe identifier (app number)
     * </pre>
     * <p>
     * All incoming message traffic from the pipe will cease and responses ignored.
     */
    public static final int SM_PIPECLOSE	= 0x81;

    /**
     * owner defines the process originating the message
     */
    protected long owner;

    /**
     * id contains a unique message ID (sequential)
     */
    protected int id;

    /**
     * time indicates the uptime in milliseconds when the message originated
     */
    protected long time;

    /**
     * history indicates message queues having processed the message
     */
    protected long history;

    /**
     * type the message type (SM_PROBE etc.)
     */
    public int type;

    /**
     * msg the message content (string, structure, etc.)
     */
    public byte[] msg;
    
    /**
     * Obtains the unique message ID assigned to the message.
     * 
     * @return message ID
     */
    public int getID()
    {}
    
    /**
     * Returns the message as a string. This may not always be appropriate.
     * 
     * @return String representation of the message content. Messages may contain binary however.
     */
    @Override
    public String toString() {}
}

