 /*
  JNIOR Monitor/Configuration Page
  INTEG Process Group, Inc., 2919 E Hardies Rd, Gibsonia PA 
  724.933.9350
  
  File: console.js
  
  Javascript to simulate a console connection with access to the JANOS Command Line
  Interpreter. This upgraded approach emulates a limited but now standard Xterm
  terminal.
  
  The user is hereby granted license to use, modify, and redistribute the contents of this
  file for any purpose, commercial or otherwise. No prior authorization by INTEG Process 
  Group, Inc. is required.
*/

/*
  XTERM emulation
  
  The original terminals supported a display surface of fixed dimensions (24 by 80)
  for instance. Each location was occupied by a blank or character each with
  variaous attributes such as bold or color (foreground and backfround). Any 
  ability to scroll back was supported by the connected equipment if it was
  possible at all.
  
  Here we utilize a DIV container that supports scrolling. The character data
  to be displayed is stored and renedered using HTML. There is no concept of
  fixed positions. The window is of arbitrary dimensions and can be resized.
  
  SPAN elements are used to render non-default attributes. These apply to groups
  of characters and not on a character-by-character basis. The external system
  cannot embed HTML as tags are blocked by forcing &lt;, &gt; and &amp; on the
  required characters.
*/

var caret = 0;	  
var col = 0;
var ins = 1;
var nlines = 0;

var connected = false;
var transfer = false;
var displaycache = "";

var esc = false;
var seq = false;
var parm;
var nparm;

var showcursor = true;
var mouse = false;
var mouse_cx = 0;
var mouse_cy = 0;
var mouse_btn = 0;
var mouse_delay = null;
var cursor = '<span id="caret">|</span>';
var columns = -1;
var lines = -1;
var resize;
var lnheight = 0;
var chwidth = 0;
var padLeft = 0;
var paddingX = 0
var cursint = 0;

// content array format 0xAABBFFCCCCCC where A-attribute, B-background,
//  F-foreground, C-character (Unicode)
// blank buffer with new line
var bfbufr = [ 0x0A];

// attributes
var lastattr = 0;
var foreattr = 0;
var backattr = 0;
var charattr = 0;

// **************************************************
// process received messages
var console_onmessage = chan.onmessage;
chan.onmessage = function(evt) {
	var jobj = JSON.parse(evt.data);

  if (jobj['Message'] === 'Console Stdout') {
  		displaycache = displaycache.concat(jobj['Data']);
  }	  

  else if (console_onmessage)
		console_onmessage(evt);	// chain message processing
}

function rundisplay() {
	if (displaycache.length > 0) {
		display(displaycache);
		displaycache = "";
	}
}

setInterval(rundisplay, 100);


// **************************************************
// bfbufr utilities

// return character value at position
function bfcharAt(pos) {
	return (bfbufr[pos] & 0xFFFFFF);
}

// search forward for character
function bfindexOf(ch, pos = 0) {
	if (pos < 0 || pos >= bfbufr.length)
		return (-1);

	for (; pos < bfbufr.length && bfcharAt(pos) != ch; pos++);
	if (pos == bfbufr.length)
		return (-1);
	return (pos);
}

// return attributes
function bfattrAt(pos) {
	return (Math.floor(bfbufr[pos] / 0x1000000));
}


// **************************************************
// Updates the DIV element with the stream data provided.
function display(cdata) {
	var screen = document.getElementById("screen");
	var status = document.getElementById("status");
	
	// enforce some maximum buffer size in lines
	maxlines = 1250
	if (nlines > maxlines + 25) {
		pos = 0;
		while (pos >= 0 && nlines > maxlines) {
			pos = bfindexOf(0x0D, pos + 1);
			nlines--;
		}
		
		// trim
		bfbufr = bfbufr.slice(pos);
		caret -= pos;
	}
	
	// process incoming data character by character
	for (i = 0; i < cdata.length; i++) {
		
		chcode = cdata.charCodeAt(i);
		ch = cdata.charAt(i);
		
		// XTERM emulation
		//  We handle ANSI escape sequences. We process only ESC-[ sqeuences
		
		// detect incoming ESC sequence
		if (chcode == 0x1B) {
			esc = true;
			seq = false;
			continue;
		}

		// detect ESC-[ sequence		
		if (esc && ch == '[') {
			seq = true;
			nparm = 0;
			parm = [];
			parm[nparm] = 0;
			continue;
		}
		
		// process ESC-[ sequences
		if (seq) {
			esc = false;
			
			// accumulate parameter value
			if (chcode >= 0x30 && chcode <= 0x39) {
				parm[nparm] = 10 * parm[nparm] + chcode - 0x30;
				continue;
			}
			
			// collect parameters - advance to the next one
			if (ch == ';') {
				nparm++;
				parm[nparm] = 0;
				continue;
			}
			
			// XY cursor movement not implemented
			if (ch == 'H') {
				seq = false;
				continue;
			}
			
			// cursor location query not implemented
			if (ch == 'n') {
				seq = false;
				continue;
			}
			
			// CONTROL
			// settings - character ignored
			if (ch == '?') {
				continue;
			}
			
			// disable control
			if (ch == 'l') {
				seq = false;
				nparm++;
				
				// process parameters
				for (n = 0; n < nparm; n++) {
					// disable cursor
					if (parm[n] == 25)
						showcursor = false;
						
					// disable mouse
					else if (parm[n] == 1000)
						mouse = false;
					else if (parm[n] == 1006)
						mouse = false;
					else if (parm[n] == 1015)
						mouse = false;
				}
					
				continue;
			}			
			
			// enable control
			if (ch == 'h') {
				seq = false;
				nparm++;
				
				// process parameters
				for (n = 0; n < nparm; n++) {
					//enable cursor
					if (parm[n] == 25)
						showcursor = true;
						
					// enable mouse
					else if (parm[n] == 1000)
						mouse = true;
					else if (parm[n] == 1006)
						mouse = true;
					else if (parm[n] == 1015)
						mouse = true;
				}
					
				continue;
			}			
			
			// UP ARROW
			if (ch == 'A') {
				seq = false;
				
				if (parm[0] == 0)
					parm[0] = 1;
					
				while (parm[0]-- > 0) {

					// locate position on this line
					for (pos = 0; caret > 0 && bfcharAt(caret - 1) != 0x0D; pos++)
						caret--;

					// move to previous line
					caret--;
		  		while (caret > 1 && bfcharAt(caret - 1) != 0x0D) 
		  			caret--;
		  		col = 0;

		  		// now to same column if we can	
		  		while (pos > 0 && bfcharAt(caret + 1) != 0x0D) {
		  			caret++;
		  			col++;
		  			pos--;
		  		}

		  	}
				continue;
			}
			
			// DOWN ARROW
			if (ch == 'B') {
				seq = false;
				
				if (parm[0] == 0)
					parm[0] = 1;
					
				while (parm[0]-- > 0) {

					// locate position on this line
					pos = 0;
					for (; caret > 1 && bfcharAt(caret - 1) != 0x0D; pos++)
						caret--;

					// move to next line
		  		while (caret < bfbufr.length && bfcharAt(caret) != 0x0D) 
		  			caret++;
		  		if (caret == bfbufr.length)
		  				bfbufr[caret] = 0x0D;
		  		caret++;
		  		col = 0;

		  		// now to same column if we can	
		  		while (pos > 0 && caret + 1 < bfbufr.length && bfcharAt(caret + 1) != 0x0D) {
		  			caret++;
		  			col++;
		  			pos--;
		  		}

		  	}
				continue;
			}
			
			// RIGHT ARROW
			if (ch == 'C') {
				seq = false;
				
				if (parm[0] == 0)
					parm[0] = 1;
					
				while (parm[0]-- > 0) {
					if (bfcharAt(caret) != 0x0D) {
						caret++;
						col++;
					}
				}
				continue;
			}
			
			// LEFT ARROW
			if (ch == 'D') {
				seq = false;
				
				if (parm[0] == 0)
					parm[0] = 1;
					
				while (parm[0]-- > 0) {
					if (bfcharAt(caret - 1) != 0x0D) {
						caret--;
						if (col > 0)
							col--;
					}
				}
				continue;
			}
			
			// clear to end of line
			if (ch == 'K') {
				seq = false;
				
				pos = bfindexOf(0x0D, caret);
				if (pos > caret) {
					arr1 = bfbufr.slice(0, caret);
					arr2 = bfbufr.slice(pos);
					bfbufr = arr1.concat(arr2);
				}
				else if (pos == -1)
					bfbufr = bfbufr.slice(0, caret);
				continue;
			}
			
			// formatting
			if (ch == 'm') {
				seq = false;
				nparm++;
				
				// process multiple params
				for (n = 0; n < nparm; n++) {
					
					// reset
					if (parm[n] == 0) {
						foreattr = 0;
						backattr = 0;
						charattr = 0;
					}
					
					// reverse video
					else if (parm[n] == 7)
						charattr |= 0x01;
					
					// bold
					else if (parm[n] == 1)
						charattr |= 0x02;
					
					// italic
					else if (parm[n] == 3)
						charattr |= 0x04;
					
					// underline
					else if (parm[n] == 4)
						charattr |= 0x08;
					
					// faint/dim
					else if (parm[n] == 2)
						charattr |= 0x10;
					
					// 16-color foreground
					else if (parm[n] >= 30 && parm[n] <= 37)
						foreattr = 8 + (parm[n] - 30);
					else if (parm[n] == 39)
						foreattr = 0;
						
					// 256-color foreground
					else if (n == 0 && nparm == 3 && parm[0] == 38 && parm[1] == 5) {
						foreattr = parm[2];
						break;
					}
					
					// 16-color background
					else if (parm[n] >= 40 && parm[n] <= 47)
						backattr = 8 + (parm[n] - 40);
					else if (parm[n] == 49)
						backattr = 0;

					// 256-color background
					else if (n == 0 && nparm == 3 && parm[0] == 48 && parm[1] == 5) {
						backattr = parm[2];
						break;
					}
				
				}
				continue;
			}
			
			// clear screen
			if (ch == 'J')
			{
				seq = false;

				if (parm[0] == 2)
				{
					bfbufr = [ 0x0D ];
					caret = 0;
				}

				continue;
			}
			
		}
		
		// reset escape sequencing
		esc = false;
		seq = false;
		
		// carriage return moves caret to beginning of line
		if (chcode == 0x0D) {

			// move to beginning or position after last newline
  		while (caret > 0 && bfcharAt(caret - 1) != 0x0D) 
  			caret--;

  		col = 0;
  		continue;
  	}
		
		// newline moves to the next line and resets the overstrike mode
		if (chcode == 0x0A) {

			// move ahead to end or next newline
  		while (caret < bfbufr.length && bfcharAt(caret) != 0x0D) 
  			caret++;

  		// if at end append newline
  		if (caret == bfbufr.length) {
  			bfbufr[caret] = 0x0D;
  		}
  		
  		//position after newline
			caret++;
			col = 0;
  		ins = 1;
  		continue;
  	}
  	
  	// backspace 
		if (chcode == 0x08) {

			if (caret > 0) {
				caret--;
				if (col > 0)
					col--;
			}

			continue;
		}
		
		// apply attributes
		if (charattr & 0x01) {
			// reverse video
			fore = foreattr ? foreattr : 15;
			back = backattr ? backattr : 16;
			chcode = chcode + 0x1000000 * back + 0x100000000 * fore;
		} 
		else if (charattr & 0x10) {
			// faint/dim
			fore = 7;
			if (foreattr) {
				if (foreattr >= 9 && foreattr <= 15)
					fore = foreattr - 8;
				else if (foreattr >= 248)
					fore = foreattr - 8;
				else 
					fore = foreattr;
			}
			chcode = chcode + 0x1000000 * fore + 0x100000000 * backattr;
		}
		else
			chcode = chcode + 0x1000000 * foreattr + 0x100000000 * backattr;
			
		chcode += 0x10000000000 * charattr;
		
		// buffer data
		col++;
		
		//  if caret is at an newline then data is inserted otherwise we overstrike
		if (caret < bfbufr.length) {
			arr1 = bfbufr.slice(0, caret);
			
			if (bfcharAt(caret) == 0x0D)
				arr2 = bfbufr.slice(caret);
			else
				arr2 = bfbufr.slice(caret + 1);
			
			arr1.push(chcode);
			bfbufr = arr1.concat(arr2);
			caret++;	
		}
		else
			bfbufr[caret++] = chcode;
	}
		
	// generate innerHTML	
	content = "";
	lastattr = 0;
	nlines = 0;
	for (n = 0; n < bfbufr.length; n++) {
		
		// count saved lines
		if (bfcharAt(n) == 0x0D)
			nlines++;
		
		// check for attribute change
		attr = bfattrAt(n);
		if (attr != lastattr && lastattr != 0)
			content += "</span>";
		
		// insert caret
		if (showcursor && n == caret)
			content += cursor;
			
		// check for new attributes
		if (attr != lastattr) {

			if (attr) {
				span = '<span style="';

				fore = attr & 0xFF;
				if (fore != 0)
					span += "color:" + color[fore] + ";";

				back = (attr >> 8) & 0xFF;
				if (back != 0)
					span += "background:" + color[back] + ";";
					
				font = (attr >> 16) & 0xFF;
				if (font & 0x02)
					span += "font-weight:bold;";
				if (font & 0x04)
					span += "font-style:italic;";
				if (font & 0x08)
					span += "text-decoration:underline;";

				span += '">';
				
				content += span;
			}
			
			lastattr = attr;
		}
			
		// we need to prevent embedding HTML
		ch = String.fromCharCode(bfcharAt(n));
		if (ch == '&')
			ch = '&amp;';
		else if (ch == '<')
			ch = '&lt;';
		else if (ch == '>')
			ch = '&gt;';
			
		content += ch;
	}
	
	// close attributes
	if (lastattr != 0) 
		content += "</span>";
	
	// insert caret
	if (showcursor && n == caret)
		content += cursor;

	// keeps scroll pinned to bottom unless cursor is off
	if (showcursor)
		noscroll = true;
	else
		noscroll = (screen.scrollTop == screen.scrollHeight - screen.clientHeight ? true : false);
		
	screen.innerHTML = content;
	if (noscroll)
		screen.scrollTop = screen.scrollHeight;

	status.innerHTML = (ins === 1 ? "Ins " : "Ovr ") + col.toString();
}


// **************************************************
// function to handle mouse wheel events
function onMousewheel(e) {
	if (!mouse)
		return;
		
	// stop default
	e.preventDefault();

	// fetch coordinates
	var x = e.offsetX;
	var y = e.offsetY;
	
	// remove padding
	getScreenParams();
	x -= padLeft;
	
	// check bounding
	x = x < 0 ? 0 : x;
	y = y < 0 ? 0 : y;
	
	// determine character position
	var cx = 1 + Math.round(x / chwidth);
	var cy = 1 + Math.floor(y / lnheight);
	
	// transmit mouse movement with wheel flag
	var btn = 64 + (e.deltaY >= 0 ? 1 : 0);
	var str = "\x1b[<" + btn + ";" + cx + ";" + cy + "M";
	chan.sendJson({'Message': 'Console Stdin', 'Data': str});
	
}


// **************************************************
// functions to handle mouse events when enabled
function onMousedown(e) {
	mouseclick(e, true);
}

function onMouseup(e) {
	mouseclick(e, false);
}

function mouseclick(e, down) {
	if (!mouse)
		return;
		
	// stop default
	e.preventDefault();

	// fetch coordinates
	var x = e.offsetX;
	var y = e.offsetY;
	
	// remove padding
	getScreenParams();
	x -= padLeft;
	
	// check bounding
	x = x < 0 ? 0 : x;
	y = y < 0 ? 0 : y;
	
	// determine character position
	var cx = 1 + Math.round(x / chwidth);
	var cy = 1 + Math.floor(y / lnheight);
	
	// encode button parameter
	var btn = e.button & 3;
	if (e.shiftKey)
		btn += 4;
	if (e.metaKey)
		btn += 8;
	if (e.ctrlKey)
		btn += 16;
	if (e.altKey)
		btn += 32;

	// cache to stabilize m,ouse performance
	if (down) {
		
		if (mouse_delay == null) {
			mouse_cx = cx;
			mouse_cy = cy;
			mouse_btn = btn;
		}
		else {
				clearTimeout(mouse_delay);
		}
		
		// set timeout
		mouse_delay = setTimeout( function() { mouse_delay = null; }, 500);
	}
	
	// button state
	var state = down ? "M" : "m"
	
	// transmit mouse movement
	str = "\x1b[<" + mouse_btn + ";" + mouse_cx + ";" + mouse_cy + state;
	chan.sendJson({'Message': 'Console Stdin', 'Data': str});
}

// **************************************************
// function to squelch mouse events
function MousePrevent(e) {
	
	// When we are handling mouse events we disregard clicks
	if (mouse)
		e.preventDefault();
}


// **************************************************
// function to process keystrokes in the text area
function onKeypress(e) {
	e.preventDefault();
	if (connected && !transfer) {
		ch = e.which;
		chan.sendJson({'Message': 'Console Stdin', 'Data': String.fromCharCode(ch)});
	}
}


// **************************************************
// determine line height
function getScreenParams() {
	
		// need to perform just once
		if (lnheight)
			return;
			
		// get padding for working with text area
		var screen = document.getElementById("screen");
		var css = getComputedStyle(screen);
		padLeft = parseFloat(css.paddingLeft);
		paddingX = padLeft + parseFloat(css.paddingRight);
		
		// fetch font dimensions
    var temp = document.createElement(screen.nodeName);
    temp.setAttribute("style", "margin:0; padding:0;  display:inline-block;"
        + "font-family:" + (screen.style.fontFamily || "inherit") + "; "
        + "font-size:" + (screen.style.fontSize || "inherit"));
    temp.innerHTML = "A";

    screen.appendChild(temp);
    rect = temp.getBoundingClientRect();
    lnheight = rect.height;
    chwidth = rect.width;
    screen.removeChild(temp);
}

// **************************************************
// function updates screen area when resized
function ResizeCheck() {

	// check characer dims
	getScreenParams();
		
	var screen = document.getElementById("screen");
	width = Math.floor((screen.clientWidth - paddingX) / chwidth);
	height = Math.floor(screen.clientHeight / lnheight);
	
	// exclude padding
	
	if (columns != width || lines != height) {
		columns = width;
		lines = height;
		chan.sendJson({'Message': 'Console Env', 'LINES': height.toString(), 'COLUMNS': width.toString() });
	}
}

// **************************************************
// function to process keystrokes in the text area
function onKeydown(e) {
	if (connected && !transfer) {
		var status = document.getElementById("status");
		
		ch = e.which;
		str = null;
		
		// pass recognized special keystrokes.
		if (ch === 8 || ch == 0x1b)
			str = String.fromCharCode(ch);
			
		// TAB
		else if (ch == 0x09) {
			if (e.shiftKey)
				str = "\x1b[Z";
			else
				str = String.fromCharCode(ch);
		}
			
		// Ctrl-A (drops/raises anchor)
		else if (e.ctrlKey && ch === 0x41) {
			str = String.fromCharCode(1);
		}
			
		// Ctrl-U (toggles UTF-8 accents)
		else if (e.ctrlKey && ch == 0x55) {
			str = String.fromCharCode(0x15);
		}
		
		// Ctrl-F
		else if (e.ctrlKey && ch === 0x46) {
			str = String.fromCharCode(6);
		}

		// Ctrl-R
		else if (e.ctrlKey && ch === 0x52) {
			str = String.fromCharCode(0x12);
		}
		
		// Ctrl-C
		else if (e.ctrlKey && ch === 0x43) {
			str = String.fromCharCode(3);
		}

		// Ctrl-X
		else if (e.ctrlKey && ch === 0x58) {
			str = String.fromCharCode(0x18);
		}

		// Ctrl-V
		else if (e.ctrlKey && ch === 0x56) {
			str = String.fromCharCode(0x16);
		}

		// Ctrl-Q
		else if (e.ctrlKey && ch === 0x51) {
			str = String.fromCharCode(0x11);
		}
		
		// Ctrl-K
		else if (e.ctrlKey && ch === 0x4b) {
			str = String.fromCharCode(0x0b);
		}
		
		// Ctrl-G
		else if (e.ctrlKey && ch === 0x47) {
			str = String.fromCharCode(0x07);
		}
		
		// Ctrl-S
		else if (e.ctrlKey && ch === 0x53) {
			str = String.fromCharCode(0x13);
		}
		
		else if (ch === 0x21)	// PAGE UP
			str = "\x1b[5~";
		else if (ch === 0x22)	// PAGE DOWN
			str = "\x1b[6~";
		else if (ch == 0x72)	// F3 key
			str = "\x1b[13~";
		else if (ch == 0x74)	// F5 key
			str = "\x1b[15~";
		
		// Up Arrow
		else if (ch === 0x26)	{
			if (e.altKey)
				str = "\x1b[;3A";
			else {
				str = "\x1b[A";
				ins = 1;
			}
		}
		
		// Down Arrow
		else if (ch === 0x28)	{
			if (e.altKey) 
				str = "\x1b[;3B";
			else {
				str = "\x1b[B";
				ins = 1;
			}
		}
		
		// Right Arrow
		else if (ch === 0x27)
			str = "\x1b[C";
			
		// Left Arrow	
		else if (ch === 0x25)
			str = "\x1b[D";
			
		// End and Ctrl-End
		else if (ch === 0x23) {
			if (e.ctrlKey)
				str = "\x1b[4;5~";
			else 
				str = "\x1b[4~";
		}
			
		// Home and Ctrl-Home
		else if (ch === 0x24) {
			if (e.ctrlKey)
				str = "\x1b[1;5~";
			else
				str = "\x1b[1~";
		}
			
		// Del
		else if (ch === 0x2e) 
			str = "\x7f";
		
		// Ins
		else if (ch === 0x2d) {
			str = "\x1b[2~";
			ins ^= 1;
		}
		
		if (str != null)
		{
			e.preventDefault();
			chan.sendJson({'Message': 'Console Stdin', 'Data': str});
		}

		status.innerHTML = (ins === 1 ? "Ins " : "Ovr ") + col.toString();
	}
}

// **************************************************
function onCut(screen) {
	var content = screen.innerHTML;
	setTimeout( function() {
		screen.innerHTML = content;
	}, 4)
}

// **************************************************
function onPaste(e) {
  var clipboardData, pastedData, str, n;

  e.stopPropagation();
  e.preventDefault();

  clipboardData = e.clipboardData || window.clipboardData;
  pastedData = clipboardData.getData('Text');
  
  // Cannot paste multi-line data
  for (n=0; n<pastedData.length; n++) 
  	if (pastedData.charCodeAt(n) < 0x20 || pastedData.charCodeAt(n) >= 0x7f)
  		break;
  		
  str = pastedData.substring(0, n);
	chan.sendJson({'Message': 'Console Stdin', 'Data': str});
}

// toggle caret visibility
function togvis() {
	var caret = document.getElementById('caret');
	if (caret)
		caret.style.visibility = (caret.style.visibility == 'hidden' ? '' : 'hidden');
}
	
// **************************************************
// session
function onOpen() {
	chan.sendJson({'Message': 'Console Open', 'TERM': 'xterm-256color' });
	document.getElementById('close').className = "button";
	document.getElementById('open').className = "button hide";

	var screen = document.getElementById("screen");
	screen.addEventListener('keydown', onKeydown );
	screen.addEventListener('keypress', onKeypress );
	screen.addEventListener('mousedown', onMousedown );
	screen.addEventListener('mouseup', onMouseup );
	screen.addEventListener('wheel', onMousewheel );
	screen.addEventListener('mouseup', MousePrevent );
	screen.addEventListener('click', MousePrevent );
	screen.addEventListener('dblclick', MousePrevent );
	screen.addEventListener('contextmenu', MousePrevent );
	
	connected = true;
	transfer = false;
	screen.className = "screen";
	screen.focus();
	
	// establish blinking for cursor
	if (dopoll && cursint == 0) 
		cursint = setInterval(togvis, 750);
	
	// move past screen data
	var screen = document.getElementById("screen");
	screen.innerHTML += "\n";
	caret = screen.innerHTML.length;
	
	// set console environment
	columns = lines = -1;
	resize = setInterval(ResizeCheck, 1000);
}

function onClose() {
	chan.sendJson({'Message': 'Console Close'});
	document.getElementById('close').className = "button hide";
	document.getElementById('open').className = "button";
	document.getElementById('open').focus();
	document.getElementById('status').innerHTML = "No Active Session";
	connected = false;
	transfer = false;

	var screen = document.getElementById("screen");
	screen.removeEventListener('keydown', onKeydown );
	screen.removeEventListener('keypress', onKeypress );
	clearInterval(resize);
	clearInterval(cursint);
	cursint = 0;
	screen.className = "screen inactive";
	}
	
// **************************************************
// checks if console should be reopened
chan.onopen = function() {
	if (connected)
		setTimeout(onOpen, 1000);
}

function onClear() {
	var screen = document.getElementById("screen");
	var status = document.getElementById("status");
	
	screen.focus();			
	screen.innerHTML = "";
	bfbufr = [ 0x0D ];
	caret = 0;
	col = 0;
	
	if (connected && !transfer)
		status.innerHTML = (ins === 1 ? "Ins " : "Ovr ") + col.toString();
	else
		status.innerHTML = "No Active Session";		
};
	
function scnpaste(e) {
	if (e.dataTransfer) {
		var data = e.dataTransfer.getData("Text");
		if (data) {
			removeSel();
			chan.sendJson({'Message': 'Console Stdin', 'Data': data});
			e.stopPropagation();
		  e.preventDefault();
		}
	}
}


// **************************************************
// XTERM color mappings
var color = [
	'#000000', '#800000', '#008000', '#808000', // 003
	'#000080', '#800080', '#008080', '#C0C0C0', // 007
	'#808080', '#FF0000', '#00FF00', '#FFFF00', // 011
	'#0000FF', '#FF00FF', '#00FFFF', '#FFFFFF', // 015
	'#000000', '#00005f', '#000087', '#0000af', // 019
	'#0000d7', '#0000ff', '#005f00', '#005f5f', // 023
	'#005f87', '#005faf', '#005fd7', '#005fff', // 027
	'#008700', '#00875f', '#008787', '#0087af', // 031
	'#0087d7', '#0087ff', '#00af00', '#00af5f', // 035
	'#00af87', '#00afaf', '#00afd7', '#00afff', // 039
	'#00d700', '#00d75f', '#00d787', '#00d7af', // 043
	'#00d7d7', '#00d7ff', '#00ff00', '#00ff5f', // 047
	'#00ff87', '#00ffaf', '#00ffd7', '#00ffff', // 051
	'#5f0000', '#5f005f', '#5f0087', '#5f00af', // 055
	'#5f00d7', '#5f00ff', '#5f5f00', '#5f5f5f', // 059
	'#5f5f87', '#5f5faf', '#5f5fd7', '#5f5fff', // 063
	'#5f8700', '#5f875f', '#5f8787', '#5f87af', // 067
	'#5f87d7', '#5f87ff', '#5faf00', '#5faf5f', // 071
	'#5faf87', '#5fafaf', '#5fafd7', '#5fafff', // 075
	'#5fd700', '#5fd75f', '#5fd787', '#5fd7af', // 079
	'#5fd7d7', '#5fd7ff', '#5fff00', '#5fff5f', // 083
	'#5fff87', '#5fffaf', '#5fffd7', '#5fffff', // 087
	'#870000', '#87005f', '#870087', '#8700af', // 091
	'#8700d7', '#8700ff', '#875f00', '#875f5f', // 095
	'#875f87', '#875faf', '#875fd7', '#875fff', // 099
	'#878700', '#87875f', '#878787', '#8787af', // 103
	'#8787d7', '#8787ff', '#87af00', '#87af5f', // 107
	'#87af87', '#87afaf', '#87afd7', '#87afff', // 111
	'#87d700', '#87d75f', '#87d787', '#87d7af', // 115
	'#87d7d7', '#87d7ff', '#87ff00', '#87ff5f', // 119
	'#87ff87', '#87ffaf', '#87ffd7', '#87ffff', // 123
	'#af0000', '#af005f', '#af0087', '#af00af', // 127
	'#af00d7', '#af00ff', '#af5f00', '#af5f5f', // 131
	'#af5f87', '#af5faf', '#af5fd7', '#af5fff', // 135
	'#af8700', '#af875f', '#af8787', '#af87af', // 139
	'#af87d7', '#af87ff', '#afaf00', '#afaf5f', // 143
	'#afaf87', '#afafaf', '#afafd7', '#afafff', // 147
	'#afd700', '#afd75f', '#afd787', '#afd7af', // 151
	'#afd7d7', '#afd7ff', '#afff00', '#afff5f', // 155
	'#afff87', '#afffaf', '#afffd7', '#afffff', // 159
	'#d70000', '#d7005f', '#d70087', '#d700af', // 163
	'#d700d7', '#d700ff', '#d75f00', '#d75f5f', // 167
	'#d75f87', '#d75faf', '#d75fd7', '#d75fff', // 171
	'#d78700', '#d7875f', '#d78787', '#d787af', // 175
	'#d787d7', '#d787ff', '#d7af00', '#d7af5f', // 179
	'#d7af87', '#d7afaf', '#d7afd7', '#d7afff', // 183
	'#d7d700', '#d7d75f', '#d7d787', '#d7d7af', // 187
	'#d7d7d7', '#d7d7ff', '#d7ff00', '#d7ff5f', // 191
	'#d7ff87', '#d7ffaf', '#d7ffd7', '#d7ffff', // 195
	'#ff0000', '#ff005f', '#ff0087', '#ff00af', // 199
	'#ff00d7', '#ff00ff', '#ff5f00', '#ff5f5f', // 203
	'#ff5f87', '#ff5faf', '#ff5fd7', '#ff5fff', // 207
	'#ff8700', '#ff875f', '#ff8787', '#ff87af', // 211
	'#ff87d7', '#ff87ff', '#ffaf00', '#ffaf5f', // 215
	'#ffaf87', '#ffafaf', '#ffafd7', '#ffafff', // 219
	'#ffd700', '#ffd75f', '#ffd787', '#ffd7af', // 223
	'#ffd7d7', '#ffd7ff', '#ffff00', '#ffff5f', // 227
	'#ffff87', '#ffffaf', '#ffffd7', '#ffffff', // 231
	'#080808', '#121212', '#1c1c1c', '#262626', // 235
	'#303030', '#3a3a3a', '#444444', '#4e4e4e', // 239
	'#585858', '#626262', '#6c6c6c', '#767676', // 243
	'#808080', '#8a8a8a', '#949494', '#9e9e9e', // 247
	'#a8a8a8', '#b2b2b2', '#bcbcbc', '#c6c6c6', // 251
	'#d0d0d0', '#dadada', '#e4e4e4', '#eeeeee'  // 255
];

