﻿using Integpg.JniorWebSocket;
using Integpg.JniorWebSocket.Messages;
using Integpg.Logging;
using Newtonsoft.Json.Linq;
using System;
using System.Threading;
using System.Windows.Forms;

namespace WebSocketExample
{
    public partial class Form1 : Form
    {
        private NLog.Logger _log = LogDictionary.GetLog("");

        private JniorWebSocket _jniorWebsocket;
        private bool _reconnected = false;



        public Form1()
        {
            InitializeComponent();
        }



        private void JniorWebSocket_Log(object sender, LogEventArgs e)
        {
            LogToWindow(e.Message + "\r\n");
            _log?.Info(e.Message);
        }



        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }



        // Callback from the jnior web socket object.  here we will log the message that was 
        // received straigh to the log window
        private void _jniorWebsocket_Log(object sender, LogEventArgs args)
        {
            LogToWindow(args.Message + "\r\n");
        }



        // connected callback from the jnior web socket object
        private void _jniorWebsocket_Connected(object sender, EventArgs e)
        {
            LogToWindow("Connected\r\n");
            UpdateMenu();
        }



        // disconnected callback from the jnior web socket object
        private void _jniorWebsocket_Disconnected(object sender, EventArgs e)
        {
            // try to connect in a different thread
            new Thread(delegate ()
            {
                LogToWindow("Disconnected\r\n");
                UpdateMenu();

                if (!_jniorWebsocket.Closing)
                {
                    // try to reconnect
                    for (int i = 0; i < 5; i++)
                    {
                        Thread.Sleep(10000);
                        _jniorWebsocket.Reconnect();
                        //Connect(_jniorWebsocket.Host, false);
                        if (_jniorWebsocket.IsOpened)
                        {
                            _reconnected = true;
                            break;
                        }
                    }
                }
            }).Start();
        }



        private void _jniorWebsocket_Error(object sender, ExceptionEventArgs e)
        {
            LogToWindow(e.Exception.ToString() + "\r\n");
        }



        private void _jniorWebsocket_Authorized(object sender, EventArgs e)
        {
            UpdateMenu();
        }



        private void _jniorWebsocket_Unauthorized(object sender, UnauthorizedEventArgs e)
        {
            Invoke((MethodInvoker)delegate ()
            {
                var jniorWebSocket = sender as JniorWebSocket;

                if (_reconnected)
                {
                    jniorWebSocket?.Login(_jniorWebsocket.Credentials.UserName, _jniorWebsocket.Credentials.Password);
                }
                else
                {
                    // prompt for user name and password
                    var loginDlg = new LoginDialog(jniorWebSocket.Host);
                    loginDlg.UserName = jniorWebSocket.Credentials.UserName;
                    loginDlg.Password = jniorWebSocket.Credentials.Password;

                    if (DialogResult.OK == loginDlg.ShowDialog())
                    {
                        var username = loginDlg.UserName;
                        var password = loginDlg.Password;

                        jniorWebSocket?.Login(username, password);
                    }
                    else
                    {
                        // user jniorWebSocket cancel to logging in.  abort the connection
                        _jniorWebsocket?.Close();
                    }
                }
            });
        }



        private void UpdateMenu()
        {
            if (!IsDisposed)
            {
                Invoke((MethodInvoker)delegate
                {
                    if (null != _jniorWebsocket)
                    {
                        connectToolStripMenuItem.Enabled = !_jniorWebsocket.IsOpened;
                        secureConnectToolStripMenuItem.Enabled = !_jniorWebsocket.IsOpened;
                        loginToolStripMenuItem.Enabled = !_jniorWebsocket.IsAuthenticated;

                        closeOutput1ToolStripMenuItem.Enabled = _jniorWebsocket.IsOpened;
                    }

                });
            }
        }



        // our logging funciton
        private void LogToWindow(string text)
        {
            if (!IsDisposed)
            {
                //Console.WriteLine(text + string.Format("   StackTrace: '{0}'", Environment.StackTrace));

                try
                {
                    // make sure to invoke this on the main window thread
                    Invoke((MethodInvoker)delegate
                    {
                        textBox1?.AppendText(DateTime.Now.ToString("HH:mm:ss.fff") + "   " + text);
                    });
                }
                catch (Exception) { }
            }
        }



        // toggle menu item click handler
        private void toggleOutput1ToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // send a new toggle output object for channel 1
            _jniorWebsocket.Send(new ToggleOutput(1));
        }



        // connect menu item click handler
        private void connectToolStripMenuItem_Click(object sender, EventArgs e)
        {
            //Connect("10.0.0.64", false);

            var connectProperties = GetConnectProperties(false);
            if (null != connectProperties)
            {
                Connect(connectProperties.IpAddress, connectProperties.Port, connectProperties.IsSecure);
            }
        }



        private ConnectionProperties GetConnectProperties(bool secure)
        {
            var connectProperties = new ConnectionProperties();
            if (null != Properties.Settings.Default.LastIpAddress)
                connectProperties.IpAddress = Properties.Settings.Default.LastIpAddress;
            if (null != Properties.Settings.Default.LastIpAddress)
                connectProperties.Port = Properties.Settings.Default.LastPort;
            if (null != Properties.Settings.Default.LastIpAddress)
                connectProperties.IsSecure = Properties.Settings.Default.LastIsSecure;

            if (secure)
            {
                connectProperties.Port = 443;
                connectProperties.IsSecure = true;
            }

            var connectDialog = new ConnectDialog();
            connectDialog.ConnectProperties = connectProperties;
            if (DialogResult.OK == connectDialog.ShowDialog(this))
            {
                connectProperties = connectDialog.ConnectProperties;

                if (null != connectProperties)
                {
                    Properties.Settings.Default.LastIpAddress = connectProperties.IpAddress;
                    Properties.Settings.Default.LastPort = connectProperties.Port;
                    Properties.Settings.Default.LastIsSecure = connectProperties.IsSecure;
                    Properties.Settings.Default.Save();
                }

                return connectProperties;
            }
            return null;
        }



        private void secureConnectToolStripMenuItem_Click(object sender, EventArgs e)
        {
            //Connect("10.0.0.63", true);

            var connectProperties = GetConnectProperties(true);
            if (null != connectProperties)
            {
                Connect(connectProperties.IpAddress, connectProperties.Port, true);
            }
        }



        private void Connect(string ipAddress, bool secure)
        {
            Connect(ipAddress, 80, secure);
        }



        private void Connect(string ipAddress, int port, bool secure)
        {
            // create the jnior web socket object with the IP Address that we want to connect to
            _jniorWebsocket = new JniorWebSocket(ipAddress, port);

            // if we are connecting securely we need to specify to accept untrusted certificates
            _jniorWebsocket.IsSecure = secure;
            _jniorWebsocket.AllowUnstrustedCertificate = secure;

            // wireup some callbacks
            _jniorWebsocket.Log += _jniorWebsocket_Log;
            _jniorWebsocket.Error += _jniorWebsocket_Error;
            _jniorWebsocket.Connected += _jniorWebsocket_Connected;
            _jniorWebsocket.Disconnected += _jniorWebsocket_Disconnected;
            _jniorWebsocket.Authorized += _jniorWebsocket_Authorized;
            _jniorWebsocket.Unauthorized += _jniorWebsocket_Unauthorized;
            _jniorWebsocket.MessageReceived += _jniorWebsocket_MessageReceived;

            // connect!
            _jniorWebsocket.Connect();
        }



        private void _jniorWebsocket_MessageReceived(object sender, MessageReceivedEventArgs e)
        {
            LogToWindow(e.Message + Environment.NewLine);

            // grab input 1 counter value
            var json = JObject.Parse(e.Message);

            var message = json["Message"].ToString();
            if ("Monitor".Equals(message))
            {
                HandleMonitorMessage(json);
            }

        }



        private void HandleMonitorMessage(JObject json)
        {
            {
                // get the inputs.  it will be an array of json objects
                var inputsArray = json["Inputs"];

                var input1 = inputsArray[0];
                // get input 1 counter value
                var counter = (int)input1["Count"];
            }
        }



        private void readClockToolStripMenuItem_Click(object sender, EventArgs e)
        {
            var readClock = new ClockRead();
            _jniorWebsocket.Send(readClock);
        }



        private void setClockToolStripMenuItem_Click(object sender, EventArgs e)
        {
            SetClock();
        }



        private void SetClock()
        {
            // get the time since epoch JAN 1 1970
            var epoch = new DateTime(1970, 1, 1, 0, 0, 0, DateTimeKind.Utc);
            var timeSinceJan11970 = DateTime.UtcNow - epoch;

            // set the clock via websocket
            var setClock = new ClockSet((long)timeSinceJan11970.TotalMilliseconds);
            _jniorWebsocket.Send(setClock);
        }



        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            _jniorWebsocket?.Close();
        }



        private void rebootToolStripMenuItem_Click(object sender, EventArgs e)
        {
            _jniorWebsocket.ConsoleSession.Open();
            _jniorWebsocket.ConsoleSession.Send("reboot -f\r");
        }



        private void loginToolStripMenuItem_Click(object sender, EventArgs e)
        {
            _jniorWebsocket?.Login("jnior", "jnior");
        }



        private void Form1_Shown(object sender, EventArgs e)
        {
            UpdateMenu();
        }



        private void Command_Log(object sender, LogEventArgs e)
        {
            LogToWindow(e.Message + Environment.NewLine);
        }



        private void openConsoleToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (null != _jniorWebsocket)
            {
                var consoleOpen = new ConsoleOpen();
                _jniorWebsocket.Send(consoleOpen);
            }
        }
    }
}
